package com.openexchange.guard.server;

import java.io.IOException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.guard.exceptions.BadPasswordException;
import com.openexchange.guard.logging.LogAction;
import com.openexchange.guard.mail.SignMime;
import com.openexchange.guard.server.authentication.GuardAuthRole;
import com.openexchange.guard.server.authentication.GuardAuthRoles;
import com.openexchange.guard.util.Core;


public class AdminServlet extends HttpServlet  {
    private static Logger logger = LoggerFactory.getLogger(AdminServlet.class);

    public enum AuthLevel {
        NONE (0), BAD (1), Backend (2), Admin (3);
        private final int value;
        AuthLevel (int value) {
            this.value = value;
        }
        public int val() { return(value); }
        public static AuthLevel check (HttpServletRequest request) {
            if (request.isUserInRole("backend")) {
                return(Backend);
            }
            if (request.isUserInRole("admin")) {
                return (Admin);
            }
            return (NONE);
        }
    }

    @Override
    protected void doPut(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        LogAction.getPID();
        String url = request.getRequestURI();
        String command = request.getQueryString();
        String action = getAction(command);
        LogAction.setURL(Core.getIP(request), url, action);
        logger.info("Admin command sent " + command + "  from IP: " + Core.getIP(request));
        logger.info("Action: " + action);

        JsonObject json = Core.getJSON(request);
        if (action == null) {
            Core.sendFail(response, "Missing action");
            return;
        }

        //Checking if the user is allowed to use this servlet
        if(GuardAuthRoles.isUserInRole(request, GuardAuthRole.OXBackend))
        {
            switch (action) {
                case "user_delete":
                    try {
                        if (json == null) {
                            LogAction.setFail();
                            logger.info("Missing JSON for user delete");
                            Core.sendFail(response, "Missing JSON");
                            return;
                        }
                        int userid = Core.getIntFromJson(json, "user_id", true);
                        int context = Core.getIntFromJson(json, "context_id", true);
                        Admin admin = new Admin();
                        if (admin.delete(userid, context)) {
                            LogAction.setSuccess();
                            logger.info("Success deleting user_id:" + userid + " context:" + context);
                        } else {
                            LogAction.setFail();
                            logger.error("Problem deleting user_id:" + userid + " context:" + context);
                        }
                        Core.sendOK(response, "OK");
                    } catch (Exception e) {
                        LogAction.setFail();
                        logger.error("Problem deleting user", e);
                        MainServlet.handleException (e, request, response);
                    }
                    break;
                case "context_delete":
                    try {
                        if (json == null) {
                            LogAction.setFail();
                            logger.info("Missing JSON for context_delete");
                            Core.sendFail(response, "Missing JSON");
                            return;
                        }
                        int context = Core.getIntFromJson(json, "context_id", true);
                        Admin admin = new Admin();
                        if (admin.deleteContext(context)) {
                            LogAction.setSuccess();
                            logger.info("Success deleting context:" + context);
                        } else {
                            LogAction.setFail();
                            logger.error("Problem deleting context:" + context);
                        }
                        Core.sendOK(response, "OK");
                    } catch (Exception e) {
                        LogAction.setFail();
                        logger.error("Problem deleting context", e);
                        MainServlet.handleException (e, request, response);
                    }
                    break;
            }
        }
        else {
            LogAction.setFail();
            String errorMessage =
                String.format("User is not allowed to execute command %s",
                              action);
            logger.info(errorMessage);
            Core.sendAnswer(response, HttpServletResponse.SC_FORBIDDEN, errorMessage);

            logger.info("Bad authorization for user delete");
            Core.sendFail(response, "Bad auth");
        }
    }

    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        String url = request.getRequestURI();

        String command = request.getQueryString();
        String action = getAction(command);
        LogAction.getPID();
        LogAction.setURL(Core.getIP(request), url, action);
        logger.debug("Get request " + url + " command: " + command);
        Core.sendOK(response, "OK");

        doPut(request,response);
    }

    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        String url = request.getRequestURI();

        String command = request.getQueryString();
        String action = getAction(command);
        LogAction.getPID();
        LogAction.setURL(Core.getIP(request), url, action);
        logger.debug("Post request " + url + " command: " + command);
        switch (action) {
            case "process_message":
                try {
                    SignMime sm = new SignMime();
                    sm.processIncomingMime(request, response);
                } catch (BadPasswordException ex) {
                    Core.sendFail(response, "Bad password");
                } catch (Exception e) {
                    logger.error("Error processing incoming mime message", e);
                    Core.sendFail(response, "Error");
                }
                break;
        }

    }

    private String getAction (String command) {
        String [] options = command.split("&");
        for (String opt : options) {
            if (opt.toLowerCase().startsWith("action")) {
                if (opt.contains("=")) {
                    return (opt.substring(opt.indexOf("=") + 1));
                }
            }
        }
        return (null);
    }


}
