/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.security.SignatureException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.bcpg.HashAlgorithmTags;
import org.bouncycastle.bcpg.PublicKeyPacket;
import org.bouncycastle.bcpg.SignatureSubpacketTags;
import org.bouncycastle.bcpg.SymmetricKeyAlgorithmTags;
import org.bouncycastle.bcpg.sig.KeyFlags;
import org.bouncycastle.bcpg.sig.RevocationReasonTags;
import org.bouncycastle.crypto.generators.RSAKeyPairGenerator;
import org.bouncycastle.crypto.params.RSAKeyGenerationParameters;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPKeyPair;
import org.bouncycastle.openpgp.PGPKeyRingGenerator;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureGenerator;
import org.bouncycastle.openpgp.PGPSignatureSubpacketGenerator;
import org.bouncycastle.openpgp.PGPSignatureSubpacketVector;
import org.bouncycastle.openpgp.PGPUserAttributeSubpacketVector;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.KeyFingerPrintCalculator;
import org.bouncycastle.openpgp.operator.PBESecretKeyDecryptor;
import org.bouncycastle.openpgp.operator.PBESecretKeyEncryptor;
import org.bouncycastle.openpgp.operator.PGPDigestCalculator;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.bc.BcPBESecretKeyDecryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPBESecretKeyEncryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPDigestCalculatorProvider;
import org.bouncycastle.openpgp.operator.bc.BcPGPKeyPair;
import org.bouncycastle.openpgp.operator.jcajce.JcaKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentVerifierBuilderProvider;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPKeyConverter;
import org.bouncycastle.openpgp.operator.jcajce.JcePBESecretKeyDecryptorBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.BadPasswordException;
import com.openexchange.guard.translate.GetText;
import com.openexchange.guard.util.Core;

/**
 * Class to create PGPKeys
 * 
 * @author greg
 */
public class PgpKeys {
	private static Logger logger = LoggerFactory.getLogger(PgpKeys.class);
    public PgpKeys() {
    }

    public GuardKeys generateKeys(String name, String email, String password) throws Exception {

        // Generate key-pairs
        RSAKeyPairGenerator kpgen = new RSAKeyPairGenerator();
        // 2048 strength. Change for export?
        kpgen.init(new RSAKeyGenerationParameters(BigInteger.valueOf(0x10001), new SecureRandom(), Config.RSAKeyLen, Config.rsa_certainty));

        // Create master (signing) key -- These had to be RSA_GENERAL FOR GNU TO work with the signatures
        PGPKeyPair rsa_master = new BcPGPKeyPair(PGPPublicKey.RSA_GENERAL, kpgen.generateKeyPair(), new Date());
        // SubKey
        PGPKeyPair rsa_encrypt = new BcPGPKeyPair(PGPPublicKey.RSA_GENERAL, kpgen.generateKeyPair(), new Date());
        return (generateKeys(name, email, password, rsa_master, rsa_encrypt));
    }

    /**
     * Generate oxGuard KeyGroup
     * 
     * @param id ID of the user
     * @param password for encryption
     * @return
     * @throws Exception
     */
    public GuardKeys generateKeys(String name, String email, String password, PGPKeyPair rsa_master, PGPKeyPair rsa_encrypt) throws Exception {
        long start = System.currentTimeMillis();
        String salt = EncrLib.getsalt();
        char[] pass = EncrLib.getSHA(password, salt).toCharArray();
        if (password.equals("")) { // If password is blank (unknown), then just create with salt as password
            pass = EncrLib.getSHA(salt, salt).toCharArray();
            password = salt;
        }

        // Add self-signature
        PGPSignatureSubpacketGenerator signhashgenGenerator = new PGPSignatureSubpacketGenerator();

        // This is for master, just signing and certifying
        signhashgenGenerator.setKeyFlags(false, KeyFlags.SIGN_DATA | KeyFlags.CERTIFY_OTHER);
        signhashgenGenerator.setPreferredSymmetricAlgorithms(false, new int[] {
            SymmetricKeyAlgorithmTags.AES_256, SymmetricKeyAlgorithmTags.AES_192, SymmetricKeyAlgorithmTags.AES_128 });

        signhashgenGenerator.setPreferredHashAlgorithms(
            false,
            new int[] {
                HashAlgorithmTags.SHA256, HashAlgorithmTags.SHA1, HashAlgorithmTags.SHA384, HashAlgorithmTags.SHA512,
                HashAlgorithmTags.SHA224 });
        
        // Now for subKey (encrypting)
        PGPSignatureSubpacketGenerator enchashgen = new PGPSignatureSubpacketGenerator();

        enchashgen.setKeyFlags(false, KeyFlags.ENCRYPT_COMMS | KeyFlags.ENCRYPT_STORAGE);

        enchashgen.setPreferredSymmetricAlgorithms(false, new int[] {
            SymmetricKeyAlgorithmTags.AES_256, SymmetricKeyAlgorithmTags.AES_192, SymmetricKeyAlgorithmTags.AES_128 });

        enchashgen.setPreferredHashAlgorithms(
            false,
            new int[] {
                HashAlgorithmTags.SHA256, HashAlgorithmTags.SHA1, HashAlgorithmTags.SHA384, HashAlgorithmTags.SHA512,
                HashAlgorithmTags.SHA224 });
        
        if (Config.key_valid_days != 0) {  // If expiration date is chosen, set now
            int seconds = Config.key_valid_days * 24 * 60 * 60;
            signhashgenGenerator.setKeyExpirationTime(false, seconds);
            enchashgen.setKeyExpirationTime(false, seconds);
        }
        

        PGPDigestCalculator sha1Calc = new BcPGPDigestCalculatorProvider().get(HashAlgorithmTags.SHA1);

        PGPDigestCalculator sha256Calc = new BcPGPDigestCalculatorProvider().get(getHashAlgorithmTags());

        // Encrypting the private key
        PBESecretKeyEncryptor keyEncr = (new BcPBESecretKeyEncryptorBuilder(getPGPEncryptedData(), sha256Calc, 0x60)).build(pass);

        // Generate Keyring with name and email, signed
        PGPKeyRingGenerator keyRingGen = new PGPKeyRingGenerator(
            PGPSignature.POSITIVE_CERTIFICATION,
            rsa_master,
            (name + " <" + email + ">"),
            sha1Calc,
            signhashgenGenerator.generate(),
            null,
            new BcPGPContentSignerBuilder(rsa_master.getPublicKey().getAlgorithm(), HashAlgorithmTags.SHA1),
            keyEncr);

        keyRingGen.addSubKey(rsa_encrypt, enchashgen.generate(), null);

        JcaPGPKeyConverter conv = new JcaPGPKeyConverter();

        // Get the private and public encr RSA keys
        PrivateKey privkey = conv.getPrivateKey(rsa_encrypt.getPrivateKey());
        PublicKey pubkey = conv.getPublicKey(rsa_encrypt.getPublicKey());
        // Save to keys format
        GuardKeys keys = new GuardKeys(keyRingGen, privkey, pubkey, password, salt);
        logger.info("Created PGP/RSA Keys at " + start);
        keys.email = email;
        return keys;

    }

    public PGPSecretKeyRing changePasswordPGP(String oldpass, String newpass, PGPSecretKeyRing keyring) throws NoSuchProviderException, PGPException {
        Security.addProvider(new BouncyCastleProvider());
        PGPDigestCalculator sha256Calc = new BcPGPDigestCalculatorProvider().get(getHashAlgorithmTags());
        PBESecretKeyDecryptor oldencr = new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(oldpass.toCharArray());
        PBESecretKeyEncryptor newencr = (new BcPBESecretKeyEncryptorBuilder(getPGPEncryptedData(), sha256Calc, 0x60)).build(newpass.toCharArray());
        PGPSecretKeyRing newkeyring = PGPSecretKeyRing.copyWithNewPassword(
                keyring,
                oldencr,
                newencr);
        return (newkeyring);

    }

    public GuardKeys changePassword(String oldpass, String newpass, GuardKeys oldKey) throws NoSuchProviderException, PGPException, BadPasswordException, UnsupportedEncodingException {
        if (!oldKey.privKeyNull()) {
            PrivateKey priv = oldKey.getDecodedPrivate(oldpass);
    
            if (priv == null)
                return (null);
            oldKey.setPrivate(priv, newpass);
        }
        oldpass = EncrLib.getSHA(oldpass, oldKey.getSalt());
        newpass = EncrLib.getSHA(newpass, oldKey.getSalt());
        PGPSecretKeyRing keyring = oldKey.getPGPSecretKeyRing();
        PGPDigestCalculator sha256Calc = new BcPGPDigestCalculatorProvider().get(getHashAlgorithmTags());
        PBESecretKeyDecryptor oldencr = new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(oldpass.toCharArray());
        PBESecretKeyEncryptor newencr = (new BcPBESecretKeyEncryptorBuilder(getPGPEncryptedData(), sha256Calc, 0x60)).build(newpass.toCharArray());
        try {
            PGPSecretKeyRing newkeyring = PGPSecretKeyRing.copyWithNewPassword(
                    keyring,
                    oldencr,
                    newencr);
            if (newkeyring == null)
                return (null);
            oldKey.setPGPSecretKeyRing(newkeyring);
            return (oldKey);
        } catch (Exception ex) {
            return(null);
        }

    }

    public GuardKeys changePasswordWithRecovery(String oldpassHash, String newpass, GuardKeys oldKey) throws NoSuchProviderException, PGPException {

        if (!oldKey.privKeyNull()) {
            PrivateKey priv = oldKey.getDecodedPrivateWithRecovery(oldpassHash);
    
            if (priv == null)
                return (null);
            oldKey.setPrivate(priv, newpass);
        }
        newpass = EncrLib.getSHA(newpass, oldKey.getSalt());
        Security.addProvider(new BouncyCastleProvider());
        PGPSecretKeyRing keyring = oldKey.getPGPSecretKeyRing();
        PGPDigestCalculator sha256Calc = new BcPGPDigestCalculatorProvider().get(getHashAlgorithmTags());
        PBESecretKeyDecryptor oldencr = new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(oldpassHash.toCharArray());
        PBESecretKeyEncryptor newencr = (new BcPBESecretKeyEncryptorBuilder(getPGPEncryptedData(), sha256Calc, 0x60)).build(newpass.toCharArray());
        PGPSecretKeyRing newkeyring = PGPSecretKeyRing.copyWithNewPassword(
                keyring,
                oldencr,
                newencr);
        if (newkeyring == null)
            return (null);
        oldKey.setPGPSecretKeyRing(newkeyring);
        return (oldKey);
    }

    private int getSymKeyAllgorithm() {
        if (Config.AESKeyLen == 256) {
            return (SymmetricKeyAlgorithmTags.AES_256);
        }
        return (SymmetricKeyAlgorithmTags.AES_128);
    }

    private static int getPGPEncryptedData() {
        if (Config.AESKeyLen == 256) {
            return (PGPEncryptedData.AES_256);
        }
        return (PGPEncryptedData.AES_128);
    }

    private static int getHashAlgorithmTags() {
        return (HashAlgorithmTags.SHA256);
    }
    
    
    public static void addPublicKeyIndex (GuardKeys key) throws Exception {
    	addPublicKeyIndex (key.pubring, key.email, key.contextid, true, 0l);
    }
    
    public static void addPublicKeyIndex (PGPPublicKeyRing pubring, String email, int cid, boolean local) throws Exception {
        addPublicKeyIndex (pubring, email, cid, local, 0l);
    }
/**
 * Add the key to the Public Key table index
 * @param key
 * @throws Exception
 */
    public static void addPublicKeyIndex (PGPPublicKeyRing pubring, String email, int cid, boolean local, long keyid) throws Exception {
		Iterator<PGPPublicKey> it = pubring.getPublicKeys();
		String command = "INSERT INTO PGPKeys (id, email, hexid, cid, local, keyid) VALUES (?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE cid = ?";
		while (it.hasNext()) {
			PGPPublicKey pubkey = it.next();
			if (keyid == 0) keyid = pubkey.getKeyID();
			DbCommand com2 = new DbCommand(command);
			com2.addVariables(pubkey.getKeyID());
			com2.addVariables(email);
			String hex = Core.PGPKeyId(pubkey.getKeyID());
			com2.addVariables(hex);
			com2.addVariables(cid);
			if (local) {
				com2.addVariables(1);
			} else com2.addVariables(0);
			com2.addVariables(keyid);
			com2.addVariables(cid);
			DbQuery db2 = new DbQuery();
			db2.writeOxGuard(com2);
			db2.close();
		}
    }
    
    public static PGPPrivateKey decodePrivate (PGPSecretKey secretKey, String password, String salt) throws PGPException {
    	return(secretKey.extractPrivateKey(new BcPBESecretKeyDecryptorBuilder(new BcPGPDigestCalculatorProvider()).build(EncrLib.getSHA(password, salt).toCharArray())));
    }
    
    public static String ExportPGPPrivate (GuardKeys keys, String password) throws PGPException, BadPasswordException, IOException {
    	PGPDigestCalculator sha256Calc = new BcPGPDigestCalculatorProvider().get(getHashAlgorithmTags());
    	String oldpass = EncrLib.getSHA(password, keys.getSalt());
        PBESecretKeyDecryptor oldencr = new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(oldpass.toCharArray());
        PBESecretKeyEncryptor newencr = (new BcPBESecretKeyEncryptorBuilder(getPGPEncryptedData(), sha256Calc, 0x60)).build(password.toCharArray());
        PGPSecretKeyRing newkeyring = PGPSecretKeyRing.copyWithNewPassword(
            keys.getPGPSecretKeyRing(),
            oldencr,
            newencr);
        if (newkeyring == null) {
        	throw new BadPasswordException ("Bad password");
        }
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ArmoredOutputStream arm = new ArmoredOutputStream(out);
        arm.write(newkeyring.getEncoded());
        arm.close();
        out.close();
        return (new String(out.toByteArray()));
    }
    
    public static GuardKeys ImportPGPPrivate (GuardKeys keys, String PGPPrivateASC, String pgpPassword, String guardPassword) throws UnsupportedEncodingException, IOException, BadPasswordException {
    	// PGPUtil.getDecoderStream() will detect ASCII-armor automatically and decode it,
        // the PGPObject factory then knows how to read all the data in the encoded stream
        PGPObjectFactory factory = new PGPObjectFactory(PGPUtil.getDecoderStream(new java.io.ByteArrayInputStream(PGPPrivateASC.getBytes("UTF-8"))), new BcKeyFingerprintCalculator());
        // these files should really just have one object in them,
        // and that object should be a PGPPublicKeyRing.
        Object o = factory.nextObject();
        if (o instanceof PGPSecretKeyRing) {
            PGPSecretKeyRing secretRing = (PGPSecretKeyRing) o;
            keys.setPGPSecretKeyRing(secretRing);

            try {
            	 String newpass = EncrLib.getSHA(guardPassword, keys.getSalt());
                 Security.addProvider(new BouncyCastleProvider());
                 PGPSecretKeyRing keyring = keys.getPGPSecretKeyRing();
                 
                 PGPDigestCalculator sha256Calc = new BcPGPDigestCalculatorProvider().get(PgpKeys.getHashAlgorithmTags());
                 PBESecretKeyDecryptor oldencr = new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(pgpPassword.toCharArray());
                 PBESecretKeyEncryptor newencr = (new BcPBESecretKeyEncryptorBuilder(PgpKeys.getPGPEncryptedData(), sha256Calc, 0x60)).build(newpass.toCharArray());
                 PGPSecretKeyRing newkeyring = null;
                 try {
                     newkeyring = PGPSecretKeyRing.copyWithNewPassword(
                         keyring,
                         oldencr,
                         newencr);
                 } catch (Exception ex) {
                     if (ex.getMessage().contains("checksum")) {
                         throw new BadPasswordException ("Bad password");
                     } else {
                         logger.error("Error changing password of imported key ", ex);
                         return(null);
                     }
                 }
                 if (newkeyring == null) {
                     logger.error("Error importing PGP Private key");
                     return (null);
                 }
                 else 
                     keys.setPGPSecretKeyRing(newkeyring);
                 
                 
                 
			} catch (PGPException e) {
				logger.error("Problem importing PGP Private " , e);
				return (null);
			}
            try {
				PGPPrivateKey pkey = decodePrivate(keys.getPGPSecretKey(), guardPassword, keys.getSalt());
				if (pkey == null) {
				    logger.error("Error decoding PGP Private key for inport");
				    return (null);
				}
			} catch (PGPException e) {
			    logger.error("Problem importing PGP Private " , e);
                return (null);
			}
            
            
        }
        return (keys);
    }
    
    /**
     * Create public key ring from asc armor
     * @param asc
     * @return
     */
    public static PGPPublicKeyRing createKeyRing (String asc) {
        try {
            PGPObjectFactory factory = new PGPObjectFactory(PGPUtil.getDecoderStream(new java.io.ByteArrayInputStream(asc.getBytes("UTF-8"))), new BcKeyFingerprintCalculator());
            // these files should really just have one object in them,
            // and that object should be a PGPPublicKeyRing.
            Object o = factory.nextObject();
            if (o instanceof PGPPublicKeyRing) {
                return (PGPPublicKeyRing) o;
            }
        } catch (Exception ex) {
            logger.error("Error creating keyring " ,ex);
             
        }

        throw new IllegalArgumentException("Input text does not contain a PGP Public Key");
    }
    
    /**
     * Create public keyring from private key
     * @param keys
     * @throws IOException
     */
    public static GuardKeys createPubFromPrivate (GuardKeys keys) throws IOException {
        PGPPublicKeyRing npr = null;
        PGPSecretKeyRing newkeyring = keys.getPGPSecretKeyRing();
        
        for (Iterator it = newkeyring.getPublicKeys(); it.hasNext();)
        {
            PGPPublicKey key = (PGPPublicKey) it.next();
            npr = addToKeyRing (npr, key);
        }
        for (Iterator it2 = newkeyring.getExtraPublicKeys(); it2.hasNext();) 
        {
            PGPPublicKey key = (PGPPublicKey) it2.next();
            PGPPublicKeyRing.insertPublicKey(npr, key);
        }
        keys.pubring = npr;
        return(keys);
    }
    
    /**
     * Add to PGPPublicKey Ring.  Ring should be null if empty (start)
     * @param npr
     * @param key
     * @throws IOException
     */
    public static PGPPublicKeyRing addToKeyRing (PGPPublicKeyRing npr, PGPPublicKey key) throws IOException {
        if (npr == null) {
            npr = new PGPPublicKeyRing(key.getEncoded(), new JcaKeyFingerprintCalculator());
        }
        else {
            npr = PGPPublicKeyRing.insertPublicKey(npr, key);
        }
        return (npr);
    }
    
    private static byte revokeReason (String reason) {
        switch (reason) {
            case "NO_REASON":
                return RevocationReasonTags.NO_REASON;
            case "KEY_SUPERSEDED":
                return RevocationReasonTags.KEY_SUPERSEDED;
            case "KEY_COMPROMISED":
                return RevocationReasonTags.KEY_COMPROMISED;
            case "KEY_RETIRED":
                return RevocationReasonTags.KEY_RETIRED;
            case "USER_NO_LONGER_VALID":
                return RevocationReasonTags.USER_NO_LONGER_VALID;
        }
        return (RevocationReasonTags.NO_REASON);
    }
    
    /**
     * Revokes the PGP Public keys associated with the GuardKeys and returns with the revokation certificate
     * @param keys
     * @param password
     * @return
     * @throws SignatureException
     * @throws PGPException
     */
    public static GuardKeys revoke (GuardKeys keys, String password, String reason ) throws SignatureException, PGPException {
        PGPSecretKey secretKey = keys.getPGPSecretKey();
        Iterator<PGPPublicKey> pkeys = keys.pubring.getPublicKeys();
        PGPPublicKey master = secretKey.getPublicKey();
        while (pkeys.hasNext()) {
            PGPPublicKey pub = pkeys.next();
            keys.pubring = PGPPublicKeyRing.removePublicKey(keys.pubring, pub);
            PGPSignatureSubpacketGenerator subHashGenerator = new PGPSignatureSubpacketGenerator();
            PGPSignatureSubpacketGenerator subUnHashGenerator = new PGPSignatureSubpacketGenerator();
            PGPSignatureGenerator generator = new PGPSignatureGenerator(new BcPGPContentSignerBuilder(pub.getAlgorithm(),
                PGPUtil.SHA1));
            PGPPrivateKey priv = PgpKeys.decodePrivate(secretKey, password, keys.getSalt());
            if (pub.isMasterKey()) {
                generator.init(PGPSignature.KEY_REVOCATION, priv);
                master = pub;
            } else {
                generator.init(PGPSignature.SUBKEY_REVOCATION, priv);
            }
            subHashGenerator.setSignatureCreationTime(false, new Date());
            subHashGenerator.setRevocationReason(false, revokeReason(reason), reason);
            subUnHashGenerator.setRevocationKey(false, pub.getAlgorithm(), pub.getFingerprint());
            generator.setHashedSubpackets(subHashGenerator.generate());
            generator.setUnhashedSubpackets(subUnHashGenerator.generate());
            if (pub.isMasterKey()) {
                PGPSignature signature = generator.generateCertification(pub);
                pub = PGPPublicKey.addCertification(pub, signature);
            } else {
                PGPSignature signature = generator.generateCertification(master, pub);
                pub = PGPPublicKey.addCertification(pub, signature);
            }
            
            keys.pubring = PGPPublicKeyRing.insertPublicKey(keys.pubring, pub);
        }
        
        return(keys);
    }
    
////////SIGN PUBLIC KEY.  NOT YET WORKING
    public static PGPPublicKey signKey (PGPPublicKey tosign, GuardKeys key, String password) throws PGPException, SignatureException {
        Iterator<String> ids = tosign.getUserIDs();
        String id = "";
        if (ids.hasNext()) {
            id = ids.next();
        }
        PGPPrivateKey priv = PgpKeys.decodePrivate(key.getPGPSigningKey(), password, key.getSalt());
        PGPSignatureGenerator signatureGenerator = new PGPSignatureGenerator(new JcaPGPContentSignerBuilder(key.getPGPSecretKey().getPublicKey().getAlgorithm(), PGPUtil.SHA256));
        signatureGenerator.init(PGPSignature.DIRECT_KEY, priv);
        PGPSignature sig = signatureGenerator.generateCertification(tosign);
        return (PGPPublicKey.addCertification(tosign, sig));
        
        
    }
    
    
    public static PGPPublicKeyRing sign (PGPSecretKey secretKey, PGPPrivateKey priv, PGPPublicKeyRing keyring) throws IOException, Exception {
        PGPPublicKeyRing sRing = new PGPPublicKeyRing(new ByteArrayInputStream(signPublicKey(secretKey, priv, keyring.getPublicKey())), new JcaKeyFingerprintCalculator());
		return (sRing);
}
    private static byte[] signPublicKey(PGPSecretKey secretKey, PGPPrivateKey pgpPrivKey, PGPPublicKey keyToBeSigned) throws Exception
    {

        PGPSignatureGenerator       sGen = new PGPSignatureGenerator(new JcaPGPContentSignerBuilder(secretKey.getPublicKey().getAlgorithm(), PGPUtil.SHA1).setProvider("BC"));

        sGen.init(PGPSignature.DIRECT_KEY, pgpPrivKey);

        PGPSignatureSubpacketGenerator spGen = new PGPSignatureSubpacketGenerator();

        boolean isHumanReadable = true;

     //   spGen.setNotationData(true, isHumanReadable, notationName, notationValue);

        PGPSignatureSubpacketVector packetVector = spGen.generate();

        sGen.setHashedSubpackets(packetVector);

        return PGPPublicKey.addCertification(keyToBeSigned, sGen.generate()).getEncoded();
    }
    
    /**
     * Add new User ID to publicKeyRing.  Must have private key for signing
     * @param priv
     * @param keyring
     * @param newid
     * @return
     * @throws SignatureException
     * @throws PGPException
     */
    public static PGPPublicKeyRing addUID (PGPPrivateKey priv, PGPPublicKeyRing keyring, String newid) throws SignatureException, PGPException {
    			PGPPublicKey pub = keyring.getPublicKey();
    			PGPSignatureGenerator generator = new PGPSignatureGenerator(new BcPGPContentSignerBuilder(PGPPublicKey.RSA_GENERAL,
    			  PGPUtil.SHA1)); 
    			generator.init(PGPSignature.POSITIVE_CERTIFICATION, priv);
    			PGPSignatureSubpacketGenerator signhashgen = new PGPSignatureSubpacketGenerator();      
    			generator.setHashedSubpackets(signhashgen.generate());    
    			PGPSignature certification = generator.generateCertification(newid, pub);
    			PGPPublicKey newPubKey = PGPPublicKey.addCertification(pub, newid, certification);
    			keyring = PGPPublicKeyRing.removePublicKey(keyring, pub);
    			keyring = PGPPublicKeyRing.insertPublicKey(keyring, newPubKey);
    			return (keyring);
    }
    
    public static void checkBadUID (GuardKeys key, String password) throws Exception {
    	PGPPublicKey pubkey = key.pubring.getPublicKey();
    	Iterator it = pubkey.getUserIDs();
    	while (it.hasNext()) {
    		String uid = (String) it.next();
    		if (uid.contains(" <")) {
    			// already updated
    			markUIDUpdated (key);
    		} else {
    			int i = uid.indexOf("<");
    			if (i < 0) {
    				// no email address found
    				return;
    			} 
    			String newuid = uid.substring(0, i) + " " + uid.substring(i);
    			PGPPrivateKey priv = PgpKeys.decodePrivate(key.getMasterPGPSecretKey(), password, key.getSalt());
    			if (priv != null) {
    				key.pubring = addUID (priv, key.pubring, newuid);
    				Access acc = new Access();
    				acc.updatePGPKeys(key.userid, key.contextid, key);
    				markUIDUpdated(key);
    				logger.info("Corrected missing space for userid " + newuid);
    			}
    		}
    	}
    }
    
    public static String getSignatureType (int signature) {
    	switch (signature) {
    	case -1:
    	    return("Fail");
    	case 0:
    	    return("Missing Public key");
    	case PGPSignature.POSITIVE_CERTIFICATION:
    		return ("Positive");
    	case PGPSignature.CASUAL_CERTIFICATION:
    		return ("Casual");
    	case PGPSignature.CERTIFICATION_REVOCATION:
    		return ("Revoke Cert");
    	case PGPSignature.DEFAULT_CERTIFICATION:
    		return ("Default Cert");
    	case PGPSignature.DIRECT_KEY:
    		return ("Direct");
    	case PGPSignature.KEY_REVOCATION:
    		return ("Revoke");
    	case PGPSignature.SUBKEY_REVOCATION:
    		return ("Sub revoke");
    	}
    	return ("");
    }
    
    /**
     * Verify a specific signature in key
     * @param sig
     * @param key
     * @param userid
     * @param cid
     * @return
     * @throws Exception
     */
    public static ArrayList<SignatureResults> verifySignature (PGPSignature sig, PGPPublicKey key, int userid, int cid) throws Exception {
        Access acc = new Access();
        PGPPublicKeyRing ring = acc.getPublicKeyRingById(sig.getKeyID(), userid, cid);
            
        ArrayList<SignatureResults> results = new ArrayList<SignatureResults> ();
        if (ring == null) {
            results.add(new SignatureResults("", 0, sig.getKeyID()));
            return(results);
        }
        PGPPublicKey sigKey = ring.getPublicKey(sig.getKeyID());
        if (sigKey == null) {
            results.add(new SignatureResults("", 0, sig.getKeyID()));
            return(results);
        }
        String uid = "";
        Iterator it = sigKey.getUserIDs();
        if (it.hasNext()) {
            uid = (String) it.next();
        }
        sig.init(new JcaPGPContentVerifierBuilderProvider(), sigKey);
        if (sig.getSignatureType() == PGPSignature.KEY_REVOCATION) {
            if (sig.verifyCertification(key)) {
                SignatureResults result = new SignatureResults("", sig.getSignatureType(), uid);
                results.add(result);
                return (results);
            }
        }
        // Check if signed userid
        Iterator<String> userids = key.getUserIDs();
        boolean verified = false;
        while (userids.hasNext()) {
            String id = userids.next();
            try {
                byte[] signature = sig.getSignature();
                if (signature.length == 255) {
                    logger.debug("Bad signature length");
                    SignatureResults result = new SignatureResults("", -1, uid);
                    results.add(result);
                    return(results);      
                }
                if (sig.verifyCertification(id, key)) {
                    SignatureResults result = new SignatureResults(id, sig.getSignatureType(), uid);
                    results.add(result);
                    verified = true;
                }
            } catch (Exception e) {
                logger.error("Problem verifying signature ", e);
            }
        }
        // If not, check if signed attribute (picture, etc)
        if (!verified) {
            Iterator<PGPUserAttributeSubpacketVector> userattr = key.getUserAttributes();
            while (userattr.hasNext()) {
                PGPUserAttributeSubpacketVector id = userattr.next();
                if (sig.verifyCertification(id, key)) {
                    SignatureResults result = new SignatureResults("image", sig.getSignatureType(), uid);
                    result.setImageType(id.getImageAttribute().getType());
                    result.setImage(id.getImageAttribute().getImageData());
                    results.add(result);
                    verified = true;
                }
            }
        }
        if (!verified) {
            try {
                if (sig.verifyCertification(ring.getPublicKey(sig.getKeyID()), key)) {
                    SignatureResults result = new SignatureResults("", sig.getSignatureType(), uid);
                    results.add(result);
                    verified = true;
                }
            } catch (Exception e) {
                
            }
        }
        if (!verified) {
            SignatureResults result = new SignatureResults("", -1, uid);
            results.add(result);
        }
        return (results);
    }
    
    /**
     * Check all of the signatures in a key
     * @param key
     * @param userid
     * @param cid
     * @return
     * @throws Exception
     */
    public static ArrayList<SignatureResults> checkSignatures (PGPPublicKey key, int userid, int cid) throws Exception {
        Iterator <PGPSignature> sigs = key.getSignatures();
        ArrayList<SignatureResults> results = new ArrayList<SignatureResults>();
        while (sigs.hasNext()) {
            PGPSignature sig = sigs.next();
            results.addAll(verifySignature (sig, key, userid, cid));
        }
        return (results);
    }
    
    /**
     * Check all the signatures of the keys in a pubring
     * @param ring
     * @param userid
     * @param cid
     * @return
     * @throws Exception
     */
    public static ArrayList<SignatureResults> checkSignatures (PGPPublicKeyRing ring, int userid, int cid) throws Exception {
        ArrayList<SignatureResults> results = new ArrayList<SignatureResults> ();
        Iterator <PGPPublicKey> keys = ring.getPublicKeys();
        while (keys.hasNext()) {
            PGPPublicKey key = keys.next();
            results.addAll(checkSignatures(key, userid, cid));
        }
        return (results);
    }
    
    /**
     * Check if a public key is expired
     * @param key
     * @return
     */
    public static boolean isExpired (PGPPublicKey key) {
        if (key == null) return (false);
        if (key.getValidSeconds() == 0) return (false);
        Date now = new Date();
        return (key.getCreationTime().getTime() + (key.getValidSeconds() * 1000) - now.getTime() < 0);
    }
    
    private static void markUIDUpdated (GuardKeys key) {
    	String command = "UPDATE og_KeyTable SET version = 0 WHERE cid = ? AND id = ? AND version = -1";
    	DbCommand com = new DbCommand(command);
    	com.addVariables(key.contextid);
    	com.addVariables(key.userid);
    	DbQuery db = new DbQuery();
    	try {
			db.write(com, key.userid, key.contextid);
		} catch (Exception e) {
			logger.error("Problem updating version from -1 for PGP correction, User " + key.userid + " CID: " + key.contextid);
		}
    	db.close();
    }
    
    
    public static JsonArray getAllKeys (int id, int cid) throws Exception {
        Access acc = new Access();
        ArrayList <GuardKeys> keys = acc.getAllKeys(id, cid);
        JsonArray keysData = new JsonArray();
        for (int i = 0; i < keys.size(); i++) {
            GuardKeys key = keys.get(i);
            JsonArray pubKeyData = PgpHandler.getKeyInfo(key.pubring);
            if (key.getPGPSecretKeyRing() != null) {  // If we have private key data, see if associated with each public
                JsonArray privData = PgpHandler.getPrivateInfo(key.getPGPSecretKeyRing());
                for (int j = 0; j < pubKeyData.size(); j++) {
                    JsonObject pubkey = pubKeyData.get(j).getAsJsonObject();
                    String keyid = pubkey.get("KeyID").getAsString();
                    boolean hasPrivate = false;
                    for (int k = 0; k < privData.size(); k++) {
                        if (privData.get(k).getAsJsonObject().get("KeyID").getAsString().equals(keyid)) {
                            hasPrivate = true;
                            continue;
                        }
                    }
                    if (hasPrivate) {
                        pubkey.addProperty("hasPrivate", true);
                    } else {
                        pubkey.addProperty("hasPrivate", false);
                    }
                    if (key.current) pubkey.addProperty("current", true);
                    pubkey.addProperty("GuardKey", Long.toString(key.keyid));
                }  
            } else {  // No private key, just public
                for (int j = 0; j < pubKeyData.size(); j++) {
                    JsonObject pubkey = pubKeyData.get(j).getAsJsonObject();
                    if (key.current) pubkey.addProperty("current", true);
                    pubkey.addProperty("GuardKey", Long.toString(key.keyid));
                    pubkey.addProperty("hasPrivate", false);
                    
                }
            }
            
            keysData.add(pubKeyData);
        }
        return (keysData);
        
    }

}
