/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.SignatureException;
import java.util.ArrayList;
import java.util.Iterator;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xbill.DNS.Lookup;
import org.xbill.DNS.Record;
import org.xbill.DNS.SRVRecord;
import org.xbill.DNS.TextParseException;
import org.xbill.DNS.Type;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.server.connectionPooling.HttpConnectionPoolService;
import com.openexchange.guard.validator.EmailValidator;

public class PgpKeyLookup {


	private static Logger logger = LoggerFactory.getLogger(PgpKeyLookup.class);


	/**
	 * Get a list of SRV Records for PGP Servers for given domain
	 * @param email
	 * @return
	 * @throws TextParseException
	 */
	private static ArrayList<SRVRecord> getSrvs (String email) throws TextParseException {
	    ArrayList <SRVRecord> list = new ArrayList<SRVRecord>();
	    if (!email.contains("@")) {
            return (list);
        }
	    String domain = email.substring(email.indexOf("@") + 1);
	    Record[] records = new Lookup("_hkp._tcp." + domain, Type.SRV).run();
        if (records == null) {
            return (list);
        }
        for (int i = 0; i < records.length; i++) {
            SRVRecord srv = (SRVRecord) records[i];
            list.add(srv);

        }
        return(list);
	}

	/**
	 * Get response from PGP Public Key server
	 * @param url
	 * @param params
	 * @return
	 * @throws IllegalStateException
	 * @throws IOException
	 */
    private static String getResponse(String url, String params) throws IllegalStateException, IOException {

        //CloseableHttpClient httpClient = OxDbConn.httpClient;
        CloseableHttpClient httpClient = HttpConnectionPoolService.getClient();
        String header = "";
        String port = "";
        if (url.startsWith("hkp")) {
        	url = url.replace("hkp", "http");
        }
        HttpGet getRequest = new HttpGet(url + port + params);
        getRequest.addHeader("accept", "application/json");
        getRequest.setHeader("Connection", "close");
        CloseableHttpResponse response = httpClient.execute(getRequest);
        if (response.getStatusLine().getStatusCode() != 200) {
        	try {
        		response.close();
        		getRequest.releaseConnection();
        	} catch (Exception e2) {
        		logger.error("unable to close failed connection at PGP getRequest", e2);
        	}
            throw new RuntimeException("Failed : HTTP error code : " + response.getStatusLine().getStatusCode() + " " + response.getStatusLine().getReasonPhrase());
        }
        String data = EntityUtils.toString(response.getEntity(), "UTF-8");
        logger.debug("Data from remote PGP server received");
        response.close();
		getRequest.releaseConnection();
        return (data);
    }

    /**
     * Parse machine readable response from PGP Public Key server
     * @param response
     * @param email
     * @return
     */
    private static ArrayList<String> parseResponse (String response, String email) {
    	String [] lines = response.split("\n");
    	int count = 0;
    	ArrayList<String> pubids = new ArrayList<String>();
    	String pub = "";
    	for (String line : lines) {
    		if (line.startsWith("info")) {
    			int i = line.lastIndexOf(":");
    			try {
    				count = Integer.parseInt(line.substring(i+1).trim());
    			} catch (Exception ex) {
    			    logger.error("Problem parsing response from remote server ", ex);
    				return (null);
    			}
    		}
    		if (line.startsWith("pub")) {
    			String [] data = line.split(":");
    			pub = data[1].trim();
				if (pub.length() > 16) {
					pub = pub.substring(pub.length() - 16);
				}
    		}
    		if (line.startsWith("uid")) {
    			if (line.contains(email)) {
    				if (!pubids.contains(pub)) {
                        pubids.add(pub);
                    }
    			}
    		}

    	}
    	return (pubids);
    }

    private static String addQueryString(String url) {

        if(url.endsWith("?")) {
            //Query string has already been defined in the configuration
            return url;
        }
        //adding default query string
        return url + "/pks/lookup?";
    }

    /**
     * Query configured PGP Public Key servers for PGP keys for email address.  Returns array of PGPRings
     * @param email
     * @return
     * @throws IllegalStateException
     * @throws IOException
     * @throws SignatureException
     * @throws PGPException
     */
	private static ArrayList<PGPPublicKeyRing> getPublic (String email) throws IllegalStateException, IOException, SignatureException, PGPException {
		if (Config.pgpLocations.size() == 0) {
            return (null);
        }
		ArrayList<PGPPublicKeyRing> rings = new ArrayList<PGPPublicKeyRing>();
		for (int i = 0; i < Config.pgpLocations.size(); i++) {
			String url = Config.pgpLocations.get(i);
			url = addQueryString(url);
			//String params = "/pks/lookup?op=index&options=mr&search=" + email;
			String params = "op=index&options=mr&search=" + email;
			try {
				String response = getResponse (url, params);
				ArrayList <String> ids = parseResponse (response, email);
				for (int j = 0; j < ids.size(); j++) {
					PGPPublicKeyRing ring = getKeys(ids.get(j), url);
					if (ring != null) {
					    rings.add(ring);
					}
				}
			} catch (RuntimeException e) {
				logger.debug("Negative response geting PGP key for " + email + " : " + e.getMessage());
			}
			if (rings.size() > 0)
             {
                return (rings);   // We aren't going to iterate all of the available servers if we have a hit
            }
		}
		return (rings);
	}

	private static PGPPublicKeyRing getKeys (String id, String url) throws IOException, SignatureException, PGPException {
	    //String params = "/pks/lookup?op=get&search=0x" + id;
	    //String params = "/pks/lookup?op=get&options=mr&search=0x" + id;
	    String params = "op=get&options=mr&search=0x" + id;
        String output = getResponse (url, params);
        if (output.contains("PUBLIC KEY BLOCK")) {
            output = output.substring(output.indexOf("-----BEGIN"));
            output = output.substring(0, output.lastIndexOf("BLOCK-----") + 10);
            PGPPublicKeyRing ring = PGPPublicHandler.getKeyRing(output);
            if (ring == null) {
                logger.error("Unable to import Public KeyRing");
            } else {
                if (ring.getPublicKey().isRevoked()) {
                    logger.debug("Key revoked, not adding");
                } else {
                    return(ring);
                }
            }
        }
        return (null);
	}

	private static ArrayList<PGPPublicKeyRing> getSRVRings (SRVRecord current, String email) {
	    ArrayList<PGPPublicKeyRing> rings = new ArrayList<PGPPublicKeyRing>();
	    int port = current.getPort();
        String target = current.getTarget().toString();
        if (target.endsWith(".")) {
            target = target.substring(0, target.length()-1);
        }
        String url = "http://" + target + ":" + port;
        if (port == 443) {
            url = "https://" + target;
        }
        //We use the default query string here, because the URL  does not come
        //from the configuration but from the SVR record so we assume that the
        //Server talks default HKP: RFC draft http://tools.ietf.org/html/draft-shaw-openpgp-hkp-00
        String params = "/pks/lookup?op=index&options=mr&search=" + email;
        String getKeyUrl = url + "/pks/lookup?";
        try {
            String response = getResponse (url, params);
            ArrayList <String> ids = parseResponse (response, email);
            if (ids.size() > 0) {
                for (int j = 0; j < ids.size(); j++) {

                    PGPPublicKeyRing ring = getKeys(ids.get(j), getKeyUrl);
                    if (ring != null) {
                        rings.add(ring);
                    }
                }
            }
        } catch (Exception ex) {
            logger.error("Error parsing SRV response", ex);
            return (null);
        }
        return (rings);


	}

	/**
	 * Find the next in the priority list of SRV records
	 * @param current
	 * @param records
	 * @return
	 */
	private static SRVRecord findNext (int current, ArrayList <SRVRecord> records) {
	    int index = -1;
	    int lowest = 0;
	    for (int i = 0; i < records.size(); i++) {
	        SRVRecord tocheck = records.get(i);
	        if (tocheck.getPriority() > current) {
	            if ((lowest == 0) || (tocheck.getPriority() < lowest)) {
	                lowest = tocheck.getPriority();
	                index = i;
	            }
	        }
	    }
	    if (index == -1) {
            return(null);
        }
	    return (records.get(index));
	}

	/**
	 * Get key from SRV listed pgp server.  Loop through the priorities if errors
	 * @param email
	 * @return
	 * @throws TextParseException
	 */
	private static ArrayList<PGPPublicKeyRing> getPublicFromSrv (String email) throws TextParseException {
	    ArrayList <SRVRecord> records = getSrvs(email);
	//    Collections.sort(records);
	    ArrayList<PGPPublicKeyRing> rings = new ArrayList<PGPPublicKeyRing>();
	    int current = 0;
	    SRVRecord nextcheck;
	    while ((nextcheck = findNext(current, records)) != null) {
	        rings = getSRVRings(nextcheck, email);
	        if (rings != null)
             {
                return(rings);  // Returns null if there is an error, look for other servers
            }
	        current = nextcheck.getPriority();
	    }
	    return (rings);
	}

    private static String exportPgpPublicKey(PGPPublicKeyRing pubring) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ArmoredOutputStream arm = new ArmoredOutputStream(out);
        Iterator<PGPPublicKey> keys = pubring.getPublicKeys();
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        while (keys.hasNext()) {
            PGPPublicKey k = keys.next();
            k.encode(bout);
        }
        arm.write(bout.toByteArray());
        arm.close();
        bout.close();
        out.close();
        return (new String(out.toByteArray()));
    }

	private static boolean saveRing (PGPPublicKeyRing ring) throws IOException {
	    String command = "INSERT INTO remote_key_cache (id, ring, created) VALUES (?, ?, NOW()) ON DUPLICATE KEY UPDATE ring = ?";
	    DbCommand com = new DbCommand(command);
	    com.addVariables(ring.getPublicKey().getKeyID());
	    com.addVariables(exportPgpPublicKey(ring));
	    com.addVariables(exportPgpPublicKey(ring));
	    DbQuery db = new DbQuery();
	    try {
	        db.writeOxGuard(com);
	    } catch (Exception ex) {
	        logger.error("Problem saving ring to remote_key_cache", ex);
	        db.close();
	        return(false);
	    }
	    db.close();
	    return(true);
	}

	private static boolean saveKeyInfo (long id, String email, long refid) {
	    String command = "INSERT INTO remote_keys (id, email, ref) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE ref = ?";
	    DbCommand com = new DbCommand(command);
	    com.addVariables(id);
	    com.addVariables(email);
	    com.addVariables(refid);
	    com.addVariables(refid);
	    DbQuery db = new DbQuery();
	    try {
	        db.writeOxGuard(com);
	    } catch (Exception ex) {
	        logger.error("Problem storing key reference for cache ", ex);
	        db.close();
	        return(false);
	    }
	    db.close();
	    return (true);

	}

	private static void saveCache (ArrayList<PGPPublicKeyRing> rings) throws IOException {
	    for (int i = 0; i < rings.size(); i++) {
	        PGPPublicKeyRing ring = rings.get(i);
	        Long refid = ring.getPublicKey().getKeyID();
	        Iterator <String> ids = ring.getPublicKey().getUserIDs();
	        saveRing(ring);
	        while (ids.hasNext()) {
	            String id = ids.next().trim();
	            if (id.contains("<")) {
	                id = id.substring(id.indexOf("<") + 1);
	                if (id.endsWith(">")) {
                        id = id.substring(0, id.length()-1);
                    }
	            }
	            if (EmailValidator.validate(id)) {
	                Iterator <PGPPublicKey> keys = ring.getPublicKeys();
	                while (keys.hasNext()) {
	                    PGPPublicKey key = keys.next();
	                    if (!key.isRevoked()) {
                            saveKeyInfo (key.getKeyID(), id, refid);
                        }
	                }
	            }
	        }
	    }
	}

	private static ArrayList<PGPPublicKeyRing> checkCache (String email) throws Exception {
	    ArrayList<PGPPublicKeyRing> rings = new ArrayList<PGPPublicKeyRing> ();
	    String command = "SELECT ring from remote_key_cache rkc INNER JOIN remote_keys rk ON rkc.id = rk.ref WHERE rk.email = ? GROUP BY rkc.id";
	    DbCommand com = new DbCommand(command);
	    com.addVariables(email);
	    DbQuery db = new DbQuery();
	    db.readOG(com);
	    while (db.next()) {
	        PGPPublicKeyRing ring = PgpKeys.createKeyRing(db.rs.getString("ring"));
	        if (ring != null) {
                rings.add(ring);
            }
	    }
	    db.close();
	    if (rings.size() > 0) {
            logger.debug("Remote keys found in cache");
        }
	    return(rings);
	}

	/**
	 * Lookup public keys for email address.  Checks SRV records first, then configured public servers
	 * @param email
	 * @return
	 * @throws Exception
	 */
	public static ArrayList<PGPPublicKeyRing> lookupKeys (String email) throws Exception {

	    ArrayList<PGPPublicKeyRing> rings = checkCache(email);
	    if (rings.size() > 0) {
            return (rings);
        }
	    rings = getPublicFromSrv (email); // Check guard servers, or places with srv records
	    if (rings != null) {
	        if (rings.size() > 0) {
	            logger.debug("Remote keys found through SRV");
	            saveCache (rings);
	            return (rings);
	        }
	    }

	    rings = getPublic (email);  // Parse through Configured list of PGP Public key servers
	    if (rings == null) {
            return(null);
        }
	    if (rings.size() > 0) {
	        logger.debug("Remote keys found in public server");
	        saveCache(rings);
	    }
	    logger.debug("Remote key lookup done, no keys found");
	    return (rings);
	}

	public static RecipKey getRemoteKeys (String email) {
	    try {
	        ArrayList<PGPPublicKeyRing> rings = lookupKeys (email);
	        if (rings == null) {
                return(null);
            }
	        if (rings.size() == 0) {
                return (null);
            }
	            RecipKey recip = new RecipKey();
	            recip.pgpring = rings.get(0);
	            recip.email = email;
	            recip.pgp = true;
	            recip.expired = PgpKeys.isExpired(recip.pgpring.getPublicKey());
	            return (recip);


	    } catch (Exception ex) {
	        logger.error("Error getting remote keys", ex);

	    }
	    return (null);
	}
}
