/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.security.NoSuchProviderException;
import java.security.SignatureException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.Properties;
import javax.mail.Address;
import javax.mail.Authenticator;
import javax.mail.BodyPart;
import javax.mail.Message;
import javax.mail.Message.RecipientType;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.PasswordAuthentication;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimeUtility;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.io.IOUtils;
import org.apache.commons.net.util.Base64;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.bcpg.SymmetricKeyAlgorithmTags;
import org.bouncycastle.bcpg.attr.ImageAttribute;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPPublicKeyRingCollection;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureGenerator;
import org.bouncycastle.openpgp.PGPSignatureSubpacketGenerator;
import org.bouncycastle.openpgp.PGPSignatureSubpacketVector;
import org.bouncycastle.openpgp.PGPUserAttributeSubpacketVector;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.bc.BcPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcaKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcePBESecretKeyDecryptorBuilder;
import org.eclipse.jetty.server.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.encr.Crypto;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.EncryptedObj;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.BadPasswordException;
import com.openexchange.guard.exceptions.GuardMissingParameter;
import com.openexchange.guard.logging.LogAction;
import com.openexchange.guard.mailcreator.Attachment;
import com.openexchange.guard.mailcreator.PGPMailCreator;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.server.CheckBad;
import com.openexchange.guard.server.OxCookie;
import com.openexchange.guard.server.UserData;
import com.openexchange.guard.util.Core;
import com.openexchange.guard.validator.Normalizer;


public class PgpHandler {

	private static Logger logger = LoggerFactory.getLogger(PgpHandler.class);
	

	/**
	 * Decode PGP Inline message
	 * @param request
	 * @param response
	 * @param cookie
	 * @throws Exception
	 */
	 public void getPGPInline(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
		 if (CheckBad.isBad(cookie.JSESSIONID, Config.bad_password_count)) {
             com.openexchange.guard.util.Core.sendFail(response, "Exceeded : " + Config.bad_minute_lock);
             return;
         }
     	JsonObject jdata = Core.getJSON(request);
	    int userid = Core.getIntParameter(request, "userid", true);
	    int cid = Core.getIntParameter(request, "cid", true);
	    LogAction.setUser(userid, cid);
     	String auth = Core.getStringParameter(request, "auth");
     	String password = Core.getStringFromJson(jdata, "password");
     	if (auth != null) {
     		if (auth.length() > 20) {
	        		UserData data = new UserData(auth, cookie);
	        		if ((data.userid == -1) || (data.encr_password == null)) {
	        			LogAction.setFail();
	        			logger.debug("Bad authorization");
	        			com.openexchange.guard.util.Core.sendFail(response, "Bad authorization");
	                    CheckBad.addBad(cookie.JSESSIONID);
	                    return;
	        		} else
	        			password = data.encr_password;
     		}
     	}
     	if (password == null) {
     		LogAction.setFail();
     		logger.error("No password for pgp inline");
     		Core.sendFail(response, "No password");
     		return;
     	}
     	String folder = Core.getStringParameter(request, "folder", true);

        String emailid = request.getParameter("emailid");
        int index = Core.getIntParameter(request, "index", true);
        com.openexchange.guard.ox.Api ap = new com.openexchange.guard.ox.Api(cookie, request);
        String emailString = ap.getMail(emailid, folder);
        JsonObject emailData = Core.getJSON(emailString);
        JsonObject email = null;
        try {
        	email = emailData.get("data").getAsJsonObject();
        } catch (Exception ex) {
        	LogAction.setFail();
        	logger.error("Bad email JSON ", ex);
        	Core.sendFail(response, "Bad email JSON");
        	return;
        }
        JsonObject att = email.get("attachments").getAsJsonArray().get(index).getAsJsonObject();
        String content = att.get("content").getAsString();
        int i = content.indexOf("-----");
        if (i < 0) {
        	LogAction.setFail();
        	logger.info("No PGP Data found");
        	Core.sendFail(response, "No PGP");
        	return;
        }
        content = "\r\n" + content.substring(i).replace("<br>", "\r\n");
        PGPStringProcessor pgpDecoder = new PGPStringProcessor();
        try {
            PGPResult decoded = pgpDecoder.DecryptString(password, content, userid, cid);
            JsonObject results = new JsonObject();
            if (decoded.error != null) {
            	LogAction.setFail();
            	logger.info("Failed decode " + decoded.error);
            	Core.sendFail(response, decoded.error);
            	return;
            }
            results.addProperty("newcontent", Normalizer.normalizeString(checkHTML(new String (decoded.decoded, "UTF-8"))));
            JsonObject pgpresults = new JsonObject();
            
            pgpresults.add("signature", new JsonPrimitive(decoded.signature));
            pgpresults.add("verified", new JsonPrimitive(decoded.verified));
            pgpresults.add("missingPublicKey", new JsonPrimitive(decoded.missingPublicKey));
            results.add("results", pgpresults);
            results.addProperty("epassword", Crypto.encodeEPass(password, cookie, userid, cid));
            LogAction.setSuccess(true);
            logger.info("OK Decode PGP Inline email");
            Core.sendOK(response, results.toString());
        } catch (BadPasswordException e) {
            LogAction.setFail();
            logger.info("Bad password");
            Core.sendFail(response, "Bad Password");
            return;
        }
        
	 }
	 
	private String checkHTML(String data) {
		if (data.contains("<div") || data.contains("<br")) return(data);
		data = data.replace("\n", "<br>");
		return (data);
	}
	
	/**
	 * Verify the signature
	 * inline implies search for inline signature
	 * @param request
	 * @param response
	 * @param cookie
	 * @throws Exception
	 */
	public void verifySignature (HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
        String folder = request.getParameter("folder");
        int userid = Integer.parseInt(request.getParameter("userid"));
        int cid = Integer.parseInt(request.getParameter("cid"));
        LogAction.setUser(userid, cid);
        String emailid = request.getParameter("emailid"); 
        int index = Core.getIntParameter(request, "attach", true);
        String inline = Core.getStringParameter(request, "inline");
        com.openexchange.guard.ox.Api ap = new com.openexchange.guard.ox.Api(cookie, request);
        String mime = ap.getMime(emailid, folder);
        InputStream in = new ByteArrayInputStream(mime.getBytes("UTF-8"));
        Session s = Session.getDefaultInstance(new Properties());
        MimeMessage message = new MimeMessage(s, in);
        message.saveChanges();

        
        String signature = "";
        String content = "";
        
        if (inline != null) {
            Object body = message.getContent();
        	PGPResult result = new PGPResult();
        	String section = "";
        	if (body instanceof Multipart || body instanceof MimeMultipart)  {
        	    Multipart mp = (Multipart) body;
        	    BodyPart bp = mp.getBodyPart(index);
        	    String[] encoding = bp.getHeader("Content-Transfer-Encoding");
        	    if (encoding != null) {
        	        String encoded = encoding[0];
        	        InputStream encodedStream = bp.getInputStream();
        	        InputStream decoded = MimeUtility.decode(encodedStream, encoded);
        	        ByteArrayOutputStream out = new ByteArrayOutputStream();
        	        IOUtils.copy(decoded, out);
        	        section = new String(out.toByteArray(), "UTF-8");
        	        out.close();
        	        in.close();
        	        encodedStream.close();
        	    } else {
                    ByteArrayOutputStream writer = new ByteArrayOutputStream();
                    mp.getBodyPart(index).writeTo(writer);
                    section = writer.toString("UTF-8");
                    section = MimeUtility.decodeText(section);
                    writer.close();
        	    }

        	} else {
        	    if (body instanceof String) {
        	        section = (String) body;
        	    }
        	}
        	int i = section.indexOf("----BEGIN PGP SIGNATURE");
        	if (i > -1) {
        		signature = section.substring(i);
        		int j = section.indexOf("Hash:");
        		if (j > -1) {
        			j = section.indexOf("\n", j) + 3;
        			section = section.substring(j, i-3);
        			if (section.startsWith("\n")) {  // If begins with \n and not \r\n, likely thunderbird
        			    section = section.substring(1);
        			}
        		}
        		
        	} else {
        		result.signature = false;
        		LogAction.setSuccess();
        		Core.sendOK(response, result.getJsonString());
        		return;
        	}
        	PGPUtils utils = new PGPUtils();
        	ByteArrayInputStream data = new ByteArrayInputStream (signature.getBytes("UTF-8"));
        	result = utils.verifyInlineSignature(data, section, userid, cid);
        	JsonObject returnJson = new JsonObject();
            returnJson.addProperty("result", result.getJsonString());
            returnJson.addProperty("content", section.replace("\n", "<br>"));
            LogAction.setSuccess();
        	Core.sendOK(response, returnJson.toString());
        	return;
        } else {
            Multipart mp = (Multipart) message.getContent();
	        for (int i = 0; i < mp.getCount(); i++) {
	        	ByteArrayOutputStream writer = new ByteArrayOutputStream();
	        	mp.getBodyPart(i).writeTo(writer);
	        	String section = writer.toString("UTF-8");
	        	if (section.contains("application/pgp-signature")) {
	        		signature = section;
	        	} else {
	        		content = section;
	        	}
	        	writer.close();
	        }
        }
        PGPUtils util = new PGPUtils();
        PGPResult result = util.verifySignature(content, signature, userid, cid);
        JsonObject resultJson = new JsonObject();
        resultJson.addProperty("result", result.getJsonString());
        LogAction.setSuccess(true);
        logger.info("Signature verification, signature verification: " + Boolean.toString(result.verified));
        Core.sendOK(response, resultJson.toString());
	}
	
	
	/**
	 * Get an inline PGP attachment (essentially file decode)
	 * @param request
	 * @param response
	 * @param cookie
	 * @throws Exception
	 */
	public void getInlineAttach (HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
		int userid = Integer.parseInt(request.getParameter("userid"));
        int cid = Integer.parseInt(request.getParameter("cid"));
        LogAction.setUser(userid, cid);
        String folder = Core.getStringParameter(request, "folder", true);
        String  emailid = Core.getStringParameter(request, "emailid", true);
        String attach = Core.getStringParameter(request, "attach", true);
        String filename = Core.getStringParameter(request, "attname");
        String epass = Core.getStringParameter(request, "epassword", true);
        String password = Crypto.decodeEPass(epass, cookie);
        String downloadStr = Core.getStringParameter(request, "download");
        if (CheckBad.isBad(emailid + "a:" + attach, Config.bad_password_count)) {
            Core.sendFail(response, "Lockout");
            logger.warn("Lockout");
            return;
        }
        boolean download = false;
        if (downloadStr != null) {
        	if (downloadStr.toLowerCase().trim().equals("true")) download = true;
        }
        if (password == null) {
        	LogAction.setFail();
        	logger.info("Password absent");
        	Core.sendFail(response, "Bad Password");
        	CheckBad.addBad(emailid + "a:" + attach);
        	return;
        }
        
        if (download)
        	response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
        decodeInlineAttach(emailid, folder, attach, filename, userid, cid, password, response, request, cookie);
        LogAction.setSuccess(true);
 
	}
	
	/**
	 * Do the actual decode to stream of the inline file.  Retrieve from OX backend and send decoded
	 * @param emailid
	 * @param folder
	 * @param attach
	 * @param filename
	 * @param userid
	 * @param cid
	 * @param password
	 * @param response
	 * @param request
	 * @param cookie
	 * @throws Exception
	 */
	public void decodeInlineAttach (String emailid, String folder, String attach, String filename, int userid, int cid, String password, Object response, HttpServletRequest request, OxCookie cookie) throws Exception {
	    Api ap = new Api(cookie, request);
        Attachment file = ap.getPlainAttachment(emailid, attach, folder);
        ByteArrayInputStream in = new ByteArrayInputStream (file.content);
        OutputStream out = null;
        if (response.getClass().equals(Response.class)) {
            HttpServletResponse resp = (HttpServletResponse) response;
            if (CheckBad.isBad(folder + "/" + emailid + "a:" + attach, Config.bad_minute_lock)) {
                resp.setStatus(HttpServletResponse.SC_FORBIDDEN);
                resp.getWriter().write("Lockout");
                logger.warn("Lockout");
                return;
            }
            resp.setContentType(file.type);
            resp.addHeader("Access-Control-Allow-Origin", "*");
            resp.setStatus(HttpServletResponse.SC_OK);
            out = ((HttpServletResponse) response).getOutputStream();
        } else out = (OutputStream) response;
        try {
            PGPResult result = PGPUtils.decryptFile(in, out, userid, cid, password, true);
            if (result.error != null) {
                if (response.getClass().equals(Response.class)) {
                    HttpServletResponse resp = (HttpServletResponse) response;
                    resp.setContentType("text/html");
                    resp.addHeader("Access-Control-Allow-Origin", "*");
                    resp.setStatus(HttpServletResponse.SC_NOT_ACCEPTABLE);
                    out.write(result.error.getBytes());
                    logger.error(result.error);
                } else {
                    out.write(result.error.getBytes());
                }
            }
        } catch (BadPasswordException e) {
            LogAction.setFail();
            logger.info("Bad password");
            sendFail(response, "Bad Password");
            CheckBad.addBad(folder + "/" + emailid + "a:" + attach);
            return;
        }
        in.close();
        out.close();
        logger.info("Attachment retrieved " + filename);
	}
	
	public byte[] decodeInlineAttach (String emailid, String folder, String attach, String filename, int userid, int cid, String password, HttpServletRequest request, OxCookie ck) throws Exception {
	    ByteArrayOutputStream out = new ByteArrayOutputStream();
	    decodeInlineAttach(emailid, folder, attach, filename, userid, cid, password, out, request, ck);
	    out.close();
	    return(out.toByteArray());
	}
	
	public void getAttach (HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) {
		String name = "";
		try {
			name = Core.getStringParameter(request, "attname", true);
		} catch (GuardMissingParameter e) {
			LogAction.setFail();
			logger.info("Missing attname parameter");
			Core.sendFail(response, "Missing attname parameter");
			return;
		}
		DecodePGPemail (request, response, cookie, "open", name);
	}
	
	/**
	 * Get an attachment from PGP Mime email for frowarding or from draft
	 * @param out
	 * @param request
	 * @param cookie
	 * @param emailid
	 * @param folder
	 * @param userid
	 * @param cid
	 * @param name
	 * @param password
	 * @throws Exception
	 */
	public byte[] getAttach (HttpServletRequest request, OxCookie cookie, String emailid, String folder, int userid, int cid, String name, String password ) throws Exception {
	    ByteArrayOutputStream out = new ByteArrayOutputStream();
	    doDecode(emailid, folder, userid, cid, "open", name, password, request, cookie, out);
	    out.close();
	    return(out.toByteArray());
	}
	
	public void getAttachCID (HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) {
        String contentid = "";
        try {
            contentid = Core.getStringParameter(request, "content_id", true);
        } catch (GuardMissingParameter e) {
        	LogAction.setFail();
        	logger.info("Missing attname parameter");
            Core.sendFail(response, "Missing attname parameter");
            return;
        }
        DecodePGPemail (request, response, cookie, "cid", contentid);
    }
	
	   public void getPGPemail(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) {
	        DecodePGPemail (request, response, cookie, "mail", null);
	    }
	   
	/**
	 * Decode PGP Mime email
	 * @param request
	 * @param response
	 * @param cookie
	 * @param action
	 * @param name
	 */
    public void DecodePGPemail(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie, String action, String name) {
        try {
        	JsonObject jdata = Core.getJSON(request);
            int userid = Integer.parseInt(request.getParameter("userid"));
            int cid = Integer.parseInt(request.getParameter("cid"));
            LogAction.setUser(userid, cid);
            if (CheckBad.isBad(cookie.JSESSIONID, Config.bad_password_count)) {
                com.openexchange.guard.util.Core.sendFail(response, "Exceeded : " + Config.bad_minute_lock);
                LogAction.setFail();
                logger.info("Lockout");
                return;
            }
        	String auth = request.getParameter("auth");
        	String password = jdata == null ? null : (jdata.has("password") ? jdata.get("password").getAsString() : null);
        	if (auth != null) {
        		if (auth.length() > 20) {
	        		UserData data = new UserData(auth, cookie);
	        		if ((data.userid == -1) || (data.encr_password == null)) {
	        			com.openexchange.guard.util.Core.sendFail(response, "Bad authorization");
	                    CheckBad.addBad(cookie.JSESSIONID);
	                    LogAction.setFail();
	                    logger.info("Bad authorization");
	                    return;
	        		} else
	        			password = data.encr_password;
        		}
        	}
        	if (password == null) {
        		String epassword = Core.getStringParameter(request, "epassword");
        		if (epassword != null) {
                        password = Crypto.decodeEPass(epassword, cookie);
        		}
        	}
        	if (password == null) {
        		Core.sendFail(response, "No password");
        		LogAction.setFail();
        		logger.info("No Password");
        		return;
        	}
        	String folder = request.getParameter("folder"); 
            String emailid = request.getParameter("emailid");
            
            doDecode(emailid, folder, userid, cid, action, name, password, request, cookie, response);

        } catch (Exception ex) {
        	LogAction.setFail();
            logger.error("Error getting PGP email " + request.getParameter("emailid"), ex);
        }

    }
    
    private void sendFail (Object response, String answer) throws IOException {
        if (response.getClass().equals(Response.class)) {
            Core.sendFail((Response) response, answer);
        } else {
            ((OutputStream) response).write(answer.getBytes());
        }
    }
    
    private void sendOK (Object response, String answer) throws IOException {
        if (response.getClass().equals(Response.class)) {
            Core.sendOK((Response) response, answer);
        } else {
            ((OutputStream) response).write(answer.getBytes());
        }
    }
    
    /**
     * Do the actual decode of PGP Mime email, send to response object
     * @param emailid
     * @param folder
     * @param userid
     * @param cid
     * @param action
     * @param name
     * @param password
     * @param request
     * @param cookie
     * @param response
     * @throws Exception
     */
    private void doDecode (String emailid, String folder, int userid, int cid, String action, String name, String password, HttpServletRequest request, OxCookie cookie, Object response) throws Exception {
        com.openexchange.guard.ox.Api ap = new com.openexchange.guard.ox.Api(cookie, request);
        String mime = ap.getMime(emailid, folder);
        Session s = Session.getDefaultInstance(new Properties());
        InputStream in = new ByteArrayInputStream(mime.getBytes("UTF-8"));
        MimeMessage message = new MimeMessage(s, in);
        try {
            message.saveChanges();
        } catch (Exception ex) {
            LogAction.setFail();
            logger.error("Problem getting mime from OX backend", ex);
            return;
        }

        Multipart mp = (Multipart) message.getContent();
        Part p = null;
        if (message.getHeader("X-OXGUARD-GUEST") != null) {
            // Guest email being read by user
            p = getGuestPGPPart (mp);
        } else {
            p = getPGPPart(mp);  // Regular MIME email
        }


        if (p == null) {
            LogAction.setFail();
            logger.error("Problem getting PGP email components");
            if (response.getClass().equals(Response.class)) {
                Core.sendFail((HttpServletResponse) response, "Not properly formatted PGP email");
            } else 
            return;
        }
        

        PGPStringProcessor pgp = new PGPStringProcessor();
        Access acc = new Access();

        Address[] fromaddr = message.getFrom();
        String fromemail = fromaddr[0].toString();
        if (fromemail.contains("<")) {
            fromemail = fromemail.substring(fromemail.indexOf("<") + 1, fromemail.indexOf(">"));
        }
        ArrayList<PGPPublicKeyRing> senderRing = acc.getPublicKeyRing(fromemail, userid, cid);
        PGPPublicKeyRing sender = senderRing.size() == 0 ? null : senderRing.get(0);
        if (sender == null) {
            logger.info("No public key for " + fromemail + " found");
        }
        InputStream inp = p.getInputStream();
        PGPResult result = null;
        try {
            result = pgp.DecryptStream(password, inp, userid, cid);
        } catch (BadPasswordException ex) {
            LogAction.setFail();
            CheckBad.addBad(cookie.JSESSIONID);
            sendFail(response, "Bad password");
            logger.info("Bad Password");
            return;
        }
        if (result.error != null) {
            LogAction.setSuccess(false);
            logger.info("Fail decode email " + result.error);
            sendFail(response, result.error);
        } else {
            switch (action) {
            case "mail": 
                LogAction.setSuccess(true);
                logger.info("Decoded PGP Mime email");
                sendOK(response, createReturnJson(result, password, userid, cid, cookie));
                break;
            case "open":
                boolean download = (Core.getStringParameter(request, "download") == null ? false : (Core.getStringParameter(request, "download").toLowerCase().equals("true"))); 
                getAttach (response, result, name, download);
                break;
            case "cid":
                getAttachByCid (response, result, name);
            }
        }
    }
    

    /**
     * Search Mulipart message for PGP MIME
     * @param mp
     * @return
     * @throws MessagingException
     * @throws IOException
     */
    private Part getPGPPart (Multipart mp) throws MessagingException, IOException {
    	boolean pgp = false;
    	Part p = null;
    	for (int i = 0; i < mp.getCount(); i++) {
    		Object content = mp.getBodyPart(i).getContent();
    		String type = mp.getBodyPart(i).getContentType();
    		if (type.contains("pgp-encrypted")) pgp = true;
    		if (type.contains("application/octet-stream") && pgp) {
    			return (mp.getBodyPart(i));
    		}
    		if (content instanceof Multipart || content instanceof MimeMultipart) {
    			p = getPGPPart ((Multipart) content);
    		}
    		if (p!= null) return (p);
    		
    	}
    	return (null);
        
    }
    
    private Part getGuestPGPPart(Multipart mp) throws MessagingException, IOException {
        Part p = null;
        for (int i = 0; i < mp.getCount(); i++) {
            Object content = mp.getBodyPart(i).getContent();
            String type = mp.getBodyPart(i).getContentType();
            if (type.contains("encrypted.asc")) {
                return (mp.getBodyPart(i));
            }
            if (content instanceof Multipart || content instanceof MimeMultipart) {
                p = getGuestPGPPart ((Multipart) content);
            }
            if (p!= null) return (p);
        }
        return(null);
    }
    
    /**
     * Save vcard attachment into contacts
     * @param request
     * @param response
     * @param ck
     * @throws Exception
     */
    public void importVCard (HttpServletRequest request, HttpServletResponse response, OxCookie ck) throws Exception {
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        String emailid = Core.getStringParameter(request, "emailid", true);
        String attachid = Core.getStringParameter(request, "attach", true);
        String epass = Core.getStringParameter(request, "epass", true);
        String folder = Core.getStringParameter(request, "folder");
        String filename = Core.getStringParameter(request, "filename", true);
        String contactFolder = Core.getStringParameter(request, "contacts", true);
        boolean inline = Core.getBooleanParameter(request, "inline");
        String password = Crypto.decodeEPass(epass, ck);
        if (password.equals("")) {
            Core.sendFail(response, "Bad password");
            return;
        }
        byte[] vcard = null;
        if (inline) {
            vcard = decodeInlineAttach (emailid, folder, attachid, filename, userid, cid, password, request, ck);  
        } else {
            vcard = getAttach (request, ck, emailid, folder, userid, cid, filename, password);
        }
        if (vcard != null) {
            String test = new String(vcard, "UTF-8");
            if (test.indexOf("BEGIN:VCARD") > -1) {
                Api ap = new Api(ck, request);
                Attachment vc = new Attachment();
                vc.encrContent = vcard;
                vc.filename = filename;
                vc.type = "text/vcard";
                JsonObject resp = ap.importVcard(vc, contactFolder);
                if (Core.checkFail(resp)) {
                    Core.sendFail(response, "unable to import");
                    return;
                }
                if (resp.has("data")) {
                    try {
                        JsonObject data = resp.get("data").getAsJsonArray().get(0).getAsJsonObject();
                        Core.sendOK(response, data.toString());
                    } catch (Exception e) {
                        logger.error("Problem getting JSON response for uploaded card", e);
                        Core.sendOK(response, "OK");
                    }
                    return;
                }
                Core.sendOK(response, "OK");
            } else {
                Core.sendFail(response, "No Vcard");
            }

        }
        
        
    }
    
    /**
     * Get an attachment from PGP Mime email and return to stream
     * @param response
     * @param result
     * @param name
     * @param download
     * @throws MessagingException
     * @throws IOException
     */
    public static void getAttach (Object response, PGPResult result, String name, boolean download) throws MessagingException, IOException {
    	Session s = Session.getDefaultInstance(new Properties());
        InputStream in2 = new ByteArrayInputStream(result.decoded);
        MimeMessage message2 = new MimeMessage(s, in2);
        Multipart mp = (Multipart) message2.getContent();
        for (int i = 0; i < mp.getCount(); i++) {
            JsonObject attachment = new JsonObject();
            Part p = mp.getBodyPart(i);
            String filename = p.getFileName();
            String type = p.getContentType();
            if (filename != null) {
            	if (filename.toLowerCase().trim().equals(name.toLowerCase().trim())) {
                    if (type!=null) {
                    	if (type.contains(";")) {
                    		type = type.substring(0, type.indexOf(";"));
                    		if (type.startsWith("grd/")) {  // For old Guard, remove encoding
                    		    type = type.replace("grd/", "").replace("--", "/");
                    		}
                    	}
                    }
                    if (response.getClass().equals(Response.class)){
                        HttpServletResponse resp = (HttpServletResponse) response;
                        if (download) {
                            resp.setContentType(type);
                            resp.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
                            resp.addHeader("Access-Control-Allow-Origin", "*");
                            resp.setStatus(HttpServletResponse.SC_OK);
                        } else {
                            resp.setContentType(type);
                            resp.setStatus(HttpServletResponse.SC_OK);
                        }
                        IOUtils.copy(p.getInputStream(), resp.getOutputStream());
                        resp.flushBuffer();
                    } else {
                        IOUtils.copy(p.getInputStream(), (OutputStream) response);
                    }
            		
            		LogAction.setSuccess(true);
            		logger.info("Sending decoded attachment " + filename);
            		break;
            	}
            }
        }
        in2.close();
    }
    
    /**
     * Get an attachment from PGP Mime email and return to stream
     * @param response
     * @param result
     * @param cid
     * @throws MessagingException
     * @throws IOException
     */
    public static void getAttachByCid (Object responseObj, PGPResult result, String match_cid) throws MessagingException, IOException {
        HttpServletResponse response = (HttpServletResponse) responseObj;
        Session s = Session.getDefaultInstance(new Properties());
        InputStream in2 = new ByteArrayInputStream(result.decoded);
        MimeMessage message2 = new MimeMessage(s, in2);
        Multipart mp = (Multipart) message2.getContent();
        for (int i = 0; i < mp.getCount(); i++) {
            Part p = mp.getBodyPart(i);
            String [] cids = p.getHeader("Content-ID");
            String cid = null;
            if (cids != null) {
                cid = cids[0].replace("<", "").replace(">", ""); 
                if (cid.toLowerCase().trim().equals(match_cid.toLowerCase().trim())) {
                    String type = p.getContentType();
                    if (type!=null) {
                        if (type.contains(";")) {
                            type = type.substring(0, type.indexOf(";"));
                        }
                    }
                    response.setContentType(type);
                    response.setStatus(HttpServletResponse.SC_OK);
                    IOUtils.copy(p.getInputStream(), response.getOutputStream());
                    LogAction.setSuccess(true);
                    logger.info("Sending decoded content-id attachment");
                    response.flushBuffer();
                    break;
                }
            }
        }
        in2.close();
    }
    
    public static void ParseMultipart (Multipart mp, JsonArray attachments) throws MessagingException, IOException {
    	 JsonObject plain = null;
    	 boolean html = false;
    	 for (int i = 0; i < mp.getCount(); i++) {
             JsonObject attachment = new JsonObject();
             Part p = mp.getBodyPart(i);
             if (p.getContentType() != null) {
                 String contentType = p.getContentType();
                 boolean isAttachment = false;
                 if (contentType.startsWith("multip")) {
                	 ParseMultipart ((Multipart) p.getContent(), attachments);
                 } else {
	                 if (contentType.contains(";"))
	                     contentType = contentType.substring(0, contentType.indexOf(";"));
	                 attachment.add("content_type", new JsonPrimitive(contentType));
	                 if (contentType.contains("text/html") || contentType.contains("text/plain")) {
	                 	attachment.add("content", new JsonPrimitive((String) p.getContent()));
	                     if (p.getFileName() == null) {
	                         attachment.add("disp", new JsonPrimitive("inline"));
	                     } else isAttachment = true;
	                 } else {
	                 	attachment.addProperty("content", "");
	                 	}
	
	             }
	             if (p.getFileName() != null)
	                 attachment.add("filename", new JsonPrimitive(p.getFileName()));
	             if (p.getDisposition() != null)
	                 attachment.add("disp", new JsonPrimitive(p.getDisposition()));
	             attachment.add("size", new JsonPrimitive(p.getSize()));
	             
	             attachment.add("id", new JsonPrimitive(i));
	             if (contentType.contains("text/html")) html = true;
	             if (contentType.contains("text/plain")) {
	                 if (isAttachment) {  // If attachment, then just append
	                     attachments.add(attachment);
	                 } else  // otherwise, if plaintext inline, then save in case we don't have HTML
	                     plain = attachment;
	             } else 
	            	 if (!contentType.startsWith("multip")) attachments.add(attachment);
             }
         }
    	 if (!html) {  // If no HTML, then add the plain
    		 if (plain != null) attachments.add(plain);
    	 }
    }

    private String createReturnJson (PGPResult result, int userid, int cid) throws Exception {
    	return (createReturnJson(result, null, userid, cid, null));
    }
    /**
     * Create JSON to return from decoded PGP including signature verification, epassword
     * @param result
     * @param password
     * @param cookie
     * @return
     * @throws Exception 
     */
    public static String createReturnJson(PGPResult result, String password, int userid, int cid, com.openexchange.guard.server.OxCookie cookie) throws Exception {
        Session s = Session.getDefaultInstance(new Properties());
        InputStream in2 = new ByteArrayInputStream(result.decoded);
        MimeMessage message2 = new MimeMessage(s, in2);
        JsonArray attachments = new JsonArray();
        if (message2.getContent().getClass() == String.class) {
            String content = (String) message2.getContent();
            JsonObject attachment = new JsonObject();
            attachment.addProperty("content_type", "text/plain");
            attachment.addProperty("content", content);
            attachment.addProperty("size", content.length());
            attachment.add("disp", new JsonPrimitive("inline"));
            attachment.add("id", new JsonPrimitive(1));
            attachments.add(attachment);
        } else {
            Multipart mp = (Multipart) message2.getContent();
            ParseMultipart (mp, attachments);
        }
        JsonObject Results = new JsonObject();
        Results.add("signature", new JsonPrimitive(result.signature));
        Results.add("verified", new JsonPrimitive(result.verified));
        Results.add("missingPublicKey", new JsonPrimitive(result.missingPublicKey));
        if (password != null) {
        	Access acc = new Access();
            String token = acc.getToken(cookie.JSESSIONID);
            if (token == null) token = acc.newToken(cookie.JSESSIONID, userid, cid);
        	EncrLib encr = new EncrLib();
        	Results.addProperty("epassword", encr.encryptAES(password, token));
        }
        if (result.error != null)
            Results.add("error", new JsonPrimitive(result.error));
        JsonObject data = new JsonObject();
        data.add("attachments", attachments);
        
        Address [] toaddrs = message2.getRecipients(RecipientType.TO);
        if (toaddrs != null) {
            JsonArray to = new JsonArray();
            for (Address add : toaddrs) {
                JsonArray recip = new JsonArray();

                InternetAddress inetAdd = (InternetAddress) add;
                if(inetAdd.getPersonal() != null) {
                    recip.add(new JsonPrimitive(MimeUtility.decodeText(inetAdd.getPersonal())));
                }
                recip.add(new JsonPrimitive(inetAdd.getAddress()));
                to.add(recip);
            }
            data.add("to", to);
            
        }
        Address [] ccaddrs = message2.getRecipients(RecipientType.CC);
        if (ccaddrs != null) {
            JsonArray cc = new JsonArray();
            for (Address add : ccaddrs) {
                JsonArray recip = new JsonArray();
                InternetAddress inetAdd = (InternetAddress) add;
                if(inetAdd.getPersonal() != null) {
                    recip.add(new JsonPrimitive(MimeUtility.decodeText(inetAdd.getPersonal())));
                }
                recip.add(new JsonPrimitive(((InternetAddress) add).getAddress()));
                cc.add(recip);
            }
            data.add("cc", cc);
            
        }
        data.add("results", Results);
        in2.close();
        // Return normalized result
        return (Normalizer.normalize(data).toString());
    }

    /**
     * Display Armor output of PGP Key Ring from OX user
     * @param request
     * @param response
     * @throws GuardMissingParameter 
     */
    public void getKey(HttpServletRequest request, HttpServletResponse response) throws GuardMissingParameter {
        String email = Core.getStringParameter(request, "email", true);
        if (!email.contains("@")) {
        	LogAction.setFail();
        	logger.info("Bad email format");
            Core.sendFail(response, "Bad email format");
            return;
        }
        ;
        String download = Core.getStringParameter(request, "download");
        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        try {
            GuardKeys keys = acc.getKeysFromEmail(email);
            if (keys != null) {
            	if (download != null) {
            		String filename = "public.asc";  // Default filename
            		Iterator it = keys.pubring.getPublicKey().getUserIDs(); // Change filename to first ID if avail
            		if (it.hasNext()) {
            			filename = ((String) it.next()) + ".asc";
            		}
            		response.setHeader("Content-Disposition", "attachment; filename*=UTF-8''" + URLEncoder.encode(filename, "UTF-8"));
            	}
            	LogAction.setSuccess(true);
            	logger.info("Key retrieved for email " + email);
                Core.sendOK(response, keys.exportPGPPublicKeyRing());
            }
        } catch (Exception ex) {
            LogAction.setSuccess(false);
            logger.error("Error getting key for " + email, ex);
            Core.sendFail(response, "Error processing request");
        }

    }
    
    /**
     * Export private key by ID
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void getPrivateKeybyId(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws Exception {
    	String auth = Core.getStringParameter(request, "auth");
    	int cid = Core.getIntParameter(request, "cid", true);
    	int id = Core.getIntParameter(request, "id", true);
    	LogAction.setUser(id, cid);
    	String guest = Core.getStringParameter(request, "guest");
    	String keyidstring = Core.getStringParameter(request, "keyid", false);
    	Long keyid;
    	if (keyidstring == null) {
    	    keyid = 0L;
    	} else {
    	       try {
    	            keyid = Long.parseLong(keyidstring);
    	        } catch (Exception ex) {
    	        	LogAction.setFail();
    	            logger.error("Problem getting keyid " , ex);
    	            Core.sendFail(response, "Bad keyid");
    	            return;
    	        }
    	}
    	

    	if (guest != null) {
    		cookie.JSESSIONID = cookie.OxReaderID;
    	}
    	String password = "";
     	if (auth != null) {
     		if (auth.length() > 20) {
	        		UserData data = new UserData(auth, cookie);
	        		if ((data.userid == -1) || (data.encr_password == null)) {
	        			com.openexchange.guard.util.Core.sendFail(response, "Bad authorization");
	                    CheckBad.addBad(cookie.JSESSIONID);
	                    LogAction.setFail();
	                    logger.info("Bad authorization");
	                    return;
	        		} else
	        			password = data.encr_password;
     		}
     	}
     	Access acc = new Access();
     //	GuardKeys key = acc.getKeys(id, cid);
     	GuardKeys key;
     	if (keyid != 0) {
     	    key = acc.getKeys(id, cid, keyid, false);
     	} else {
     	    key = acc.getKeys(id, cid);  // If keyid isn't specified, then just get the current  
     	}
     	 
     	if (key == null) {
     		LogAction.setFail();
     		logger.error("Unable to get private key");
     		Core.sendFail(response, "Unable to get key");
     		return;
     		
     	}
     	String exported = PgpKeys.ExportPGPPrivate(key, password);
     	if (exported == null) {
     		LogAction.setFail();
     		logger.error("Unable to export private key");
     		Core.sendFail(response, "Unable to decode");
     		return;
     	}
     	logger.debug("Sending private key");
     	response.addHeader("Content-Disposition", "attachment; filename=" + key.email + "-private.asc");
     	response.addHeader("Content-Type", "application/download");
     	LogAction.setSuccess();
     	Core.sendOK(response, "application/pgp-keys", exported.getBytes("UTF-8"));
    }

    /**
     * Return to HTTP stream a key by ID
     * @param request
     * @param response
     * @throws GuardMissingParameter
     */
    public void getKeybyId(HttpServletRequest request, HttpServletResponse response) throws GuardMissingParameter {
    	 int id = Core.getIntParameter(request, "id", true);
    	 int cid = Core.getIntParameter(request, "cid", true);
    	 LogAction.setUser(id, cid);
    	 String keyid = Core.getStringParameter(request, "keyid");
    	 Long kid = (long) 0;
    	 try {
    	     if (keyid != null)
    	         kid = Long.parseLong(keyid);
    	 } catch (Exception ex) {
    		 LogAction.setFail();
    	     logger.error("problem converting keyid " , ex);
    	     Core.sendFail(response, "Bad keyid");
    	     return;
    	 }
    	 com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
         try {
             GuardKeys keys;
             if (kid != 0) {
                 keys = acc.getKeys(id, cid, kid, false);
             } else 
                 keys = acc.getKeys(id, cid);
             if (keys != null) {
            	 String filename = "public.asc";  // Default filename
         		 Iterator it = keys.pubring.getPublicKey().getUserIDs(); // Change filename to first ID if avail
         	     if (it.hasNext()) {
         			filename = ((String) it.next()) + ".asc";
         		 }
         		response.setHeader("Content-Disposition", "attachment; filename*=UTF-8''" + URLEncoder.encode(filename, "UTF-8"));
         		LogAction.setSuccess();
                 Core.sendOK(response, "application/pgp-keys", keys.exportPGPPublicKeyRing().getBytes());
             }
         } catch (Exception ex) {
        	 LogAction.setFail();
             logger.error("Error getting key for " + id + " " + cid, ex);
             Core.sendFail(response, "Error processing request");
         }
    }
    
    public void getUserKeys (HttpServletRequest request, HttpServletResponse response, OxCookie ck) throws Exception {
        int id = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(id, cid);
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
        	LogAction.setFail();
        	logger.info("Not logged into UI");
            Core.sendFail(response, "Must be logged in");
            return;
        }
        LogAction.setSuccess();
        Core.sendOK(response, PgpKeys.getAllKeys(id, cid).toString());
        return;
    }
    
    /**
     * Return JsonArray of info for PGPSecretKeyRing
     * @param skeyring
     * @return
     */
    public static JsonArray getPrivateInfo (PGPSecretKeyRing skeyring) {
        if (skeyring == null) return (null);
        Iterator skeys = skeyring.getSecretKeys();
        JsonArray keyRingData = new JsonArray();
        String self = "";
        while (skeys.hasNext()) {
            PGPSecretKey key = (PGPSecretKey) skeys.next();
            JsonObject pubKey = new JsonObject();
            if (key.isMasterKey()) self = toHex(key.getKeyID());
            pubKey.addProperty("Key", toHex(key.getKeyID()));
            pubKey.addProperty("KeyID", toFullHex(key.getKeyID()));
            pubKey.addProperty("Master", key.isMasterKey());
            Iterator it = key.getUserIDs();
            int i = 0;
            JsonArray ids = new JsonArray();
            while (it.hasNext()) {
                ids.add(new JsonPrimitive((String) it.next()));
            }
            pubKey.add("ids", ids);
            keyRingData.add(pubKey);
        }
        return (keyRingData);
    }
    
    public static JsonArray getKeyInfo (ArrayList<PGPPublicKeyRing> pkeys) throws SignatureException, PGPException {
        JsonArray data = new JsonArray();
        for (int i = 0; i < pkeys.size(); i++) {
            data.add(getKeyInfo(pkeys.get(i)));
        }
        return (data);
    }
    
    public void checkSignatures (HttpServletRequest request, HttpServletResponse response, OxCookie ck) throws Exception {
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        String keyidString = Core.getStringParameter(request, "keyid", true);
        Long keyid = 0L;
        try {
            keyid = Long.parseLong(keyidString.trim());
        } catch (Exception e) {
            LogAction.setFail();
            logger.error("Bad keyid", e);
            Core.sendFail(response, "Bad keyid");
            return;
        }
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            Core.sendFail(response, "Must be logged in");
            LogAction.setFail();
            logger.info("Fail to check keys, not logged in");
            return;
        }
        Access acc = new Access();
        PGPPublicKeyRing pubring = acc.getPublicKeyRingById(keyid, userid, cid);
        if (pubring == null) {
            Core.sendFail(response, "Unable to retrieve key");
            return;
        }
        ArrayList <SignatureResults> results = PgpKeys.checkSignatures(pubring.getPublicKey(keyid), userid, cid);
        LogAction.setSuccess();
        Core.sendOK(response, getJson(results).toString());
        
    }
    
    private JsonArray getJson (ArrayList<SignatureResults> sigs) {
        JsonArray results = new JsonArray();
        for (int i = 0; i < sigs.size(); i++) {
            JsonObject sig = new JsonObject();
            sig.addProperty("ID", sigs.get(i).getIdVerified());
            sig.addProperty("result", sigs.get(i).getLevel());
            sig.addProperty("certifier", sigs.get(i).getCertifier());
            if (sigs.get(i).hasImage) {
                if (sigs.get(i).getImageType() == 1) {
                    sig.addProperty("image", sigs.get(i).getEmbeddedImage());
                }
            }
            results.add(sig);
        }
        return (results);
    }
    
    /**
     * Print the user info for a PGP public key ring.  Returns as JSON
     * @param pkey
     * @return
     * @throws PGPException
     * @throws SignatureException
     */
    public static JsonArray getKeyInfo (PGPPublicKeyRing pkey) throws PGPException, SignatureException {
    	JsonArray keyRingData = new JsonArray();
    	Iterator keys = pkey.getPublicKeys();
        PGPPublicKey key = null;
        String self = "";
        while (keys.hasNext()) {
        	key = (PGPPublicKey) keys.next();
        	JsonObject pubKey = new JsonObject();
        	if (key.isMasterKey()) self = toHex(key.getKeyID());
        	pubKey.addProperty("Key", toHex(key.getKeyID()));
        	pubKey.addProperty("KeyID", toFullHex(key.getKeyID()));
        	pubKey.addProperty("KeyLong", Long.toString(key.getKeyID()));
        	pubKey.addProperty("Fingerprint", getFingerPrintString(key.getFingerprint()));
        	pubKey.addProperty("Created", key.getCreationTime().toString());
        	pubKey.addProperty("Master", key.isMasterKey());
        	pubKey.addProperty("Encryption", key.isEncryptionKey());
        	pubKey.addProperty("Revoked", key.isRevoked());
        	boolean expired = false;
        	long validSeconds = key.getValidSeconds();
        	if (validSeconds != 0) {  // 0 expiration time means doesn't expire
        	    Date now = new Date();
        	    long timeLeft = key.getCreationTime().getTime() + (validSeconds * 1000) - now.getTime();
        	    if (timeLeft < 0) {
        	        expired = true;
        	    } else {
        	        pubKey.addProperty("Expires", timeLeft / 86400000);  // Number of days left
        	    }
        	} else {
        	    pubKey.addProperty("Expires", "never");
        	}
        	pubKey.addProperty("Expired", expired);
            Iterator it = key.getUserIDs();
            int i = 0;
            JsonArray ids = new JsonArray();
        	while (it.hasNext()) {
        		ids.add(new JsonPrimitive((String) it.next()));
        	}
        	pubKey.add("ids", ids);
        	Iterator<PGPUserAttributeSubpacketVector> im = key.getUserAttributes();
        	int imagecount = 0;
        	JsonArray images = new JsonArray();
        	while (im.hasNext()) {
        	    PGPUserAttributeSubpacketVector vector = im.next();
        	    ImageAttribute image = vector.getImageAttribute();
        	    if (image != null) {
        	        JsonObject imageJson = new JsonObject();
        	        imageJson.add("type", new JsonPrimitive(image.getType()));
        	        if (image.getType() == 1) {
        	            imageJson.addProperty("data", "data:image/jpg;base64," + Base64.encodeBase64String(image.getImageData()));
        	        }
        	        imageJson.add("index", new JsonPrimitive(imagecount));
        	        imageJson.addProperty("keyid", Long.toString(key.getKeyID()));
        	        imagecount++;
        	        images.add(imageJson);
        	    }
        	    
        	}
        	pubKey.add("images", images);
        	it = key.getSignatures();
        	i = 0;
        	while (it.hasNext()) {
        		PGPSignature sig = (PGPSignature) it.next();
        		String sigid = toHex(sig.getKeyID());
        	//	if (!sigid.equals(self)) {
        			pubKey.addProperty(i++ + "-" + PgpKeys.getSignatureType(sig.getSignatureType()) + "- Signed by", sigid);
        	//	}
        	}
        	keyRingData.add(pubKey);
        }
        return (keyRingData);
    	
    }
    
    public void getKeyImage (HttpServletRequest request, HttpServletResponse response, OxCookie ck) throws Exception {
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        int imagenumber = Core.getIntParameter(request, "image", true);
        String keyString = Core.getStringParameter(request, "keyid", true);
        long keyid = 0l;
        try {
            keyid = Long.parseLong(keyString);
        } catch (Exception ex) {
            Core.sendFail(response, "Bad keyid");
            return;
        }
        GuardKeys key = PGPPublicHandler.getKeyForPGP(userid, cid, keyid);
        if (key != null) {
            Iterator<PGPPublicKey> keys = key.pubring.getPublicKeys();
            while (keys.hasNext()) {
                PGPPublicKey pubkey = keys.next();
                Iterator<PGPUserAttributeSubpacketVector> vectors = pubkey.getUserAttributes();
                if (vectors.hasNext()) {
                    int count = 0;
                    while (vectors.hasNext()) {
                        if (count == imagenumber) {
                            PGPUserAttributeSubpacketVector vector = vectors.next();
                            ImageAttribute image = vector.getImageAttribute();
                            if (image.getType() == 1) {
                                Core.sendOK(response, "image/jpg", image.getImageData());
                            } else {
                                Core.sendOK(response, "application", image.getImageData());
                            }
                            return;
                        } else count++;
                    }
                }
            }
        }
        
    }
   
    
 /**
  * Convert byte array fingerprint into standard String format with spacing
  * @param fingerprint
  * @return
  */
    public static String getFingerPrintString (byte[] fingerprint) {
            StringBuffer fpstring = new StringBuffer();
            for (int i = 0; i < fingerprint.length; ++i) {
            String hex = Integer.toHexString((char)fingerprint[i]);
            hex = hex.toUpperCase();
            while (hex.length() < 2) {
                hex = '0' + hex;
            }
            if (hex.length() > 2) {
                hex = hex.substring(hex.length() - 2);
            }
            fpstring.append(hex);
            if (i % 2 == 1) fpstring.append(" ");
            }
            return fpstring.toString();
    }
    
    /**
     * Convert PGP long key ID into standard HEX
     * @param num
     * @return
     */
    private static String toHex(Long num) {
    	return Long.toHexString(num).substring(8);
    }
    
    /**
     * Get full hex ID of PGP long id
     * @param num
     * @return
     */
    private static String toFullHex (Long num) {
        return Long.toHexString(num);
    }


}
