/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.security.SignatureException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.bcpg.BCPGOutputStream;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPCompressedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.bouncycastle.openpgp.PGPEncryptedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedDataList;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPLiteralData;
import org.bouncycastle.openpgp.PGPLiteralDataGenerator;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPOnePassSignature;
import org.bouncycastle.openpgp.PGPOnePassSignatureList;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyEncryptedData;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureGenerator;
import org.bouncycastle.openpgp.PGPSignatureList;
import org.bouncycastle.openpgp.PGPSignatureSubpacketGenerator;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.bc.BcPGPDataEncryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPublicKeyKeyEncryptionMethodGenerator;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentVerifierBuilderProvider;
import org.bouncycastle.openpgp.operator.jcajce.JcePublicKeyDataDecryptorFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.BadPasswordException;
import com.openexchange.guard.exceptions.NoMasterKeysException;
import com.openexchange.guard.util.Core;



public class PGPUtils {
	private static Logger logger = LoggerFactory.getLogger(PGPUtils.class);

    private static final int BUFFER_SIZE = 1 << 16;
    private static final int bufftest = 256;



    private static final int[] MASTER_KEY_CERTIFICATION_TYPES = new int[] {
        PGPSignature.POSITIVE_CERTIFICATION, PGPSignature.CASUAL_CERTIFICATION, PGPSignature.NO_CERTIFICATION,
        PGPSignature.DEFAULT_CERTIFICATION };

    /**
     * Decode PGP item and return decoded in PGPResult
     * @param in
     * @param userid
     * @param cid
     * @param password
     * @return PGP Result with decoded in result.decoded
     * @throws Exception
     */
    public static PGPResult decryptFile (InputStream in, int userid, int cid, String password) throws Exception {
    	ByteArrayOutputStream fOut = new ByteArrayOutputStream();
    	PGPResult result = decryptFile (in, fOut, userid, cid, password, true);
    	result.decoded = fOut.toByteArray();
    	return (result);
    }

    /**
     * Decrypt PGP Content
     * @param in The input stream to decrypt
     * @param fOutT he output stream writing the decrypted data to
     * @param userid The user's id
     * @param cid
     * @param password The user's password
     * @param close Output stream should be closed and flushed
     * @return PGPResult The result
     * @throws Exception On an error during encoding
     */
    public static PGPResult decryptFile(InputStream in,
                                        OutputStream fOut,
                                        int userid,
                                        int cid,
                                        String password,
                                        boolean close) throws Exception {

        PGPResult result = new PGPResult();
        in = PGPUtil.getDecoderStream(in);
        PGPObjectFactory pgpF = new PGPObjectFactory(in, new BcKeyFingerprintCalculator());
        PGPEncryptedDataList enc;

        //reading first part of the stream
        Object o = pgpF.nextObject();
        if (o == null) {
            result.error = "No Encrypted items";
            return (result);
        }

        //the first object might be a PGP marker packet.
        if (o instanceof PGPEncryptedDataList) {
            enc = (PGPEncryptedDataList) o;
        } else {
            enc = (PGPEncryptedDataList) pgpF.nextObject();
        }

        // find the secret key
        Iterator<?> it = enc.getEncryptedDataObjects();
        PGPPrivateKey sKey = null;
        PGPPublicKeyEncryptedData pbe = null;
        Access acc = new Access();
        StringBuilder missing = new StringBuilder();
        while (sKey == null && it.hasNext()) {
            pbe = (PGPPublicKeyEncryptedData) it.next();
            GuardKeys keys = PGPPublicHandler.getKeyForPGP(userid, cid, pbe.getKeyID(), true);
            if (keys != null) {
                //For upgrade from prior system.  Eventually can remove
                if (keys.version < 0) {  // Look if we need upgraded UUID
                    try {
                        PgpKeys.checkBadUID(keys, password);
                    } catch (PGPException e) {
                        logger.info("Bad password");
                        throw new BadPasswordException("Bad PGP password");
                    }
                }
                PGPSecretKeyRing PGPSecretKR = keys.getPGPSecretKeyRing();
                if (PGPSecretKR == null) {
                    result.error = "No Private Key associated with the Public Key";
                    return (result);
                }
                PGPSecretKey protectedSKey = PGPSecretKR.getSecretKey(pbe.getKeyID());
                if (protectedSKey == null) {
                    result.error = "No Private Key associated with the Public Key";
                    return (result);
                }
                try {
                    sKey = PgpKeys.decodePrivate(protectedSKey, password, keys.getSalt());
                } catch (org.bouncycastle.openpgp.PGPException ex) {
                    logger.info("Bad password");
                    throw new BadPasswordException("Bad PGP password");
                }
            } else {
                missing.append(Core.PGPKeyId(pbe.getKeyID()) + " ");
            }
        }
        if (sKey == null) {
            result.error = "No secret key : " + missing.toString();
            logger.debug("No secret key");
            return (result);
        }

        InputStream clear =
            pbe.getDataStream(
                new JcePublicKeyDataDecryptorFactoryBuilder().setProvider("BC").build(sKey));
        PGPObjectFactory plainFact = new PGPObjectFactory(clear, new BcKeyFingerprintCalculator());
        PGPOnePassSignatureList onePassSignatureList = null;
        PGPOnePassSignature ops = null;
        PGPPublicKey publicKey = null;

        //Parsing messages
        Object message = plainFact.nextObject();
        boolean signatureInitialized = false;
        while (message != null) {
            if (message instanceof PGPCompressedData) {
                PGPCompressedData compressedData = (PGPCompressedData) message;
                plainFact = new PGPObjectFactory(compressedData.getDataStream(), new BcKeyFingerprintCalculator());
                message = plainFact.nextObject();
            }

            if (message instanceof PGPLiteralData) {
                //processing the message
                PGPLiteralData pgpLiteralData = (PGPLiteralData)message;
                InputStream literalInputStream = pgpLiteralData.getInputStream();
                byte[] buffer = new byte[bufftest];
                int len = 0;
                while ((len = literalInputStream.read(buffer)) > -1 ) {
                    if(result.signature && signatureInitialized){
                        //If we have a signature we are going to update the OPS for verifying the signature
                        ops.update(buffer, 0, len);
                    };
                    //Writing the decrypted message to the output stream
                    fOut.write(buffer, 0, len);
                }
                message = plainFact.nextObject();
            }
            else if (message instanceof PGPOnePassSignatureList) {
                //Initializing the signature
                onePassSignatureList = (PGPOnePassSignatureList) message;
                ops = onePassSignatureList.get(0);
                PGPPublicKeyRing pgpRing = null;
                pgpRing = acc.getPublicKeyRingById(ops.getKeyID(), userid, cid);
                result.signature = true;
                if(pgpRing != null) {
                    publicKey = pgpRing.getPublicKey(ops.getKeyID());
                    ops.init(new JcaPGPContentVerifierBuilderProvider().setProvider("BC"), publicKey);
                    signatureInitialized = true;
                }
                else {
                    result.missingPublicKey = true;
                    result.verified = false;
                }
                message = plainFact.nextObject();

            }
            else if (message instanceof PGPSignatureList) {
                //Verify signatures
                PGPSignatureList signatureList = (PGPSignatureList) message;
                if(result.signature && signatureInitialized){
                    for (int i = 0; i < onePassSignatureList.size(); i++) {
                        PGPSignature signature = signatureList.get(i);
                        try {
                            if (ops.verify(signature)) {
                                Iterator<?> userIds = publicKey.getUserIDs();
                                while (userIds.hasNext()) {
                                    String userId = (String) userIds.next();
                                    logger.debug("Signed by " + userId);
                                }
                                result.verified = true;
                                logger.info("Signature verified");
                            }
                        } catch (Exception e) {
                            result.verified = false;
                            logger.info("Unable to verify signature");
                        }
                    }
                }
                message = plainFact.nextObject();
            }
            else {
                result.error = "Unk type";
                break;
            }
        }
        return result;
    }


    /**
     * Verify signature in inline signature
     * @param in
     * @param data
     * @param userid
     * @param cid
     * @return
     * @throws IOException
     * @throws SignatureException
     * @throws PGPException
     */
    public PGPResult verifyInlineSignature (InputStream in, String data, int userid, int cid) throws IOException, SignatureException, PGPException {

    	        com.openexchange.guard.pgp.PGPResult result = new com.openexchange.guard.pgp.PGPResult();
    	        in = PGPUtil.getDecoderStream(in);

    	        PGPObjectFactory pgpF = new PGPObjectFactory(in, new BcKeyFingerprintCalculator());
    	        PGPEncryptedDataList enc;

    	        Object o = pgpF.nextObject();

    	        if (o == null) {
    	        	result.error = "No Encrypted items";
    	        	return (result);
    	        }
    	        //
    	        // the first object might be a PGP marker packet.
    	        //
    	        if (o instanceof PGPSignatureList) {
    	        	result.signature = true;
    	        	PGPSignature signature = ((PGPSignatureList) o).get(0);
    	        	Access acc = new Access();

    	        	PGPPublicKeyRing pgpRing = acc.getPublicKeyRingById(signature.getKeyID(), userid, cid);
    	        	if (pgpRing == null) {
    	        	    result.missingPublicKey = true;
    	        		result.error = "No public key";
    	        		return (result);
    	        	}
    	        	PGPPublicKey publicKey = pgpRing.getPublicKey(signature.getKeyID());
    	        	signature.init(new JcaPGPContentVerifierBuilderProvider().setProvider("BC"), publicKey);
    	        	signature.update(data.getBytes("UTF-8"));
    	        	if (signature.verify()) {
    	        		result.verified = true;
    	        	} else {
                        result.verified = false;
                    }

    	        } else {
    	        	result.signature = false;
    	            // no signature
    	        }
    	        return (result);
    }

    /**
     * Verify seperate signature
     * @param data
     * @param signature
     * @param userid
     * @param cid
     * @return
     * @throws Exception
     */
    public PGPResult verifySignature (String data, String signature, int userid, int cid) throws Exception {
    	PGPResult result = new PGPResult();
    	InputStream datastream = new ByteArrayInputStream(signature.getBytes(StandardCharsets.UTF_8));
    	InputStream in = PGPUtil.getDecoderStream(datastream);
    	PGPObjectFactory factory = new PGPObjectFactory(in, new BcKeyFingerprintCalculator());
    	PGPSignatureList list = null;
    	Object o = factory.nextObject();
    	if (o == null) {
    		logger.error("No signature found");
    		result.error = "No signature found";
    		result.signature = false;
    		return (result);
    	}
    	if (o instanceof PGPCompressedData) {
    		PGPCompressedData c1 = (PGPCompressedData) o;
    		factory = new PGPObjectFactory(c1.getDataStream(), new BcKeyFingerprintCalculator());
    		list = (PGPSignatureList)factory.nextObject();
    	} else {
    		list = (PGPSignatureList) o;
    	}

    	InputStream sigfile = new ByteArrayInputStream(data.getBytes(StandardCharsets.UTF_8));
    	int ch;
    	if (list.size() > 0) {
    		result.signature = true;
    	} else {
    		result.error = "No signatures";
    		return (result);
    	}
    	PGPSignature sig = list.get(0);
    	logger.debug("Verifying signature for key " + sig.getKeyID());
    	Access acc = new Access();
    	PGPPublicKeyRing pgpRing = acc.getPublicKeyRingById(sig.getKeyID(), userid, cid);
    //	GuardKeys gkey = PGPPublicHandler.getKeyForPGP(userid, cid, sig.getKeyID());
    	if (pgpRing == null) {
    		logger.info("Unable to find pgpRing for sender");
    		result.error = "No public key";
    		result.verified = false;
    		result.missingPublicKey = true;
    		return(result);
    	}
    //	PGPPublicKeyRing pgpRing = gkey.pubring;

    	PGPPublicKey key = pgpRing.getPublicKey(sig.getKeyID());
    	if (key == null) {
    		logger.info("Unable to find pgp public key for sender");
    		result.error = "No public key";
    		result.verified = false;
    		result.missingPublicKey = true;
    		return(result);
    	}
    	sig.init(new JcaPGPContentVerifierBuilderProvider(), key);
    	while ((ch = sigfile.read()) >= 0) {
    		sig.update((byte) ch);
    	}
    	sigfile.close();
    	in.close();
    	datastream.close();
    	if (sig.verify()) {
    		result.verified = true;
    		return(result);
    	} else {
    		result.verified = true;
    		return(result);
    	}


    }

    /**
     * Encrypt byte[] data
     * @param out
     * @param data
     * @param fileName
     * @param recipients
     * @param armor
     * @param withIntegrityCheck
     * @throws IOException
     * @throws NoSuchProviderException
     * @throws PGPException
     */
    public static void encrypt(OutputStream out, byte[] data, String fileName, ArrayList<RecipKey> recipients, boolean armor, boolean withIntegrityCheck) throws IOException, NoSuchProviderException, PGPException {
    	ByteArrayInputStream in = new ByteArrayInputStream (data);
    	encrypt (out, in, fileName, recipients, armor, withIntegrityCheck);
    }

    /**
     * Encrypt a PGP Item
     * @param out
     * @param in
     * @param fileName
     * @param recipients  Arraylist of recipients
     * @param armor  Output is armor stream
     * @param withIntegrityCheck
     * @throws IOException
     * @throws NoSuchProviderException
     * @throws PGPException
     */
    public static void encrypt(OutputStream out, InputStream in, String fileName, ArrayList<RecipKey> recipients, boolean armor, boolean withIntegrityCheck) throws IOException, NoSuchProviderException, PGPException {
	    {
	        if (armor)
	        {
	            out = new ArmoredOutputStream(out);
	        }

	        BcPGPDataEncryptorBuilder builder = new BcPGPDataEncryptorBuilder(PGPEncryptedData.AES_256);
	        builder.setSecureRandom(new SecureRandom());
	        PGPEncryptedDataGenerator encryptedDataGenerator = new PGPEncryptedDataGenerator(builder);
	        for (int i = 0; i < recipients.size(); i++) {
	        	BcPublicKeyKeyEncryptionMethodGenerator encKeyGen = new BcPublicKeyKeyEncryptionMethodGenerator(getCryptKey(recipients.get(i).pgpring));
	        	encryptedDataGenerator.addMethod(encKeyGen);
	        }

	        OutputStream encryptedOut = encryptedDataGenerator.open(out, new byte[4028]);

	        PGPCompressedDataGenerator  cGen = new PGPCompressedDataGenerator(
	                                                                PGPCompressedData.ZLIB);

	        BCPGOutputStream            bOut = new BCPGOutputStream(cGen.open(encryptedOut));

	        PGPLiteralDataGenerator     lGen = new PGPLiteralDataGenerator();
	        OutputStream                lOut = lGen.open(bOut, PGPLiteralData.BINARY, "encrypted.asc", new Date(), new byte[256]);
	        int                         ch;

	        byte[] buffer = new byte[bufftest];
	        int len = 0;
	        while ((len = in.read(buffer)) > -1 ) {
	            lOut.write(buffer, 0, len);
	        }

	        lGen.close();
	        cGen.close();
	        encryptedOut.close();
	        if (armor)
	        {
	            out.close();
	        }
	    }
    }


    /**
     * Return encryption key.  Will return master if no other found
     * @param ring
     * @return
     */
    private static PGPPublicKey getCryptKey (PGPPublicKeyRing ring) {
    	PGPPublicKey found = null;
    	Iterator  it = ring.getPublicKeys();
    	while (it.hasNext()) {
    		PGPPublicKey key = (PGPPublicKey) it.next();
    		if (key.isEncryptionKey()) {
    			if (key.isMasterKey()) {  // If master key, we will use only if we don't have another encryption key
    				if (found == null) {
                        found = key;
                    }
    			} else {
    				if (!key.isRevoked()) {
                        return (key);
                    }
    			}
    		}
    	}
    	return (found);
    }


    public static void signEncryptFile(OutputStream out, byte[] data, String fileName, ArrayList<RecipKey> recipients, GuardKeys keys, String password, boolean armor, boolean withIntegrityCheck) throws Exception {
    	ByteArrayInputStream in = new ByteArrayInputStream(data);
    	signEncryptFile(out, in, fileName, recipients,keys,password, armor, withIntegrityCheck);
    }

    /**
     * Encrypt and sign PGP item with onepass signature
     * @param out
     * @param dataStream
     * @param fileName
     * @param recipients
     * @param keys
     * @param password
     * @param armor
     * @param withIntegrityCheck
     * @throws Exception
     */
    public static void signEncryptFile(OutputStream out, InputStream dataStream, String fileName, ArrayList<RecipKey> recipients, GuardKeys keys, String password, boolean armor, boolean withIntegrityCheck) throws Exception {


	    {
	        if (armor)
	        {
	            out = new ArmoredOutputStream(out);
	        }


	        PGPSecretKey pgpSec = keys.getPGPSecretKey();
	        PGPPrivateKey pgpPrivKey = null;
	        try {
	            pgpPrivKey = PgpKeys.decodePrivate(pgpSec, password, keys.getSalt());
	        } catch (PGPException e) {
	            throw (new BadPasswordException("Bad password"));
	        }
	        if (pgpPrivKey == null) {
	            throw (new NoMasterKeysException("Unable to decode pgp private key"));
	        }

	        PGPSignatureGenerator       sGen = new PGPSignatureGenerator(new JcaPGPContentSignerBuilder(pgpSec.getPublicKey().getAlgorithm(), PGPUtil.SHA1).setProvider("BC"));

	        sGen.init(PGPSignature.BINARY_DOCUMENT, pgpPrivKey);

	        Iterator    it = pgpSec.getPublicKey().getUserIDs();
	        if (it.hasNext())
	        {
	            PGPSignatureSubpacketGenerator  spGen = new PGPSignatureSubpacketGenerator();

	            spGen.setSignerUserID(false, (String)it.next());
	            sGen.setHashedSubpackets(spGen.generate());
	        }

	        BcPGPDataEncryptorBuilder builder = new BcPGPDataEncryptorBuilder(PGPEncryptedData.AES_256);
	        builder.setSecureRandom(new SecureRandom());
	        PGPEncryptedDataGenerator encryptedDataGenerator = new PGPEncryptedDataGenerator(builder);
	        for (int i = 0; i < recipients.size(); i++) {
	        	BcPublicKeyKeyEncryptionMethodGenerator encKeyGen = new BcPublicKeyKeyEncryptionMethodGenerator(getCryptKey(recipients.get(i).pgpring));
	        	encryptedDataGenerator.addMethod(encKeyGen);
	        }
	        	OutputStream encryptedOut = encryptedDataGenerator.open(out, new byte[4028]);

	        PGPCompressedDataGenerator  cGen = new PGPCompressedDataGenerator(
	                                                                PGPCompressedData.ZLIB);

	        BCPGOutputStream            bOut = new BCPGOutputStream(cGen.open(encryptedOut));

	        sGen.generateOnePassVersion(false).encode(bOut);

	       // File                        file = new File(fileName);
	        PGPLiteralDataGenerator     lGen = new PGPLiteralDataGenerator();
	        OutputStream                lOut = lGen.open(bOut, PGPLiteralData.BINARY, "encrypted.asc", new Date(), new byte[256]);
	 //      FileInputStream             fIn = new FileInputStream(file);
	        int                         ch;

	        byte[] buffer = new byte[bufftest];
	        int len = 0;
	        while ((len = dataStream.read(buffer)) > -1) {
	        	lOut.write(buffer, 0, len);
	        	sGen.update(buffer, 0, len);
	        }

	        lGen.close();

	        sGen.generate().encode(bOut);

	        cGen.close();
	        encryptedOut.close();
	        if (armor)
	        {
	            out.close();
	        }
	    }
    }

    /**
     * Create a detached signature of inputstream
     * @param in
     * @param out
     * @param key
     * @param password
     * @param armor
     * @throws IOException
     * @throws SignatureException
     * @throws PGPException
     */
    public void createDetachedSignature (InputStream in, OutputStream out, GuardKeys key, String password, boolean armor) throws IOException, SignatureException, PGPException {

        if (armor) {
            out = new ArmoredOutputStream(out);
        }

        PGPSignatureGenerator       sgen = new PGPSignatureGenerator(new JcaPGPContentSignerBuilder(key.getPGPPublicKey().getAlgorithm(), PGPUtil.SHA256).setProvider("BC"));
        sgen.init(PGPSignature.BINARY_DOCUMENT, key.getPGPSignKey(password));

        BCPGOutputStream bout = new BCPGOutputStream (out);
        byte[] buffer = new byte[bufftest];
        int len = 0;
        while ((len = in.read(buffer)) > -1) {
            sgen.update(buffer, 0, len);
        }
        sgen.generate().encode(bout);
        out.close();

    }


}
