/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.SignatureException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import javax.mail.internet.InternetAddress;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.IOUtils;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.encr.Crypto;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.GuardMissingParameter;
import com.openexchange.guard.logging.LogAction;
import com.openexchange.guard.mailcreator.Attachment;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.server.OxCookie;
import com.openexchange.guard.util.Core;
import com.openexchange.guard.validator.EmailValidator;

public class PGPPublicHandler {
	private static Logger logger = LoggerFactory.getLogger(PGPPublicHandler.class);

	public void incomingPGPForm(HttpServletRequest request, HttpServletResponse response) throws Exception {
        FileItemFactory fif = new DiskFileItemFactory();
        ServletFileUpload sfu = new ServletFileUpload(fif);
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        List files = sfu.parseRequest(request);
        String sessionId = Core.getStringParameter(request, "session", true);
        String ua = request.getHeader("User-Agent");
        String emailRestriction = Core.getStringParameter(request, "emailaddress");  // Optional uid restriction
        Iterator iterator = files.iterator(); // Iterate through attachments
        StringBuilder responseSB = new StringBuilder();
        responseSB.append("[");  // Start response array
        int count = 0;
        while (iterator.hasNext()) {
            FileItem fi = (FileItem) iterator.next();
            if (fi.getSize() > 1000000 ) {  // Key sizes over a mb are not allowed
                Core.sendFail(response, "Key size exceeded");
                return;
            }
            String name = fi.getFieldName();
            name = fi.getName();
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            InputStream input = fi.getInputStream();
            IOUtils.copy(input, out); // Copy stream to byte array
            out.close();
            input.close();
            String data = "\r\n" + new String(out.toByteArray());
            int start = data.indexOf("-----BEGIN PGP PUBLIC");
            if (start < 0) {
                LogAction.setFail();
                logger.info("Bad data");
            	Core.sendFail(response, "Bad data");
            	return;
            }
            data = data.substring(start);
            try {
            	PGPPublicKeyRing ring = getKeyRing(data);
            	if (ring == null) {
            	    LogAction.setFail();
            	    logger.info("Bad pgp data");
            		Core.sendFail(response, "Bad pgp data");
            		return;
            	}
            	if (emailRestriction != null) {
            	    String [] emails = emailRestriction.split(",");
            	    Iterator<String> uids = ring.getPublicKey().getUserIDs();
            	    boolean found = false;
            	    while (uids.hasNext()) {
            	        String uid = uids.next().toLowerCase();
            	        for (String em : emails) {
            	            if (em.contains("@")) { // If email address, check if in userid
            	                if (uid.contains(em.toLowerCase())) found = true;
            	            }
            	        }     
            	    }
            	    if (!found) {
            	        logger.debug("User email address not found in the uploaded public key" );
            	        Core.sendFail(response, "Bad UID");
            	        return;
            	    }
            	}
            	if (!saveKeyData(ring, cid, userid, data)) {
            	    LogAction.setFail();
            	    logger.warn("Unable to save uploaded public key data");
            	    Core.sendFail(response, "Problem importing key, possible bad data");
            	    return;
            	} else {
            	    try {
            	        JsonArray keydata = PgpHandler.getKeyInfo(ring);
            	        if (count > 0) responseSB.append(",");  // If more than one response, add comma seperator
            	        responseSB.append(keydata.toString());  // Append json to the response array
            	        count++;
            	     //   Core.sendOK(response, keydata.toString());
            	    } catch (Exception e) {
            	        logger.error("Problem getting uploaded key data ", e);
            	    }
            	}
            	
            } catch (Exception e) {
                LogAction.setFail();
            	logger.error("Problem importing PGP key ", e);
            	Core.sendFail(response, "Problem imporing key " + e.getMessage());
            	return;
            }
        }
        responseSB.append("]");  // Close the array
        Core.sendOK(response, responseSB.toString());
        LogAction.setSuccess();
	}
	
	public static PGPPublicKeyRing getKeyRing (String AscData) throws IOException {
        // PGPUtil.getDecoderStream() will detect ASCII-armor automatically and decode it,
        // the PGPObject factory then knows how to read all the data in the encoded stream
        PGPObjectFactory factory = new PGPObjectFactory(PGPUtil.getDecoderStream(new java.io.ByteArrayInputStream(AscData.getBytes("UTF-8"))), new BcKeyFingerprintCalculator());
        // these files should really just have one object in them,
        // and that object should be a PGPPublicKeyRing.
        Object o = factory.nextObject();
        if (o instanceof PGPPublicKeyRing) {
            return (PGPPublicKeyRing) o;
        }
        throw new IllegalArgumentException("Input text does not contain a PGP Public Key");
	}
	
	public boolean saveKeyData (PGPPublicKeyRing ring, int cid, int userid, String ascData) throws Exception {
		if (!ring.getPublicKeys().hasNext()) return (false);  // If no public keys, return false
		JsonArray keydata = PgpHandler.getKeyInfo(ring);
		JsonArray ids = new JsonArray();
		String keyhexids = "";
		for (int i = 0; i < keydata.size(); i++) {
			JsonObject key = keydata.get(i).getAsJsonObject();
			JsonArray keyids = key.get("ids").getAsJsonArray();
			keyhexids = keyhexids + key.get("Key").getAsString() + " ";
			if (keyids.size() > 0) ids = keyids;
		}
		if (keyhexids.trim().length() < 2) {
			return(false);
		}
		boolean added = false;
		for (int j = 0; j < ids.size(); j++) {
			InternetAddress addr = Core.parseEmailAdress(ids.get(j).getAsString());
			try {
				addr.validate();
			} catch (Exception e) {
				continue;
			}
			String email = addr.getAddress();
			// Update key.  On duplicate, overwrite
			String command = "INSERT INTO og_pgp_keys (ids, Email, PGPPublic, userid, share_level, cid) VALUES (?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE PGPPublic = ?, ids = ?";
			DbCommand com = new DbCommand (command);
			com.addVariables(keyhexids);
			com.addVariables(email);
			com.addVariables(ascData);
			com.addVariables(userid);
			com.addVariables(0);
			com.addVariables(cid);
			com.addVariables(ascData);
			com.addVariables(keyhexids);
			DbQuery db = new DbQuery();
			db.writeToDB(com, cid);
			db.close();
			added = true;
		}
		return (added);
	}
	
	public void saveKeyAttachment(HttpServletRequest request, HttpServletResponse response, OxCookie ck) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		LogAction.setUser(userid, cid);
		String folder = Core.getStringParameter(request, "folder", true);
        String  emailid = Core.getStringParameter(request, "emailid", true);
        String attach = Core.getStringParameter(request, "attach", true);
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            LogAction.setFail();
            logger.info("User not logged into UI");
        	Core.sendFail(response, "Must be logged in to UI");
        	return;
        }
        PGPPublicKeyRing ring = null;
        String epass = Core.getStringParameter(request, "epass");
        String data = null;
        if (epass == null) {  // If not encrypted, retrieve plaintext
            Attachment att = ap.getPlainAttachment(emailid, attach, folder);
            data = new String (att.content);
            
        } else {
            // Handle encrypted public key
            String filename = Core.getStringParameter(request, "filename", true);
            boolean inline = Core.getBooleanParameter(request, "inline");
            String password = Crypto.decodeEPass(epass, ck);
            if (password == null) {
                Core.sendFail(response, "bad password");
                return;
            }
            PgpHandler pgh = new PgpHandler();
            byte[] bytedata = null;
            if (inline) {  // pgp inline attachment
                bytedata = pgh.decodeInlineAttach(emailid, folder, attach, filename, userid, cid, password, request, ck);                
            } else {  // pgp mime attachment
                bytedata = pgh.getAttach(request, ck, emailid, folder, userid, cid, filename, password);
                
            }
            if (bytedata == null || bytedata.length < 2) {
                logger.warn("Unable to decode inline pgp public key attachment");
                Core.sendFail(response, "unable to decode");
                return;
            }
            data = new String(bytedata, "UTF-8");
        }
        ring = getKeyRing(data);
        if (ring == null) {
            LogAction.setFail();
            logger.info("Problem reading key data");
        	Core.sendFail(response, "Problem reading key data");
        	return;
        }
		if (saveKeyData (ring, cid, userid, data)) {
			LogAction.setSuccess();
			Core.sendOK(response, PgpHandler.getKeyInfo(ring).toString());
		} else {
		    LogAction.setFail();
		    logger.info("Problem saving key data");
			Core.sendFail(response, "Problem saving");
		}
		
	}
	
	
	public void getPubKeyDetail (HttpServletRequest request, HttpServletResponse response) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		LogAction.setUser(userid, cid);
		String ids = Core.getStringParameter(request, "ids", true).trim();
		boolean guard = Core.getBooleanParameter(request, "guard");
		if (guard) {  // If we are to check the Guard tables, do it here
		    JsonObject guardJson = getGuardPubKeyDetail(ids);
		    if (guardJson != null) {
		        guardJson.addProperty("guard", true);
		        Core.sendOK(response, guardJson.toString());
                return;
		    }
		}
		// Otherwise check uploaded
		String command = "SELECT PGPPublic, share_level, inline FROM og_pgp_keys WHERE cid = ? and (userid = ? OR share_level > 0) AND ids LIKE ?";
		DbCommand com = new DbCommand (command);
		com.addVariables(cid);
		com.addVariables(userid);
		com.addVariables(ids + "%");
		DbQuery db = new DbQuery();
		db.readFromDB(com, cid);
		if (db.next()) {
			String pubkey = db.rs.getString("PGPPublic");
			PGPPublicKeyRing keyring = getKeyRing(pubkey);
			JsonArray keydata = PgpHandler.getKeyInfo(keyring);
			JsonObject retJson = new JsonObject();
			retJson.add("data", keydata);
			retJson.add("share", new JsonPrimitive(db.rs.getInt("share_level")));
			retJson.add("inline", new JsonPrimitive (db.rs.getBit("inline")));
			if (keydata.size() > 0) {
				db.close();
				LogAction.setSuccess();
				Core.sendOK(response, retJson.toString());
				return;
			}
		}
		db.close();
		LogAction.setFail();
		Core.sendFail(response, "Fail");
	}
	
	private JsonObject getGuardPubKeyDetail (String id) throws SignatureException, PGPException {
	    Access acc = new Access();
	    String[] ids = id.split(" ");
	    for (String single_id : ids) {
	        PGPPublicKeyRing keyring = acc.getPublicKeyRingById(single_id, 0, 0);
	        if (keyring != null) {
	            JsonArray keydata = PgpHandler.getKeyInfo(keyring);
	            JsonObject retJson = new JsonObject();
	            retJson.add("data", keydata);
	            if (keydata.size() > 0) {
	                return (retJson);
	            }
	        }
	    }
	    return(null);  
	}
	
	public void getPubKeyList (HttpServletRequest request, HttpServletResponse response) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		LogAction.setUser(userid, cid);
		StringBuilder searchlist = new StringBuilder();
		String emails = Core.getStringParameter(request, "emails");
		ArrayList emailArray = new ArrayList();
		if (emails != null) {
		    String [] emailList = emails.split(",");
		    for (String email : emailList) {
		        if (!email.trim().equals("")) {
		            if (EmailValidator.validate(email)) {
		                if (searchlist.length() == 0) {
		                    searchlist.append("'" + Core.dbformat(email) + "'");
		                } else searchlist.append(", '" + Core.dbformat(email) + "'");
		                emailArray.add(email);
		            }
		        }
		    }
		}
		LogAction.setSuccess();
		Core.sendOK(response, getPubKeyList(userid, cid, searchlist.toString(), emailArray).toString());
	}
	
	public JsonArray getPubKeyList (int userid, int cid, String searchlist, ArrayList emailArray) throws Exception {
		JsonArray retdata = new JsonArray();
		String command = "SELECT ids, Email, share_level, userid, inline, (userid = ?) AS owned FROM og_pgp_keys WHERE cid = ? and (userid = ? OR share_level > 0)"; 
		if (searchlist.length() > 1) {
		    command += " AND Email IN (" + searchlist + ") ";
		}
		command += "  ORDER BY owned DESC, Email ASC";
		DbCommand com = new DbCommand (command);
		com.addVariables(userid);
		com.addVariables(cid);
		com.addVariables(userid);
		DbQuery db = new DbQuery();
		db.readFromDB(com, cid);
		retdata = getKeyDataFromDB(db, userid);
		if (emailArray.size() > 0) {
		    Access acc = new Access();
		    for (int i = 0; i < emailArray.size(); i++) {
		        String email = (String) emailArray.get(i);
		        GuardKeys gkey = acc.getKeysFromEmail(email);
		        if (gkey != null) {
		            JsonArray info = PgpHandler.getKeyInfo(gkey.pubring);
		            StringBuilder idBuilder = new StringBuilder();
		            for (int j = 0; j < info.size(); j++) {
		                JsonObject keyinfo = (JsonObject) info.get(j);
		                if (j>0) idBuilder.append(" ");
		                idBuilder.append(keyinfo.get("Key").getAsString());
		            }
		            JsonObject key = new JsonObject();
		            key.addProperty("ids", idBuilder.toString());
		            key.addProperty("email", email);
		            key.addProperty("share", false);
		            key.addProperty("guard", true);
		            key.addProperty("inline", gkey.inline);
		            retdata.add(key);
		            
		        }
		    }
		}
		return(retdata);
	}
	
	private JsonArray getKeyDataFromDB(DbQuery db, int userid) throws Exception {
	    JsonArray retdata = new JsonArray();
	    while (db.next()) {
            JsonObject key = new JsonObject();
            key.addProperty("ids", db.rs.getString("ids"));
            key.addProperty("email", db.rs.getString("Email"));
            key.addProperty("share", db.rs.getInt("share_level"));
            key.addProperty("shared", (userid == db.rs.getInt("userid") ? false : true));  // mark if this is a shared key
            key.add("inline", new JsonPrimitive(db.rs.getBit("inline")));
            retdata.add(key);
        }
        db.close();
        return(retdata);
	}
	
	public void deleteKey (HttpServletRequest request, HttpServletResponse response) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		LogAction.setUser(userid, cid);
		String ids = Core.getStringParameter(request, "ids", true);
		try {
			String command = "DELETE FROM og_pgp_keys WHERE CID = ? AND userid = ? AND ids = ?";
			DbCommand com = new DbCommand(command);
			com.addVariables(cid);
			com.addVariables(userid);
			com.addVariables(ids.trim());
			DbQuery db = new DbQuery();
			db.write(com, userid, cid);
			db.close();
			LogAction.setSuccess();
			Core.sendOK(response, "OK");
		} catch (Exception ex) {
		    LogAction.setFail();
			logger.error("Problem deleting public pgp key ", ex);
			Core.sendFail(response, "Fail");
		}
	}
	
	public void shareKey (HttpServletRequest request, HttpServletResponse response) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		String ids = Core.getStringParameter(request, "ids", true);
		LogAction.setUser(userid, cid);
		String share = Core.getStringParameter(request, "share", true);
		int sharelevel = share.toLowerCase().trim().equals("true") ? 1 : 0;
		try {
			String command = "UPDATE og_pgp_keys SET share_level = ? WHERE cid = ? AND userid = ? AND ids = ?";
			DbCommand com = new DbCommand(command);
			com.addVariables(sharelevel);
			com.addVariables(cid);
			com.addVariables(userid);
			com.addVariables(ids.trim());
			DbQuery db = new DbQuery();
			db.write(com, userid, cid);
			db.close();
			LogAction.setSuccess();
			Core.sendOK(response, "OK");
		} catch (Exception ex) {
		    LogAction.setFail();
			logger.error("Problem sharing public pgp key ", ex);
			Core.sendFail(response, "Fail");
		}
	}
	
	public void changeInline (HttpServletRequest request, HttpServletResponse response) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		LogAction.setUser(userid, cid);
		String ids = Core.getStringParameter(request, "ids", true);
		String inlineStr = Core.getStringParameter(request, "inline", true);
		int isInline = inlineStr.toLowerCase().trim().equals("true") ? 1 : 0;
		try {
			String command = "UPDATE og_pgp_keys SET inline = ? WHERE cid = ? AND userid = ? AND ids = ?";
			DbCommand com = new DbCommand(command);
			com.addVariables(isInline);
			com.addVariables(cid);
			com.addVariables(userid);
			com.addVariables(ids.trim());
			DbQuery db = new DbQuery();
			db.write(com, userid, cid);
			db.close();
			LogAction.setSuccess();
			Core.sendOK(response, "OK");
		} catch (Exception ex) {
		    LogAction.setFail();
			logger.error("Problem changing inline status public pgp key ", ex);
			Core.sendFail(response, "Fail");
		}
	}
	
	public static GuardKeys getKeyForPGP (int id, int cid, long keyid) throws Exception {
		return (getKeyForPGP (id, cid, keyid, false));
	}
	
	/**
	 * Return the Guard Keys populated with PGP data
	 * @param id
	 * @param cid
	 * @param keyid
	 * @param incoming - true if current private key search only
	 * @return
	 * @throws Exception
	 */
	public static GuardKeys getKeyForPGP (int userid, int cid, long keyid, boolean incoming) throws Exception {
    	String command = "SELECT keyid from PGPKeys WHERE cid = ? and id = ?";
    	DbCommand com = new DbCommand(command);
    	com.addVariables(cid);
    	com.addVariables(keyid);
    	DbQuery db = new DbQuery();
    	db.readOG(com);
    	if (db.next()) {
    		Access acc = new Access();
    		GuardKeys keys = acc.getKeys (userid, cid, db.rs.getLong("keyid"), !incoming);  // If incoming, we want all of the keys, not just the current
    		db.close();
    		return (keys);
    	}
    	db.close();
    	if (incoming) return(null);
    	return (getUploadedPublic (userid, cid, Core.PGPKeyId(keyid)));
    }
	
	private static GuardKeys getUploadedPublic (int userid, int cid, String pubid) throws Exception {
		String command = "SELECT Email, PGPPublic, userid FROM og_pgp_keys WHERE cid = ? AND ids LIKE ? AND (userid = ? OR share_level > 0)";
		DbCommand com = new DbCommand(command);
		com.addVariables(cid);
		com.addVariables("%" + pubid + "%");
		com.addVariables(userid);
		DbQuery db = new DbQuery();
		db.readFromDB(com, cid);
		GuardKeys keys = null;
		while (db.next()) {
		    try {
    			keys = new GuardKeys ();
    			keys.email = db.rs.getString("Email");
    			keys.pubring = getKeyRing(db.rs.getString("PGPPublic"));
    			keys.contextid = cid;
    			keys.userid = userid;
    			int owner = db.rs.getInt("userid");
                db.close();
                if (owner == userid) return(keys);  // If this key is owned by the sender, return preferentially
		    } catch (Exception e) {
                logger.error("Problem with uploaded key ", e);
            }
		}
		db.close();
		return (keys);
	}
	
	public static GuardKeys getPGPKeysFromEmail (String email, int userid, int cid) throws Exception {
	    try {
    		String command = "SELECT Email, PGPPublic, inline, userid FROM og_pgp_keys WHERE cid = ? AND Email LIKE ? AND (userid = ? OR share_level > 0)";
    		DbCommand com = new DbCommand(command);
    		com.addVariables(cid);
    		com.addVariables(email);
    		com.addVariables(userid);
    		DbQuery db = new DbQuery();
    		db.read(com, userid, cid);
    		GuardKeys keys = null;
    		while (db.next()) {
    		    try {
        			keys = new GuardKeys ();
        			keys.email = db.rs.getString("Email");
        			keys.pubring = getKeyRing(db.rs.getString("PGPPublic"));
        			keys.inline = db.rs.getBit("inline");
        			keys.contextid = cid;
        			keys.userid = userid;
        			int owner = db.rs.getInt("userid");
        			db.close();
        			if (owner == userid) return(keys);  // If this key is owned by the sender, return preferentially
    		    } catch (Exception e) {
    		        logger.error("Problem with uploaded key ", e);
    		        
    		    }
    		}
    		db.close();
    		// Return any keys found, null otherwise
    		return (keys);
	    } catch (Exception ex) {
	        logger.error("Error retrieving uploaded PGP keys  ", ex);
	        return(null);
	    }
	}
}
