package com.openexchange.guard.ox.streaming;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import com.openexchange.guard.file.FileTyper;
import com.openexchange.guard.pgp.PGPResult;
import com.openexchange.guard.pgp.PGPUtils;

/**
 * Decrypts a stream of data using PGP
 *
 * @author Benjamin Gruedelbach
 *
 */
public class PGPStreamDecryptionHandler implements StreamHandler {

    private final int userId;
    private final int cid;
    private final String password;

    /**
     * Constructor
     *
     * @param userId The user's id
     * @param cid The cid
     * @param password The password of the user's private key
     */
    public PGPStreamDecryptionHandler(int userId, int cid, String password) {
        this.userId = userId;
        this.cid = cid;
        this.password = password;
    }

    @Override
    public void processStream(InputStream inputStream,
        OutputStream outputStream,
        int suggestedBufferSize) throws IOException {

            PGPResult result;
            try {
                result = PGPUtils.decryptFile(inputStream,
                    outputStream,
                    userId,
                    cid,
                    password,
                    false /* do not close the output stream */);
            } catch (Exception e) {
                throw new IOException(e.getMessage());
            }
            if(result.error != null) {
                throw new IOException(result.error);
            }
            else if(result.signature == true && result.verified == false){
                throw new IOException("Signature verification failed");
            }
    }

    @Override
    public String getNewContentType(String oldContentType, String filename) {
        String newContentType = oldContentType.replace("pgp/", "").replace("--", "/");
        if (newContentType.contains("application/octet-stream") || newContentType.contains("application/pgp-signature")) {
            newContentType = FileTyper.getFileType(filename, newContentType);
        }
        return newContentType;
    }

    @Override
    public String getNewName(String oldName) {
        return oldName.replace(".pgp", "");
    }

}
