
package com.openexchange.guard.ox.streaming;

import java.io.IOException;
import java.io.InputStream;
import org.apache.commons.fileupload.FileItemIterator;
import org.apache.commons.fileupload.FileItemStream;
import org.apache.commons.fileupload.FileUploadException;

/**
 * Wraps an org.apache.commons.fileupload.FileItem into an input stream
 */
public class FileItemInputStream extends InputStream {

    private final FileItemIterator fileItemIterator;
    private final String fieldName;
    private String itemName;
    private String contentType;
    private InputStream fileStream;

    /**
     * Constructor
     * @param fieldName The name of the file item
     * @param fileItemIterator The iterator to a collection of file items
     */
    public FileItemInputStream(String fieldName, FileItemIterator fileItemIterator) {
        this.fieldName = fieldName;
        this.fileItemIterator = fileItemIterator;
    }

    /**
     * Searches for the file item and opens a stream to it
     * @throws IOException
     * @throws FileUploadException
     */
    private void EnsureStreamIsLoaded() throws IOException, FileUploadException {
        if (fileStream == null) {
            while (fileItemIterator.hasNext()) {
                FileItemStream item = fileItemIterator.next();
                if (item.getFieldName().equals(this.fieldName)) {
                    fileStream = item.openStream();
                    this.itemName = item.getName();
                    this.contentType = item.getContentType();
                    break;
                }
            }
        }
    }

    /**
     * @return The item's name
     * @throws Exception
     */
    public String getItemName() throws Exception {
        EnsureStreamIsLoaded();
        return this.itemName;
    }

    /**
     * @return The item's content type
     * @throws Exception
     */
    public String getContentType() throws Exception {
        EnsureStreamIsLoaded();
        return this.contentType;
    }

    @Override
    public int read() throws IOException {
        try {
            EnsureStreamIsLoaded();
            return fileStream.read();
        } catch (FileUploadException e) {
            throw new IOException(e.getMessage());
        }
    }

    @Override
    public int read(byte[] b) throws IOException {
        try {
            EnsureStreamIsLoaded();
            return fileStream.read(b);
        } catch (FileUploadException e) {
            throw new IOException(e.getMessage());
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        try {
            EnsureStreamIsLoaded();
            return fileStream.read(b,off,len);
        } catch (FileUploadException e) {
            throw new IOException(e.getMessage());
        }
    }

}
