/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.file;

import java.io.ByteArrayInputStream;
import java.io.StringWriter;

import javax.servlet.http.HttpServletRequest;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.GetPubKey;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.database.RetrievedItem;
import com.openexchange.guard.encr.Crypto;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.encr.RSAGen;
import com.openexchange.guard.ox.InterOx;
import com.openexchange.guard.server.OxCookie;

public class Fileshare {

	private static Logger logger = LoggerFactory.getLogger(Fileshare.class);
	
    public String shareItem(String itemId, String name, String email, String contentKey, GuardKeys owner) {
        try {
            Access acc = new Access();
            GuardKeys keys = acc.getKeysFromEmail(email);
            boolean remote = false;
            if (keys == null) {
                GetPubKey gk = new GetPubKey();
                RecipKey rkey = gk.getPubKey(email, owner.userid, owner.contextid, owner.email, true, true);
                if (rkey == null)
                    return (null); // not found
                if (rkey.pubkey == null) 
                	return (null); // unable
                if (rkey.cid != owner.contextid) {// For now, limit to same contextid.  Work on remote later
                	return ("nsc");
                }
                if (rkey.remote) {
                    RetrievedItem file = acc.getItemID(itemId, owner.userid, owner.contextid);
                    EncrLib encr = new EncrLib();
                    String encrcontentkey = encr.encryptRSA(contentKey, rkey.pubkey);
                    InterOx iox = new InterOx();
                    iox.sendItem(email, itemId, encrcontentkey, file.expiration, file.Type, file.XML, file.salt);
                    if (addShare(itemId, owner.userid, owner.contextid, name, email, rkey.userid, rkey.cid, true)) {
                        String returndata = "{\"i\":" + rkey.userid + ", \"c\": " + rkey.cid + ", \"r\": \"Y\"}";
                        return (returndata);
                    } else
                        return (null);

                }
                keys = new GuardKeys(rkey);
            }
            if (keys.contextid != owner.contextid) {
            	return ("nsc");
            }
            Crypto crypto = new Crypto();
            if (crypto.saveContentKey(itemId, keys.userid, keys.contextid, keys.getPublic(), contentKey, acc)) { // Save the content Key to
                                                                                                                 // new user
                if (addShare(itemId, owner.userid, owner.contextid, name, email, keys.userid, keys.contextid, false)) { // Store it in our
                                                                                                                        // XML shares
                    String returndata = "{\"i\":" + keys.userid + ", \"c\": " + keys.contextid + ", \"r\": \"" + (remote ? "Y" : "N") + "\"}";
                    return (returndata);
                }
            }
        } catch (Exception ex) {
            logger.error("Error sharing item", ex);
        }
        return (null);
    }

    public String share(String itemID, String name, String email, int userid, int cid, String password) {
        try {
            Access acc = new Access();
            GuardKeys owner = acc.getKeys(userid, cid);
            RetrievedItem file = acc.getItemID(itemID, userid, cid);
            EncrLib encr = new EncrLib();
            String contentKey = encr.decryptRSA(file.contentKey, owner.getDecodedPrivate(password));
            if (contentKey == null) {
                return (null);
                // Fail password
            }
            return (shareItem(itemID, name, email, contentKey, owner));
        } catch (Exception ex) {
            logger.error("Error sharing item " + itemID + "for user " + email, ex);
        }

        return (null);
    }

    public JsonArray getShares(String itemID, int ownerid, int ownercid) {
        try {
            org.w3c.dom.Document doc = getXML(itemID, ownerid, ownercid);
            if (doc == null) {
                JsonArray shares = new JsonArray();
                return (shares);
            }
            doc.getDocumentElement().normalize();
            if (doc.getElementsByTagName("share") == null) {
                doc.createElement("share");
            }
            Element share = (Element) doc.getElementsByTagName("share").item(0);
            NodeList users = share.getChildNodes();
            JsonArray shares = new JsonArray();
            for (int i = 0; i < users.getLength(); i++) {
                Element user = (Element) users.item(i);
                JsonObject u = new JsonObject();
                if (user.getElementsByTagName("n").item(0) != null) {
                    u.addProperty("n", user.getElementsByTagName("n").item(0).getTextContent());
                }
                u.addProperty("e", user.getElementsByTagName("e").item(0).getTextContent());
                u.addProperty("r", user.getElementsByTagName("r").item(0).getTextContent());
                u.addProperty("i", Integer.parseInt(user.getElementsByTagName("i").item(0).getTextContent()));
                u.addProperty("c", Integer.parseInt(user.getElementsByTagName("c").item(0).getTextContent()));
                shares.add(u);
            }
            return (shares);
        } catch (Exception ex) {
            logger.error("Error getting shares for item " + itemID, ex);
            return (null);
        }
    }

    private org.w3c.dom.Document getXML(String itemID, int userid, int ownercid) {
        try {
            String command = "SELECT XML FROM og_encrypted_items WHERE Id = ?";
            DbCommand com = new DbCommand(command);
            com.addVariables(itemID);
            DbQuery db = new DbQuery();
            db.readFromDB(com, ownercid);
            String XML = "";
            if (db.next()) {
                XML = db.rs.getString("XML");
            } else { // Item was not found, return false
                db.close();
                return (null);
            }
            db.close();
            if (XML.equals("")) {
                XML = "<share></share>";
            }
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            org.w3c.dom.Document doc = builder.parse(new InputSource(new ByteArrayInputStream(XML.getBytes("utf-8"))));
            doc.getDocumentElement().normalize();
            return (doc);
        } catch (Exception ex) {
            logger.error("Error getting XML for " + itemID, ex);
            return (null);
        }

    }

    private String getXMLString(org.w3c.dom.Document doc) throws TransformerException {
        StringWriter writer = new StringWriter();
        StreamResult result = new StreamResult(writer);
        TransformerFactory tf = TransformerFactory.newInstance();
        Transformer transformer = tf.newTransformer();
        DOMSource domSource = new DOMSource(doc);
        transformer.transform(domSource, result);
        String XML = writer.toString();
        if (XML.contains("?>")) {
            XML = XML.substring(XML.indexOf("?>") + 2);
        }
        return (XML);
    }

    public boolean addShare(String itemID, int userid, int ownercid, String name, String shareEmail, int shareUserid, int shareCid, boolean remote) {
        try {

            org.w3c.dom.Document doc = getXML(itemID, userid, ownercid);
            if (doc.getElementsByTagName("share") == null) {
                doc.createElement("share");
            }
            Element share = (Element) doc.getElementsByTagName("share").item(0);
            Element user = doc.createElement("u");
            if (name != null) {
                if (name.length() > 1) { // Has content
                    Element xname = doc.createElement("n");
                    xname.setTextContent(name);
                    user.appendChild(xname);
                }
            }
            Element email = doc.createElement("e");
            email.setTextContent(shareEmail);
            user.appendChild(email);
            Element isremote = doc.createElement("r");
            isremote.setTextContent(remote ? "Y" : "N");
            user.appendChild(isremote);
            Element id = doc.createElement("i");
            id.setTextContent(Integer.toString(shareUserid));
            user.appendChild(id);
            Element cid = doc.createElement("c");
            cid.setTextContent(Integer.toString(shareCid));
            user.appendChild(cid);
            share.appendChild(user);

            String XML = getXMLString(doc);
            // Save it

            String command = "UPDATE og_encrypted_items SET XML = ? WHERE Id = ?";
            DbCommand com = new DbCommand(command);
            com.addVariables(XML);
            com.addVariables(itemID);
            DbQuery db = new DbQuery();
            db.writeToDB(com, ownercid);
            db.write(com, userid, ownercid);
            db.close();
            return (true);
        } catch (Exception ex) {
            logger.error("Error adding share for item " + itemID, ex);

        }
        return (false);
    }

    private void deleteXMLShare(String itemID, int userid, int ownercid, String email) {
        try {
            org.w3c.dom.Document doc = getXML(itemID, userid, ownercid);
            Element share = (Element) doc.getElementsByTagName("share").item(0);
            NodeList users = share.getElementsByTagName("u");
            for (int i = 0; i < users.getLength(); i++) {
                Element user = (Element) users.item(i);
                if (user != null) {
                    String uemail = user.getElementsByTagName("e").item(0).getTextContent();
                    if (uemail.toLowerCase().equals(email.toLowerCase())) {
                        user.getParentNode().removeChild(user);
                    }
                }

            }

            String command = "UPDATE og_encrypted_items SET XML = ? WHERE Id = ?";
            DbCommand com = new DbCommand(command);
            com.addVariables(getXMLString(doc));
            com.addVariables(itemID);
            DbQuery db = new DbQuery();
            db.write(com, userid, ownercid);
            db.close();
        } catch (Exception ex) {
            logger.error("Error deleting XML share for " + itemID, ex);
        }
    }

    public boolean removeShare(String itemID, int ownerId, int ownerCid, String email) {
        try {
            JsonArray shares = getShares(itemID, ownerId, ownerCid);
            if (shares == null) {
                return (false);
            }
            for (int i = 0; i < shares.size(); i++) {
                JsonObject user = (JsonObject) shares.get(i);
                if (user.get("e").getAsString().toLowerCase().equals(email.toLowerCase())) {
                    int cid = user.get("c").getAsInt();
                    int userid = user.get("i").getAsInt();
                    boolean remote = user.get("r").getAsString().equals("Y");
                    if (deleteShare(itemID, email, userid, cid, remote, ownerId, ownerCid)) {
                        deleteXMLShare(itemID, ownerId, ownerCid, email);
                        return (true);
                    }
                    return (false);
                }
            }
        } catch (Exception ex) {
            logger.error("Error removing share of item " + itemID, ex);
        }
        return (false);
    }

    private boolean deleteShare(String itemID, String email, int userid, int cid, boolean remote, int ownerId, int ownerCid) {
        try {
            if (remote) {
                Access acc = new Access();
                RetrievedItem file = acc.getItemID(itemID, ownerId, ownerCid);
                InterOx iox = new InterOx();
                return (iox.deleteshare(email, itemID, file.salt));
            } else {
                String command = "DELETE FROM og_content_keys WHERE ItemId = ? AND UserId = ? AND CID = ?";
                DbCommand com = new DbCommand(command);
                com.addVariables(itemID);
                com.addVariables(userid);
                com.addVariables(cid);
                DbQuery db = new DbQuery();
                db.write(com, userid, cid);
                db.close();
                return (true);
            }

        } catch (Exception ex) {
            logger.error("Error deleting share for item " + itemID, ex);

        }
        return (false);
    }
}
