/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.encr;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.security.AlgorithmParameters;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.sql.SQLException;
import java.util.ArrayList;
//import java.sql.Date;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.Cipher;
import javax.crypto.EncryptedPrivateKeyInfo;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.PBEParameterSpec;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.net.util.Base64;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PKCS8Generator;
import org.bouncycastle.openssl.jcajce.JcaPKCS8Generator;
import org.bouncycastle.openssl.jcajce.JceOpenSSLPKCS8EncryptorBuilder;
import org.bouncycastle.operator.OutputEncryptor;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.exceptions.NoContentKeyException;
import com.openexchange.guard.exceptions.NoItemException;
import com.openexchange.guard.exceptions.NoMasterKeysException;
import com.openexchange.guard.ox.InterOx;
import com.openexchange.guard.pgp.PgpKeys;
import com.openexchange.guard.server.CheckBad;
import com.openexchange.guard.server.OxCookie;
import com.openexchange.guard.storage.Storage;

public class Crypto {

	private static Logger logger = LoggerFactory.getLogger(Crypto.class);

    public String EncryptEmail(int owner, int owner_cid, ArrayList<RecipKey> recipients, String data, long exp, int type, String XML, String itemID, String contentKey, String salt) throws Exception {
        long start = System.currentTimeMillis();
        EncrLib encryptor = new EncrLib();
        // String encrypted = encryptor.encryptAES(data, contentKey, salt, Config.AESKeyLen);
        String encrypted = encryptor.encryptAES(data, contentKey);
        // FileStore.File.saveEncrObj(owner, itemID, encrypted.getBytes());
        Access acc = new Access();
        StringBuilder addxml = new StringBuilder();
        addxml.append("<share>");
        for (RecipKey user : recipients) {
            addxml.append("<u><e>" + StringEscapeUtils.escapeXml11(user.email) + "</e><r>" + (user.remote ? "Y" : "N") + "</r><i>" + user.userid + "</i><c>" + user.cid + "</c></u>");
        }
        addxml.append("</share>");
        String finalXML = XML + addxml.toString();
        if (finalXML.length() > 9999) {
        	logger.error("Exceed max XML size");
            return(null); // XML can only be less than 10000 varchar
        }
        acc.storeItemID(itemID, owner, owner_cid, owner, owner_cid, exp, type, finalXML, salt);
        if (!saveContentKey(itemID, owner, owner_cid, contentKey, acc)) {
            return(null);
        }
        for (RecipKey user : recipients) {
            if (user.remote) {// If remote user, we're going to send the data.
                EncrLib crypto = new EncrLib();
                String encryptedContentKey = crypto.encryptRSA(contentKey, user.pubkey);
                InterOx iox = new InterOx();
                if (!iox.sendItem(user.email, itemID, encryptedContentKey, exp, type, XML, salt)) {
                    return (null);
                }

            } else {// If local, we need to store it here
                if (user.cid < 0) // Guest account
                {
                    // user.cid = -user.cid;
                    logger.debug("Non OX user " + user.cid + " sending to filesystem");

                    if (!Storage.saveEncrObj(user.userid, user.cid, itemID, addHeader(encrypted, itemID, data.length()))) {
                        return (null);
                    }
                }
                if (!saveContentKey(itemID, user.userid, user.cid, user.pubkey, contentKey, acc)) {
                    return (null);
                }
                acc.storeItemID(itemID, user.userid, user.cid, owner, owner_cid, exp, type, XML, salt);
                if (user.cid != owner_cid) { // We need to save the item in each context, but as not owner, userid is 0
                    acc.storeItemID(itemID, user.userid, user.cid, owner, owner_cid, exp, type, XML, salt);
                }
            }

        }
        logger.info("Encrypted email" + start);
        return (encrypted);
    }

    public String createContentKey() {
        return (EncrLib.genRandom64Key());
        // return(EncrLib.getsalt());
    }

    /**
     * Save an item without specifying a content key
     *
     * @param owner owner of object
     * @param share who is on the share list
     * @param data the data of the object
     * @param exp expiration date
     * @param type
     * @param XML
     * @param specify and itemID
     * @return
     * @throws IOException
     * @throws SQLException
     */
    public byte[] encrItem(int owner, int cid, ArrayList<RecipKey> recipients, String data, long exp, int type, String XML, String itemID) throws IOException, SQLException {
        return (encrItem(owner, cid, recipients, data, exp, type, XML, itemID, createContentKey(), false, ""));
    }

    /**
     * Save an item without specifying a content key
     *
     * @param owner owner of object
     * @param share who is on the share list
     * @param data the data of the object
     * @param exp expiration date
     * @param type
     * @param XML
     * @return
     * @throws IOException
     * @throws SQLException
     */
    public byte[] encrItem(int owner, int cid, ArrayList<RecipKey> recipients, String data, long exp, int type, String XML) throws IOException, SQLException {
        String itemID = EncrLib.getUUID();
        return (encrItem(owner, cid, recipients, data, exp, type, XML, itemID, createContentKey(), false, ""));
    }

    /**
     * @param owner Owner of the object
     * @param recipients Arraylist of RecipKey
     * @param data Base64 String data of item
     * @param exp Expiration date
     * @param type Type of item
     * @param XML Extra XML data
     * @param itemID ItemID
     * @param contentKey ContentKey for encryption
     * @param salt Salt to be used
     * @param attachment Is attachment? If so, item data not stored again...assumed stored as email already.
     * @return
     * @throws IOException
     * @throws SQLException
     */
    public byte[] encrItem(int owner, int cid, ArrayList<RecipKey> recipients, String data, long exp, int type, String XML, String itemID, String contentKey, boolean attachment, String filename) throws IOException, SQLException {

        try {
            long start = System.currentTimeMillis();
            EncrLib encryptor = new EncrLib();
            // String encrypted = encryptor.encryptAES(data, contentKey, salt, Config.AESKeyLen);
            String encrypted = encryptor.encryptAES(data, contentKey);
            Access acc = new Access();
            String salt = EncrLib.getsalt();
            if (!attachment) {// Emails already saved, so we don't need to repeat for attachments
                acc.storeItemID(itemID, owner, cid, owner, cid, exp, 0, XML, salt);
                if (!saveContentKey(itemID, owner, cid, contentKey, acc)) {
                    return(null);
                }
            }

            if (recipients != null) {
                for (RecipKey user : recipients) {
                    if (!attachment) {
                        if (!saveContentKey(itemID, user.userid, user.cid, contentKey, acc)) {
                            return (null);
                        }
                        acc.storeItemID(itemID, user.userid, user.cid, owner, cid, exp, 0, XML, salt);
                    }
                    if (user.cid < 0) // Guest   account
                    {
                    	Storage.saveEncrObj(
                            user.userid,
                            user.cid,
                            itemID + "-" + filename.replace(".grd2", "").replace(".grd", ""),
                            addHeader(encrypted, itemID, data.length()));
                    }
                }
            }
            logger.info("Encrypted item " + itemID + " / " + start);
            return (addHeader(encrypted, itemID, data.length()));
        } catch (Exception e) {
            logger.error("Erroror while encrypting", e);
            return (null);
        }
    }

    public String decodeEncrObj(int user, int cid, EncryptedObj obj, String password) throws Exception {
        return (decodeEncrObj(user, cid, obj, password, null));
    }

    /**
     * Decode and encrypted object. Requires userid and password
     *
     * @param user Id of the user
     * @param obj The EncryptedObj
     * @param password
     * @return
     * @throws Exception
     */
    public String decodeEncrObj(int user, int cid, EncryptedObj obj, String password, String contentkey) throws Exception {
        try {
        	if (password == null) {
        		return ("{\"error\" : \"empty password\"}");
        	}
            long start = System.currentTimeMillis();
            if (CheckBad.isBad(obj.ItemID, Config.bad_password_count)) {
                return ("{\"Exceeded\" : " + Config.bad_minute_lock + "}");
            }
            com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
            com.openexchange.guard.database.RetrievedItem item = acc.getItemID(obj.ItemID, user, cid); // Get the item data
            if (item == null) {
                throw new NoItemException("Item not found");
            }
            if (item.expiration > 0) {// If our expiration not set to unlimited, check
                Date now = new Date();
                if (item.expiration < now.getTime()) {
                    logger.info("Expired");
                    return ("{\"status\" : \"expired\"}");
                }
            }
            if ((item.expiration < 0) || item.contentKey.equals("retracted")) {
                return ("{\"status\" : \"retracted\"}");
            }
            EncrLib encryptor = new EncrLib();
            if ((contentkey == null) || (contentkey.equals("undefined"))) {
                com.openexchange.guard.encr.GuardKeys keys = acc.getOldGuardKeys(user, cid); // Load the users private key from store
                if (keys != null) {
                    if (keys.version < 0) {   // If version < 0, then needs upgrade checking
                        PgpKeys.checkBadUID(keys, password);
                    }
                }

                if (keys == null) {
                    throw new NoMasterKeysException ("Unable to find/retrieve user keys");
                }
                contentkey = encryptor.decryptRSA(item.contentKey, keys.getDecodedPrivate(password));
                if (contentkey == null) {
                    CheckBad.addBad(obj.ItemID);
                    // bad password
                    throw new NoContentKeyException("Unable to decode content key");
                }
                logger.debug("Content key decoded");
            }
            // String decoded = encryptor.decryptAES(new String(obj.content), contentkey, item.salt, obj.keyLength );
            String decoded = encryptor.decryptAES(new String(obj.content), contentkey);
            acc.updateAccess(obj.ItemID, user, cid);
            logger.info("Decoded item " + obj.ItemID + " / " + start);
            return (decoded);
        } catch (java.lang.NullPointerException e) {
            logger.debug("Bad password");
            CheckBad.addBad(obj.ItemID);
            return (null);
        }

    }

    public boolean saveContentKey(String itemId, int userid, int cid, PublicKey key, String contentKey, Access acc) throws Exception {
        try {
            EncrLib crypto = new EncrLib();
            String encryptedContentKey = crypto.encryptRSA(contentKey, key);
            acc.storeContentKey(itemId, userid, cid, encryptedContentKey);
            return (true);
        } catch (SQLException e) {
        	logger.error("SQL Error while storing content key", e);
        }
        return (false);
    }

    private boolean saveContentKey(String itemId, int userid, int cid, String contentKey, Access acc) throws Exception {
        try {
            GuardKeys keys = acc.getKeys(userid, cid);
            return (saveContentKey(itemId, userid, cid, keys.getPublic(), contentKey, acc));
        } catch (Exception e) {
        	logger.error("Error while saving content key", e);
        }
        return (false);

    }

    public static String getID(byte[] data) {
        String head = new String(data);
        Pattern p = Pattern.compile("\\bID: [A-F0-9]{8}(?:-[A-F0-9]{4}){3}-[A-Z0-9]{12}\\b", Pattern.CASE_INSENSITIVE);
        Matcher m = p.matcher(head);
        if (m.find()) {
            return (m.group().substring(3).trim());
        }

        return (null);
    }


    public static byte[] addHeader(String strdata, String itemID, int length) {
        String filehead = StringUtils.rightPad("ID: " + itemID + ", Key: " + Config.AESKeyLen + (length > 0 ? (", Len: " + length) : "") + ", START-----", 100, "-");// Create header
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try {
            out.write(filehead.getBytes("UTF-8"));
            out.write(strdata.getBytes("UTF-8"));
        } catch (IOException e) {
            logger.error("IO error while adding header", e);
        }

        return (out.toByteArray());
    }

    public String addExtraPass(String data, String extrapass, String itemID) {
        EncrLib encr = new EncrLib();
        return ("EXTRA" + encr.encryptAES(data, extrapass, itemID, Config.AESKeyLen));
    }

    public String decodeExtraPass(String data, String extraPass, String itemID, int keylen) {
        EncrLib encr = new EncrLib();
        return (encr.decryptAES(data.substring(5), extraPass, itemID, keylen));
    }

    public String exportPrivateKey2(int id, int cid, String password) {
        try {
            Security.addProvider(new BouncyCastleProvider());
            Access acc = new Access();
            GuardKeys keys = acc.getKeys(id, cid);

            char[] pass = password.toCharArray();

            final JceOpenSSLPKCS8EncryptorBuilder builder = new JceOpenSSLPKCS8EncryptorBuilder(PKCS8Generator.DES3_CBC);
            builder.setRandom(new SecureRandom());
            builder.setPasssword(pass);
            builder.setIterationCount(100000);
            final OutputEncryptor oe = builder.build();
            JcaPKCS8Generator gen = new JcaPKCS8Generator(keys.getDecodedPrivate(password), oe);
            PemObject result = gen.generate();

            try (PemWriter writer = new PemWriter(new PrintWriter(System.out))) {
                writer.writeObject(result);
            } catch (Exception e) {
                logger.error("Error while writing private key", e);
            }
        } catch (Exception ex) {
            logger.error("Error while writing private key", ex);
        }
        return ("");
    }

    public String exportPrivateKey(int id, int cid, String password) {
        try {
            Security.addProvider(new BouncyCastleProvider());
            Access acc = new Access();
            GuardKeys keys = acc.getKeys(id, cid);

            char[] pass = password.toCharArray();
            byte[] salt = new byte[20];
            int icount = 100;
            PBEParameterSpec pbeParamSpec = new PBEParameterSpec(salt, icount);
            String pbeAlgorithm = "PBEWithMD5AndDES";
            PBEKeySpec pbeKeySpec = new PBEKeySpec(pass, salt, icount);
            SecretKeyFactory secretKeyFact = SecretKeyFactory.getInstance(pbeAlgorithm);
            Cipher cipher = Cipher.getInstance(pbeAlgorithm);
            cipher.init(Cipher.ENCRYPT_MODE, secretKeyFact.generateSecret(pbeKeySpec));
            byte[] ciphertext = cipher.doFinal(keys.getDecodedPrivate(password).getEncoded());

            AlgorithmParameters algparms = AlgorithmParameters.getInstance(pbeAlgorithm);
            algparms.init(pbeParamSpec);
            EncryptedPrivateKeyInfo encinfo = new EncryptedPrivateKeyInfo(algparms, ciphertext);

            StringWriter sigfos = new StringWriter(); // write PKCS#8 EncryptedPrivateKeyInfo to file
            sigfos.write("-----BEGIN ENCRYPTED PRIVATE KEY-----\r\n" + Base64.encodeBase64String(encinfo.getEncoded()) + "-----END ENCRYPTED PRIVATE KEY-----");
            sigfos.close();
        } catch (Exception ex) {
            logger.error("Error while exporting private key", ex);
        }
        return ("");
    }

    public static String decodeEPass (String epass, OxCookie cookie) {
    	Access acc = new Access();
        String token = acc.getToken(cookie.JSESSIONID);
        EncrLib enc = new EncrLib();
        return(enc.decryptAES(epass, token));
    }

    public static String encodeEPass (String password, OxCookie cookie, int userid, int cid) throws Exception {
    	EncrLib encr = new EncrLib();
        Access acc = new Access();
        String token = acc.getToken(cookie.JSESSIONID);
        if (token == null) {
            token = acc.newToken(cookie.JSESSIONID, userid, cid);
        }
        return (encr.encryptAES(password, token));
    }
}
