package com.openexchange.guard.database.repositories;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.model.DeletedKey;

/**
 * Data access repository for accessing deleted Keys
 */
public class DeletedKeyRepository {

    private static Logger logger = LoggerFactory.getLogger(DeletedKeyRepository.class);

    /**
     * Creates a new deleted keys
     * @param deletedKey an array of deleted keys to create
     * @throws Exception due an error
     */
    public void Create(DeletedKey...deletedKeys) throws Exception {
        DbQuery dbQuery = null;
        try {

            for (int i = 0; i < deletedKeys.length; i++) {
                dbQuery = new DbQuery();
                DeletedKey deletedKey = deletedKeys[i];
                String commandString = "INSERT INTO deletedKeys (id, cid, email, PGPSecret, salt, recovery, version, deleted_date) VALUES ( ?, ?, ?, ?, ?, ?, ?, NOW());";
                DbCommand command = new DbCommand(commandString);
                command.addVariables(deletedKey.getUserId());
                command.addVariables(deletedKey.getCid());
                command.addVariables(deletedKey.getEmail());
                command.addVariables(deletedKey.getPGPSecret());
                command.addVariables(deletedKey.getSalt());
                command.addVariables(deletedKey.getRecovery());
                command.addVariables(deletedKey.getVersion());
                dbQuery.writeOxGuard(command);
            }
        } catch (Exception e) {
            logger.error(
                String.format("Error while inserting into table deletedKeys: %s",
                              e.getMessage()));
            e.printStackTrace();
            throw e;
        }
        finally {
            if(dbQuery != null) {
                dbQuery.close();
            }
        }
    }

    /**
     * Gets the first deleted key for a user
     * @param email the email of the user to the get the deleted key's salt
     * @param cid the context id
     * @return the salt of the first found deleted key of the given user
     * @throws Exception On error during DB access
     */
    public DeletedKey getFirst(String email, int cid) throws Exception{

        DbQuery db = new DbQuery();

        try {
            String commandString = "SELECT id, cid, PGPSecret, recovery, salt, version, email FROM deletedKeys WHERE email = ? AND cid = ? ORDER BY version DESC";
            DbCommand command = new DbCommand(commandString);
            command.addVariables(email);
            command.addVariables(cid);

            db.readOG(command);
            if(db.next()) {
                return new DeletedKey(db.rs);
            }
            else{
                return null;
            }
        } catch (Exception e) {
            logger.error(String.format("Could not get deleted key for email %s in context %d",
                                       email,
                                       cid),
                         e);
            throw e;
        }
        finally{
            db.close();
        }
    }

    /**
     * Exposes every deleted key for a user for downloading by setting the exposed
     * @param email The email of the keys to expose
     * @param cid Context id
     * @throws Exception On error during DB access
     */
    public void setExposed(String email, int cid) throws Exception{

        String commandString = "UPDATE deletedKeys SET exposed=b'1', exposed_date= ? WHERE email= ? and cid = ?";
        DbCommand command = new DbCommand(commandString);

        SimpleDateFormat sdf = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String exposedDateString = sdf.format(new Date());


        command.addVariables(exposedDateString);
        command.addVariables(email);
        command.addVariables(cid);

        DbQuery db = new DbQuery();
        db.writeOxGuard(command);
        db.close();
    }

    /**
     * Removes the exposed flag from all deleted keys which have been exposed before the given date
     * @param before the date to
     * @throws Exception On error updating the DB
     * @return the number of keys which have been reset to "unexposed"
     */
    public int setAllUnexposed(Date before) throws Exception {
        DbQuery db = new DbQuery();

        try {
            String commandString = "UPDATE deletedKeys SET exposed=b'0', exposed_date=NULL WHERE exposed_date <= ?";
            DbCommand command = new DbCommand(commandString);

            SimpleDateFormat sdf = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            String beforeDate = sdf.format(before);
            command.addVariables(beforeDate);

            JsonObject response = db.writeOxGuard(command);
            JsonObject results = response.get("results").getAsJsonObject();

            //Getting the amount of keys which were updated by the DB-command
            int updatedCount = -1;
            for (Map.Entry<String,JsonElement> entry : results.entrySet()) {
                JsonElement elem = entry.getValue();
                JsonObject result = elem.getAsJsonObject();
                if(result.has("updated")){
                    updatedCount = result.get("updated").getAsInt();
                    break;
                }
            }

            if(updatedCount > -1) {
                return updatedCount;
            }
            else {
                throw new Exception("Invalid updated response from SQL update statement.");
            }

        } finally {
            db.close();
        }
    }

    /**
     * Gets a list of all deleted keys for the user, most recent first
     * @param email
     * @param id
     * @param cid
     * @return
     * @throws Exception
     */
    public ArrayList<DeletedKey> getAll (String email, int id, int cid) throws Exception {
        ArrayList<DeletedKey> keys = new ArrayList<DeletedKey> ();
        String command = "SELECT id, cid, PGPSecret, recovery, salt, version, email FROM deletedKeys WHERE cid = ? AND email = ? AND id = ? AND exposed = 1 ORDER BY version DESC";
        DbCommand com = new DbCommand(command);
        com.addVariables(cid);
        com.addVariables(email);
        com.addVariables(id);
        DbQuery db = new DbQuery();
        db.readOG(com);
        while (db.next()) {
            DeletedKey key = new DeletedKey(db.rs);
            keys.add(key);
        }
        db.close();
        return (keys);
    }
}
