package com.openexchange.guard.database.paging;

import com.amazonaws.services.cloudfront.model.InvalidArgumentException;
import com.google.gson.JsonObject;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.OxResultSet;

/**
 * Querying the configuration DB using paging
 */
public class ConfigDbPagedDbQuery {

    public final int DEFAULT_PAGESIZE = 999;
    private final int pageSize;
    private final String baseSql;
    private String currentSql;
    private Page page;
    private DbQuery query;

    /**
     * Initializes a new {@link ConfigDbPagedDbQuery}.
     * @param sql the SQL statement to execute using paging with the default page size of DEFAULT_PAGESIZE
     */
    public ConfigDbPagedDbQuery(String sql) {
        if(!sql.toUpperCase().contains("ORDER BY")) {
            throw new InvalidArgumentException("The sql statement is missing ORDER BY");
        }
        this.baseSql = sql;
        this.pageSize = DEFAULT_PAGESIZE;
    }

    private void UpdatePage() {
        if(page == null) {
            page = new Page(0,pageSize);
        }
        else{
            page = page.next();
        }
        this.currentSql = baseSql + " LIMIT " + page.getStartIndex() + "," + page.getPageSize() + ";";
    }

    private JsonObject readFromConfigDbInternal() throws Exception {
        UpdatePage();
        DbCommand command  = new DbCommand(currentSql);
        query = new DbQuery();
        query.readFromConfigDb(command);
        return query.result;
    }

    /**
     * Executes the Sql statement
     * @throws Exception
     */
    public void readFromConfigDb() throws Exception {
        readFromConfigDbInternal();
    }

    /**
     * Initializes a new {@link ConfigDbPagedDbQuery}.
     * @param sql the SQL statement to execute using paging
     * @param pageSize the pageSize to use for each page
     */
    public ConfigDbPagedDbQuery(String sql,int pageSize) {
        if(!sql.toUpperCase().contains("ORDER BY")) {
            throw new InvalidArgumentException("The sql statement is missing GROUP BY");
        }
        this.baseSql = sql;
        this.pageSize = pageSize;
    }

    /**
     * Gets the next result set
     *
     * If the current page is already at the end, the next page will be loaded using the
     * @return The next result set
     * @throws Exception
     */
    public OxResultSet next() throws Exception {
        if(query.next()) {
            return query.rs;
        }
        else {
            readFromConfigDbInternal();
            if(query.getRowCount() > 0) {
                return next();
            }
            return null;
        }
    }
}