package com.openexchange.guard.database.model;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.jcajce.JcePBESecretKeyDecryptorBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.database.OxResultSet;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.BadPasswordException;
import com.openexchange.guard.pgp.PgpHandler;
import com.openexchange.guard.pgp.PgpKeys;

/**
 * Class for a key that was deleted and saved in the deletedKeyTable
 * @author greg
 *
 */
public class DeletedKey {

    private static Logger logger = LoggerFactory.getLogger(DeletedKey.class);

    private final int userid;
    private final int cid;
    private final PGPSecretKeyRing secretKeyRing;
    private final String recovery;
    private final String salt;
    private final int version;
    private final String email;
    private final String pgpSecret;

    public String getSalt() { return this.salt; }
    public int getUserId() { return this.userid; }
    public int getCid() { return this.cid; }
    public PGPSecretKeyRing getSecretKeyRing() { return this.secretKeyRing; }
    public String getRecovery() {return this.recovery;}
    public int getVersion() {return this.version;}
    public String getEmail() {return this.email;}
    public String getPGPSecret() {return this.pgpSecret;}

    /**
     * Creating a deleted key object based on a existing key
     * @param key The key to create the deleted key from
     */
    public DeletedKey(GuardKeys key) throws Exception {
        this.userid = key.userid;
        this.cid = key.contextid;
        this.secretKeyRing = new PGPSecretKeyRing(Base64.decodeBase64(key.getEncodedPGPSecret()), new BcKeyFingerprintCalculator());
        this.recovery = key.recovery;
        this.salt = key.getSalt();
        this.version = key.version;
        this.email = key.email;
        this.pgpSecret = key.getEncodedPGPSecret();
    }

    /**
     * Create the key from a resultSet of the database
     * @rs the result
     */
    public DeletedKey (OxResultSet rs) throws IOException, PGPException, Exception {
        this.userid = rs.getInt("id");
        this.cid = rs.getInt("cid");
        this.secretKeyRing = new PGPSecretKeyRing(Base64.decodeBase64(rs.getString("PGPSecret")), new BcKeyFingerprintCalculator());
        this.recovery = rs.getString("recovery");
        this.salt = rs.getString("salt");
        this.version = rs.getInt("version");
        this.email = rs.getString("email");
        this.pgpSecret = rs.getString("PGPSecret");
    }

    /**
     * Verify the password is correct for this key
     * @param password
     * @return
     */
    public boolean verifyPassword (String password) {
        char [] pass = EncrLib.getSHA(password, salt).toCharArray();
        PGPSecretKey sec_key = secretKeyRing.getSecretKey();
        try {
            PGPPrivateKey decryptor = sec_key.extractPrivateKey(new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(pass));
            if (decryptor != null) {
                return(true);
            }
        } catch (PGPException ex) {
            return (false);
        }
        return (false);
    }

    /**
     * Export the private key using the supplied password
     * @param password
     * @return
     * @throws PGPException
     * @throws BadPasswordException
     * @throws IOException
     */
    public String export (String password) throws PGPException, BadPasswordException, IOException {
        GuardKeys key = new GuardKeys(this);
        String export = PgpKeys.ExportPGPPrivate(key, password);
        return (export);
    }

    /**
     * Get arraylist of userids for this keyring
     * @return
     */
    public ArrayList<String> getUserIds() {
        ArrayList<String> ids = new ArrayList<String> ();
        Iterator<String> it = secretKeyRing.getSecretKey().getUserIDs();
        while (it.hasNext()) {
            ids.add(it.next());
        }
        return (ids);
    }

    /**
     * Get arraylist of fingerprints for this keyring
     * @return
     */
    public ArrayList<String> getKeyFingerprints () {
        ArrayList<String> fps = new ArrayList<String> ();
        Iterator<PGPSecretKey> keys = secretKeyRing.getSecretKeys();
        while (keys.hasNext()) {
            PGPSecretKey skey = keys.next();
            fps.add(PgpHandler.getFingerPrintString(skey.getPublicKey().getFingerprint()));
        }
        return(fps);
    }


}
