/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.guard.database;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import com.google.gson.JsonObject;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.pgp.PgpKeys;
import com.openexchange.guard.util.Core;

/**
 * Class for access of mysql database for key and content ID retrieval
 *
 * @author greg
 */
public class Access {

    private final String mysqlIP = com.openexchange.guard.config.Config.getMysqlIP();// IP address of configdb

    private final String mysqlUsername = com.openexchange.guard.config.Config.getMysqlUsername();

    private final String mysqlPassword = com.openexchange.guard.config.Config.getMysqlPassword();

    private final String OGmysqlIP = com.openexchange.guard.config.Config.getMysqlIPOxGuard();

    private final String ctxUsername = ""; // Username for access ox database that contains context

    private final String ctxPassword = ""; // Password

    private final String url = ""; // URL of the mysql server containing the context data

    private final String db = ""; // Database name

    private String OGmysqlConnectionString = "";

    private String ConfigmysqlConnectionString = "";

	private static Logger logger = LoggerFactory.getLogger(Access.class);

    static {
        try {
            Class.forName("com.mysql.jdbc.Driver");
        } catch (ClassNotFoundException e) {
            logger.error("Could not load jdbc driver", e);
        }
    }

    public Access() {
        ConfigmysqlConnectionString = Config.getConfigDbConnectionString();
        OGmysqlConnectionString = "jdbc:mysql://" + OGmysqlIP + "/oxguard?user=" + mysqlUsername + "&password=" + mysqlPassword;

    }

    public String getEmail(int userid, int cid) throws SQLException {

        String email = "";
        try {
            String command = "SELECT mail from user WHERE cid = ? AND id = ?";
            DbCommand com = new DbCommand(command);
            com.addVariables(cid);
            com.addVariables(userid);
            DbQuery db = new DbQuery();
            db.readFromDB(com, cid);
            if (db.next()) {
                email = db.rs.getString("mail");
            }
            db.close();
        } catch (Exception ex) {
            logger.error("Error reading e-mail " + cid + "/" + userid + " from db", ex);
        }
        return (email);
    }

    public static String dbformat(String data) {
        if (data == null) {
            return (data);
        }
        data = data.replace("'", "\'");
        data = data.replace("\u00ef\u00bf\u00bd", "\'");
        data = data.replace("\\", "\\\\");
        return (data);
    }

    /**
     * Execute a mysql command
     *
     * @param command
     * @throws Exception
     */
    private void mysqlExecute(String command, int cid) throws Exception {
        DbQuery db = new DbQuery();
        DbCommand com = new DbCommand(command);
        JsonObject result = db.writeToDB(com, cid);
        // DBConnection conn = manager.getConn(cid);
        // Statement stmt = null;
        // try {
        // stmt = conn.con.createStatement();
        // stmt.execute(command);
        // } catch (Exception ex) {
        // logger.error(ex);
        // } finally {
        // manager.clean(cid, stmt, conn);
        // }
    }

    /**
     * Execute a mysql command
     *
     * @param command
     * @throws Exception
     */
    private void mysqlExecute(String command) throws Exception {
        DbCommand com = new DbCommand(command);
        DbQuery db = new DbQuery();
        db.writeOxGuard(com);
        db.close();
    }

    /*
     * Store newly created keys for user
     */
    public int storeKeys(int id, int cid, String email, GuardKeys key, String recovery, String language, boolean userCreated) throws Exception {
        int shard = 0;

        if (cid < 0) { // If guest, get shard number, then check tables exist
            shard = Sharding.getNextShard();
            if (shard == 0) {
            	logger.error("Unable to get non-ox shard id");
            	return(0);
            }
            Sharding.checkTables(shard, false);
        } else {
            if (cid > 0)
             {
                Sharding.checkTables(cid, true); // Check these OG tables exist
            }
        }

        String command = "INSERT INTO og_KeyTable (id, cid, PGPSecret, PGPPublic, RSAPrivate, RSAPublic, Recovery, Salt, email, lastMod, lang, keyid) VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ";
        if (userCreated) {
            command += "NOW()";
        } else {
            command += "NULL";
        }
        command += ", ?, ?)";
        DbCommand com = new DbCommand(command);
        com.addVariables((id == 0 ? null : id));
        com.addVariables(cid);
        com.addVariables(key.getEncodedPGPSecret());
        com.addVariables(key.exportPgpPublicKey());
        com.addVariables(key.getEncodedPrivate());
        com.addVariables(key.getEncodedPublic());
        com.addVariables(recovery);
        com.addVariables(key.getSalt());
        com.addVariables(email);
        com.addVariables(language);
        com.addVariables(key.pubring.getPublicKey().getKeyID());
        try {
            DbQuery db = new DbQuery();
            if (cid >= 0) {// OX users go in OX database, otherwise guest shard
                if (id < 0) {// Master key is negative, otherwise place in proper shard
                    db.writeOxGuard(com);
                } else {
                    db.writeToDB(com, cid);
                }
            } else {
                db.writeToDb(com, Sharding.database_name(shard));
            }

            int userid = id;
            if (id == 0) {// For guest accounts, we need to return the new userid
                command = "SELECT id FROM og_KeyTable WHERE email = ?";
                com = new DbCommand(command);
                com.addVariables(email);
                if (cid >= 0) {
                    if (id > 0) {
                        db.readFromDB(com, cid);
                    } else {
                        db.readOG(com);
                    }
                } else {
                    db.readFromDB(com, Sharding.database_name(shard));
                }
                if (db.next()) {
                    userid = db.rs.getInt("id");
                }
            }
            if (userid > 0) {
                addEmail(email, userid, cid, shard);
            }
            db.close();

            key.contextid = cid;
            if (userid > 0) {
                PgpKeys.addPublicKeyIndex(key);
            }

            return (userid);
        } catch (SQLException e) {
            logger.error("SQL error while storing key", e);
            return (0);
        }

    }

    public boolean updatePubring (GuardKeys key) throws IOException {
        String command = "UPDATE og_KeyTable SET PGPPublic = ? WHERE keyid = ?";
        DbCommand com = new DbCommand(command);
        com.addVariables(key.exportPGPPublicKeyRing());
        com.addVariables(key.keyid);
        DbQuery db = new DbQuery();
        try {
            db.write(com, key.userid, key.contextid);
        } catch (Exception e) {
            logger.error("Error updating PGP Public KeyRing " , e);
            return (false);
        }
        return (true);
    }

    public boolean addNewKey (GuardKeys newkey) throws Exception {

    	int version = 0;
    	// Check for highest version of key
    	String getversion = "SELECT MAX(version) FROM og_KeyTable WHERE id = ? and cid = ?";
    	DbCommand ver = new DbCommand(getversion);
    	ver.addVariables(newkey.userid);
    	ver.addVariables(newkey.contextid);
        String update = "UPDATE og_KeyTable SET current = 0 WHERE id = ? and cid = ?";
        DbCommand up = new DbCommand(update);
        up.addVariables(newkey.userid);
        up.addVariables(newkey.contextid);
        DbQuery db = new DbQuery();
        if (newkey.contextid >= 0) {// OX users go in OX database, otherwise guest shard
            db.writeToDB(up, newkey.contextid);
        } else {
            db.write(up, newkey.userid, newkey.contextid);
        }
        db.close();
        String command = "INSERT INTO og_KeyTable (id, cid, PGPSecret, PGPPublic, RSAPrivate, RSAPublic, Recovery, Salt, email, lastMod, lang, keyid, version, question, answer, settings) VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, " +
            "NOW(), ?, ?, ?, ?, ?, ?)";
        DbCommand com = new DbCommand(command);
        com.addVariables(newkey.userid);
        com.addVariables(newkey.contextid);
        com.addVariables(newkey.getEncodedPGPSecret());
        com.addVariables(newkey.exportPgpPublicKey());
        com.addVariables(newkey.getEncodedPrivate());
        com.addVariables(newkey.getEncodedPublic());
        com.addVariables(newkey.recovery);
        com.addVariables(newkey.getSalt());
        com.addVariables(newkey.email);
        com.addVariables(newkey.language);
        com.addVariables(newkey.pubring.getPublicKey().getKeyID());
        com.addVariables(newkey.version);
        com.addVariables(newkey.question);
        com.addVariables(newkey.answer);
        com.addVariables(newkey.settings == null ? null : newkey.settings.toString());
        db = new DbQuery();
        if (newkey.contextid >= 0) {// OX users go in OX database, otherwise guest shard
            db.writeToDB(com, newkey.contextid);
        } else {
            db.write(com, newkey.userid, newkey.contextid);
        }
        db.close();
        PgpKeys.addPublicKeyIndex(newkey);
        return (true);
    }

    public void updateKeys(int id, int cid, GuardKeys key, String recovery, boolean reset) throws Exception {
        String command = "UPDATE og_KeyTable Set PGPSecret = '" + key.getEncodedPGPSecret() + "', RSAPrivate = " + (key.privKeyNull() ? "NULL" : ("'" + key.getEncodedPrivate() + "'")) + ", Recovery = '" + recovery +
        		"', lastMod = " + (reset ? "NULL" : "NOW()") + " WHERE id = " + id + " AND cid = " + cid + " AND current = 1;";
        try {
            DbCommand com = new DbCommand(command);
            DbQuery db = new DbQuery();
            db.write(com, id, cid);
            db.close();
        } catch (SQLException e) {
            logger.error("SQL error while updating keys", e);
        }

    }

    /**
     * Update PGP Key record updating PGP Secret key and recovery on duplicate
     * @param id
     * @param cid
     * @param key
     * @param recovery
     * @throws Exception
     */
    public void updateDuplPGPKey (int id, int cid, GuardKeys key, String recovery, Long dupl) throws Exception {
        String command = "UPDATE og_KeyTable SET PGPSecret = ?, PGPPublic = ?, Recovery = ?, Salt = ? WHERE id = ? AND cid = ? and keyid = ?";
        try {
            DbCommand com = new DbCommand(command);
            com.addVariables(key.getEncodedPGPSecret());
            com.addVariables(key.exportPgpPublicKey());
            com.addVariables(recovery);
            com.addVariables(key.getSalt());
            com.addVariables(id);
            com.addVariables(cid);
            com.addVariables(dupl);
            DbQuery db = new DbQuery();
            db.write(com, id, cid);
            db.close();
        } catch (SQLException e) {
            logger.error("SQL error while updating PGP keys", e);
        }
    }

    public void updatePGPKeys (int id, int cid, GuardKeys key) throws Exception {
    	String command = "UPDATE og_KeyTable Set PGPSecret = ?, PGPPublic = ? WHERE id = ? AND cid = ? AND keyid = ?";
        try {
            DbCommand com = new DbCommand(command);
            com.addVariables(key.getEncodedPGPSecret());
            com.addVariables(key.exportPgpPublicKey());
            com.addVariables(id);
            com.addVariables(cid);
            com.addVariables(key.keyid);
            DbQuery db = new DbQuery();
            db.write(com, id, cid);
            db.close();
            PgpKeys.addPublicKeyIndex(key);
        } catch (SQLException e) {
            logger.error("SQL error while updating PGP keys", e);
        }
    }

    public static void updatePin(int id, int cid, String pin) throws Exception {
    	String command = "UPDATE og_KeyTable Set question = 'PIN', answer = ? WHERE id = ? AND cid = ?;";
        try {
            DbCommand com = new DbCommand(command);
            com.addVariables(pin);
            com.addVariables(id);
            com.addVariables(cid);
            DbQuery db = new DbQuery();
            db.write(com, id, cid);
            db.close();
        } catch (SQLException e) {
            logger.error("SQL error while updating keys", e);
        }
    }


    public GuardKeys getKeys(int id, int cid) throws Exception {
        int shard = 0;
        if (cid < 0) {
            shard = Sharding.getShard(id, cid);
        }
        return (getKeys(id, cid, shard, null, true));
    }

    public GuardKeys getKeys(int id, int cid, long keyid, boolean current) throws Exception {
    	int shard = 0;
        if (cid < 0) {
            shard = Sharding.getShard(id, cid);
        }
        return (getKeys(id, cid, shard, Long.toString(keyid), current));
    }

    public GuardKeys getKeys (int id, int cid, int shard) throws Exception {
    	return (getKeys(id, cid, shard, null, true));
    }

    /**
     * Retrieve keys from Mater KeyTable
     *
     * @param id member ID
     * @return
     * @throws Exception
     */
    public GuardKeys getKeys(int id, int cid, int shard, String keyid, boolean current) throws Exception {
        GuardKeys key = null;
        String command = "SELECT PGPSecret, PGPPublic, RSAPrivate, RSAPublic, Salt, misc, settings, email, lastMod, question, answer, cid, id, keyid, version, current, lang, Recovery FROM og_KeyTable WHERE id = ? AND cid = ?";
        if (current) {
            command += " AND og_KeyTable.current = 1 ";
        }
        if (keyid != null) {
            command += " AND keyid = ?";
        }
        DbCommand com = new DbCommand(command);
        com.addVariables(id);
        com.addVariables(cid);
        if (keyid != null) {
            com.addVariables(keyid);
        }
        DbQuery db = new DbQuery();
        if (cid >= 0) {
            if (id < 0) {// minus 1 is the recovery password, which is in OG
                db.readOG(com);
            } else {
                db.readFromDB(com, cid);
            }
        } else {
            db.readFromShard(com, shard);
        }
        try {
            if (db.next()) {
                key = getKeyFromRS (db.rs);
            }
        } catch (Exception ex) {
            logger.error("Error while iterating result set for keys", ex);

        }
        return (key);
    }

    /**
     * Retrieve keys from Mater KeyTable
     *
     * @param id member ID
     * @return
     * @throws Exception
     */
    public GuardKeys getOldGuardKeys(int id, int cid) throws Exception {
        GuardKeys key = null;
        String command = "SELECT PGPSecret, PGPPublic, RSAPrivate, RSAPublic, Salt, misc, settings, email, lastMod, question, answer, cid, id, keyid, version, current, lang, Recovery FROM og_KeyTable WHERE id = ? AND cid = ? AND version < 1";  //Version may be 0 or -1 for Guard key (-1 during upgrade)
        DbCommand com = new DbCommand(command);
        com.addVariables(id);
        com.addVariables(cid);
        DbQuery db = new DbQuery();
        int shard = 0;
        if (cid < 0) {
            shard = Sharding.getShard(id, cid);
        }
        if (cid >= 0) {
            if (id < 0) {// minus 1 is the recovery password, which is in OG
                db.readOG(com);
            } else {
                db.readFromDB(com, cid);
            }
        } else {
            db.readFromShard(com, shard);
        }
        try {
            if (db.next()) {
                key = getKeyFromRS (db.rs);
            }
        } catch (Exception ex) {
            logger.error("Error while iterating result set for keys", ex);

        }
        return (key);
    }

    /**
     * Get a list of all the Keys associated with the account
     * @param id
     * @param cid
     * @return
     * @throws Exception
     */
    public ArrayList<GuardKeys> getAllKeys (int id, int cid) throws Exception {
        String command = "SELECT PGPSecret, PGPPublic, RSAPrivate, RSAPublic, Salt, misc, settings, email, lastMod, question, answer, cid, id, keyid, version, current, lang, Recovery FROM og_KeyTable WHERE id = ? AND cid = ? ORDER BY version DESC";
        DbCommand com = new DbCommand(command);
        com.addVariables(id);
        com.addVariables(cid);
        DbQuery db = new DbQuery();
        db.read(com, id, cid);
        ArrayList<GuardKeys> keys = new ArrayList<GuardKeys> ();
        while (db.next()) {
            GuardKeys key = getKeyFromRS (db.rs);
            keys.add(key);
        }
        return (keys);
    }

    public ArrayList<GuardKeys> getAllKeysInContext (int cid) throws Exception {
        String command = "SELECT PGPSecret, PGPPublic, RSAPrivate, RSAPublic, Salt, misc, settings, email, lastMod, question, answer, cid, id, keyid, version, current, lang, Recovery FROM og_KeyTable WHERE cid = ?";
        DbCommand com = new DbCommand(command);
        com.addVariables(cid);
        DbQuery db = new DbQuery();
        db.readFromDB(com, cid);
        ArrayList<GuardKeys> keys = new ArrayList<GuardKeys> ();
        while (db.next()) {
            GuardKeys key = getKeyFromRS (db.rs);
            keys.add(key);
        }
        return (keys);
    }

    /**
     * Create GuardKey from database result set
     * @param rs
     * @param id  ID of user, -1 if unknown
     * @param cid
     * @return
     * @throws InvalidKeySpecException
     * @throws NoSuchAlgorithmException
     * @throws Exception
     */
    private GuardKeys getKeyFromRS (OxResultSet rs) throws InvalidKeySpecException, NoSuchAlgorithmException, Exception {
        GuardKeys key = new GuardKeys();
        key.setPGPSecretFromString(rs.getString("PGPSecret"));
        key.setPGPPublicKeyRing(rs.getString("PGPPublic"));
        key.setEncodedPrivate(rs.getString("RSAPrivate"));
        key.setPublicKeyFrom64String(rs.getString("RSAPublic"));
        key.setSalt(rs.getString("Salt"));
        key.misc = rs.getString("misc");
        key.settings = rs.getJson("settings");
        key.email = rs.getString("email");
        key.lastup = rs.getDate("lastMod");
        key.question = rs.getString("question");
        key.answer = rs.getString("answer");
        key.contextid = rs.getInt("cid");
        key.userid = rs.getInt("id");
        key.keyid = rs.getLong("keyid");
        key.version = rs.getInt("version");
        key.current = rs.getBit("current");
        key.language = rs.getString("lang");
        key.recovery = rs.getString("Recovery");
        key.recoveryAvail = key.recovery == null ? false : !key.recovery.equals("");
        if (key.recovery != null) {
            if (key.recovery.equals("-1") || (key.lastup == null)) {
                key.passwordNeeded = true;
            }
        }
        return (key);
    }

    public GuardKeys getKeysFromEmail(String email) throws Exception {
        return (getKeysFromEmail(email, 0L));
    }
    /**
     * Retrieve keys from Mater KeyTable
     *
     * @param id member ID
     * @return
     * @throws Exception
     */
    public GuardKeys getKeysFromEmail(String email, Long keyid) throws Exception {
        GuardKeys key = null;
        String command = "SELECT cid, id, db FROM og_email WHERE email LIKE (?);";
        DbCommand com = new DbCommand(command);
        com.addVariables(email);
        DbQuery db = new DbQuery();
        db.readOG(com);
        if (db.next()) {
            int cid = db.rs.getInt("cid");
            int id = db.rs.getInt("id");
            int shard = db.rs.getInt("db");
            db.close();
            if (keyid != 0L) {
                return (getKeys(id, cid, shard, Long.toString(keyid), false));
            }
            return (getKeys(id, cid, shard));
        }
        db.close();
        return (key);
    }

    /**
     * Store Item into item table
     *
     * @param ItemID
     * @param owner
     * @param exp
     * @param type
     * @param XML
     * @throws Exception
     */
    public void storeItemID(String ItemID, int recip, int recip_cid, int owner, int owner_cid, long exp, int type, String XML, String salt) throws Exception {
        // Store item. If multiple recips within the same context database, then there will be duplicates we can ignore
        if (XML.length() > 3072) {
            XML = XML.substring(0, 3071);
        }
        String command = "INSERT INTO og_encrypted_items (Id, Owner, Expiration, Type, XML, Salt, Owner_cid) VALUES (?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE Type = Type";
        DbQuery db = new DbQuery();
        DbCommand com = new DbCommand(command);
        com.addVariables(ItemID);
        com.addVariables(owner);
        com.addVariables(exp);
        com.addVariables(type);
        com.addVariables(XML);
        com.addVariables(salt);
        com.addVariables(owner_cid);
        db.write(com, recip, recip_cid);
        db.close();

    }

    public void storeContentKey(String itemId, int userid, int cid, String contentKey) throws Exception {

        String command = "INSERT INTO og_content_keys (ItemId, UserId, CID, ContentKey, Status, ID) VALUES (?, ?, ?, ?, 0, NULL);";
        DbCommand com = new DbCommand(command);
        com.addVariables(itemId);
        com.addVariables(userid);
        com.addVariables(cid);
        com.addVariables(contentKey);
        DbQuery db = new DbQuery();
        db.write(com, userid, cid);
    }

    public void deleteRecovery (int id, int cid) throws Exception {
        String command = "UPDATE og_KeyTable SET Recovery = '' WHERE id = ? and cid = ?";
        DbCommand com = new DbCommand(command);
        com.addVariables(id);
        com.addVariables(cid);
        DbQuery db = new DbQuery();
        db.write(com, id, cid);
    }

    private String convertDate(java.util.Date dt) {
        if (dt == null) {
            return (null);
        }
        java.text.SimpleDateFormat sdf = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String currentTime = sdf.format(dt);
        return (currentTime);
    }

    public Boolean itemExists(String ItemId, int user, int cid) throws Exception {
        String command = "SELECT Id FROM og_encrypted_items WHERE og_encrypted_items.ID = ? AND Owner = ? AND Owner_cid = ?;";
        DbQuery db = new DbQuery();
        DbCommand com = new DbCommand(command);
        com.addVariables(ItemId);
        com.addVariables(user);
        com.addVariables(cid);
        db.read(com, user, cid);
        if (db.hasResults()) {
            return (true);
        }
        return (false);
    }

    public void updateAccess(String ItemID, int user, int cid) {
        String command = "UPDATE og_content_keys set Status = Status + 1 WHERE ItemID = ? AND UserId = ? AND CID = ?";
        try {
            DbQuery db = new DbQuery();
            DbCommand com = new DbCommand(command);
            com.addVariables(ItemID);
            com.addVariables(user);
            com.addVariables(cid);
            db.write(com, user, cid);

        } catch (Exception e) {
            logger.error("Error updating status", e);
        }
    }

    public void getCount(String ItemID, int userid, int cid) throws Exception {
        String xml = "";
        String command = "SELECT XML FROM og_encrypted_items WHERE ID = ? AND Owner = ? AND Owner_cid = ?;";
        DbQuery db = new DbQuery();
        DbCommand com = new DbCommand(command);
        com.addVariables(ItemID);
        com.addVariables(userid);
        com.addVariables(cid);
        db.read(com, userid, cid);
        if (db.next()) {
            xml = db.rs.getString("XML");
        }
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        try {
            DocumentBuilder builder = factory.newDocumentBuilder();
            org.w3c.dom.Document doc = builder.parse(new InputSource(new ByteArrayInputStream(xml.getBytes("utf-8"))));
            doc.getDocumentElement().normalize();
            NodeList users = doc.getElementsByTagName("u");
            StringBuilder output = new StringBuilder();
            output.append("[");
            for (int i = 0; i < users.getLength(); i++) {
                if (i > 0) {
                    output.append(", ");
                }
                Element user = (Element) users.item(i);
                String email = user.getElementsByTagName("e").item(0).getTextContent();
                int uid = Integer.parseInt(user.getElementsByTagName("i").item(0).getTextContent());
                int ucid = Integer.parseInt(user.getElementsByTagName("c").item(0).getTextContent());
                output.append("{\"Email\" : \"" + email + "\", \"Count\" : \"" + getCountbyUser(ItemID, uid, ucid) + "\"}");
            }
            output.append("]");
        } catch (Exception ex) {
            logger.error("Error while getting count of encrypted items", ex);
        }

    }

    private int getCountbyUser(String itemid, int id, int cid) {
        int count = 0;
        try {
            String command = "SELECT Status FROM og_content_keys WHERE ItemId = ? AND UserID = ? AND CID = ?";
            DbCommand com = new DbCommand(command);
            com.addVariables(itemid);
            com.addVariables(id);
            com.addVariables(cid);
            DbQuery db = new DbQuery();
            db.read(com, id, cid);
            if (db.next()) {
                count = db.rs.getInt("Status");
            }
        } catch (Exception ex) {
            logger.error("Error while getting count of encrypted items for user " + id, ex);
        }
        return (count);
    }

    public RetrievedItem getItemID(String ItemId, int user, int cid) throws Exception {
        String command = "SELECT ContentKey, Expiration, Owner, Salt, Type, XML FROM og_encrypted_items INNER JOIN og_content_keys ON og_encrypted_items.Id = og_content_keys.ItemId WHERE og_encrypted_items.ID = ? AND UserId = ? AND CID = ?;";
        DbCommand com = new DbCommand(command);
        com.addVariables(ItemId);
        com.addVariables(user);
        com.addVariables(cid);
        DbQuery db = new DbQuery();
        db.read(com, user, cid);
        if (db.next()) {
            OxResultSet rs = db.rs;
            RetrievedItem item = new RetrievedItem();
            item.contentKey = rs.getString("ContentKey");
            item.expiration = rs.getLong("Expiration");
            item.owner = rs.getInt("Owner");
            item.salt = rs.getString("Salt");
            item.Type = rs.getInt("Type");
            item.XML = rs.getString("XML");
            return (item);

        }
        return (null);
    }

    /**
     * Insert new token to oxGuard_sessions for encrypting passwords
     *
     * @param sessionID
     * @param userid
     * @return
     * @throws Exception
     */
    public String newToken(String sessionID, int userid, int cid) throws Exception {
        // String token = EncrLib.getsalt();
        String token = EncrLib.genRandom64Key();
        String command = "INSERT INTO oxguard_session (sessionid, token, date, userid, cid) VALUES (?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE sessionid = ?, token = ?, date = ?;";
        DbCommand com = new DbCommand(command);
        com.addVariables(sessionID);
        com.addVariables(token);
        String now = convertDate(new Date());
        com.addVariables(now);
        com.addVariables(userid);
        com.addVariables(cid);
        com.addVariables(sessionID);
        com.addVariables(token);
        com.addVariables(now);
        DbQuery db = new DbQuery();
        db.writeOxGuard(com);
        db.close();
        return (token);
    }

    /**
     * Delete token from sessions for logout
     *
     * @param sessionID
     * @throws Exception
     */
    public void deleteToken(String sessionID) throws Exception {
        if (sessionID == null) {
            return;
        }
        String command = "DELETE FROM oxguard_session WHERE sessionid = ?";
        DbCommand com = new DbCommand(command);
        com.addVariables(sessionID);
        DbQuery db = new DbQuery();
        try {
            db.writeOxGuard(com);
            command = "delete from oxguard_session where date < NOW() - INTERVAL 1 DAY"; // cleanup old sessions
            com = new DbCommand(command);
            db.writeOxGuard(com);
        } catch (SQLException e) {
            logger.error("SQL error while deleting oxguard session " + sessionID, e);
        }
        db.close();
    }

    /**
     * Get the token associated with the sessionID
     *
     * @param sessionID
     * @return
     */
    public String getToken(String sessionID) {
        try {
            String command = "SELECT token FROM oxguard_session WHERE sessionid = ?";
            DbCommand com = new DbCommand(command);
            com.addVariables(sessionID);
            DbQuery db = new DbQuery();
            db.readOG(com);
            String token = null;
            if (db.next()) {
                token = db.rs.getString("token");
            }
            db.close();
            return (token);
        } catch (Exception e) {
            logger.error("Error while retrieving token from session " + sessionID, e);
        }
        return (null);
    }

    public void saveCache(String itemID, int userid, String location) throws Exception {
        String command = "INSERT INTO FileCache VALUES (?, ?, ?, NOW())";
        DbCommand com = new DbCommand(command);
        com.addVariables(itemID);
        com.addVariables(userid);
        com.addVariables(location);
        DbQuery db = new DbQuery();
        try {
            db.writeOxGuard(com);
        } catch (Exception e) {
            logger.error("Error while saving in file cache", e);
        }
        db.close();
    }

    // Lookup the public key and userid for a given email address. Returns null if not found

    /**
     * Get number of times item has been opened. Returns -1 if not found
     *
     * @param userid
     * @param cid
     * @param itemid
     * @return
     */
    public int getCount(int userid, int cid, String itemid) {
        int response = -1;
        try {
            String command = "SELECT Status FROM og_content_keys WHERE ItemID = ? AND UserID = ? AND CID = ?";
            DbQuery db = new DbQuery();
            DbCommand com = new DbCommand(command);
            com.addVariables(itemid);
            com.addVariables(userid);
            com.addVariables(cid);
            db.read(com, userid, cid);
            if (db.next()) {
                response = db.rs.getInt("Status");
            }
        } catch (Exception ex) {
            logger.error("Error while getting count of content keys", ex);
        }
        return (response);
    }

    /**
     * Retract an item by setting expiration to negative number Salt is required for extra security check
     */

    public boolean retract(int userid, int cid, String itemID, String salt, boolean delete) {
        try {
            DbQuery db = new DbQuery();
            String command = "SELECT COUNT(Id) AS count FROM og_encrypted_items WHERE  og_encrypted_items.Id = ? AND Salt = ?";
            // Salt must be same to verify authorized sender
            DbCommand com = new DbCommand(command);
            com.addVariables(itemID);
            com.addVariables(salt);
            db.read(com, userid, cid);
            if (db.next()) {
            	if (db.rs.getInt("count") < 1) {
            		db.close();
            		return(false); // no items to retract
            	}
            }
            // Delete the content key if so configured, otherwise expire the item
            command = "update og_encrypted_items SET Expiration = IF (Expiration = 0, -1, -Expiration) WHERE ID = ? AND Expiration >= 0 AND Salt = ?";

            if (delete) {
            	command = "update og_content_keys SET ContentKey = 'retracted' WHERE ItemId = ?";
            }
            com = new DbCommand(command);
            com.addVariables(itemID);
            if (!delete) {
                com.addVariables(salt);
            }
            db.write(com, userid, cid);
            command = "update og_content_keys SET Status = IF (Status = 0, -1, -Status -1) WHERE ItemId = ?";
            com = new DbCommand(command);
            com.addVariables(itemID);
            db.write(com, userid, cid);
            return (true);
        } catch (Exception ex) {
            logger.error("Error while retracting an encrypted item", ex);
            return (false);
        }
    }

    /**
     * Get XML data from an item. Returns null if not found. User must be owner
     *
     * @param userid
     * @param cid
     * @param itemid
     * @return
     */
    public String getXML(int userid, int cid, String itemid) {
        String response = null;
        try {
            String command = "SELECT XML FROM og_encrypted_items WHERE Id = ? AND Owner = ? AND Owner_cid = ?";
            DbQuery db = new DbQuery();
            DbCommand com = new DbCommand(command);
            com.addVariables(itemid);
            com.addVariables(userid);
            com.addVariables(cid);
            db.read(com, userid, cid);
            if (db.next()) {
                response = db.rs.getString("XML");
            }
        } catch (Exception ex) {
            logger.error("Error while getting XML for encrypted item", ex);
        }
        return (response);
    }

    public boolean storeQuestion(int id, int cid, String question, String answer) {
        EncrLib encr = new EncrLib();
        String encranswer = null;
        if (answer != null) {
            encranswer = encr.encryptAES(answer, Config.rpass, question, Config.AESKeyLen);
        }
        String command = "UPDATE og_KeyTable SET question = ?, answer = ?" + " WHERE id = ? AND cid = ?";
        try {
            DbQuery db = new DbQuery();
            DbCommand com = new DbCommand(command);
            com.addVariables(question);
            com.addVariables(encranswer);
            com.addVariables(id);
            com.addVariables(cid);
            db.write(com, id, cid);
        } catch (Exception e) {
            logger.error("Error while storing reminder question", e);
            return(false);
        }
        return(true);
    }

    public ArrayList<PGPPublicKeyRing> getPublicKeyRing(String email, int userid, int cid) {
        return (getPublicKeyRing(email, userid, cid, false));
    }
    /**
     * Search for public keys for email
     * CID = 0 if search for local only
     * @param email
     * @param cid
     * @param oxOnly  exclude guests
     * @return
     */
    public ArrayList<PGPPublicKeyRing> getPublicKeyRing(String email, int userid, int cid, boolean oxOnly) {
        try {
        	ArrayList<PGPPublicKeyRing> pubRings = new ArrayList<PGPPublicKeyRing>();
        	String command = "select id, cid, local FROM PGPKeys WHERE email = ?";
        	if (oxOnly) {
                command += " AND cid > 0";
            }
        	DbCommand com = new DbCommand(command);
        	com.addVariables(email);
        	DbQuery db = new DbQuery();
        	db.readOG(com);
        	while (db.next()) {
        		String id = db.rs.getString("id");
        		int key_cid = db.rs.getInt("cid");
        		boolean local = db.rs.getBit("local");
        		if (local) {  // If local, we just use the one we have
        			GuardKeys key = getKeysFromEmail(email);
        			pubRings.add(key.pubring);
        			db.close();
        			return (pubRings);
        		} else {
        			if (cid == 0) {
                        return (null);
                    }
        			command = "SELECT PGPPublic FROM og_pgp_keys WHERE Email = ? AND ids LIKE ? AND cid = ? AND (userid = ? OR share_level > 0)";
        			DbCommand com2 = new DbCommand(command);
        			com2.addVariables(email);
        			com2.addVariables("%" + id + "%");
        			com2.addVariables(cid);
        			com2.addVariables(userid);
        			DbQuery db2 = new DbQuery();
        			db2.readFromDB(com2, key_cid);
        			if (db2.next()) {
        				GuardKeys key = new GuardKeys();
        				key.setPGPPublicKeyRing(db2.rs.getString("PGPPublic"));
        				pubRings.add(key.pubring);
        			}
        			db2.close();
        		}
        	}
        	return (pubRings);
        } catch (Exception ex) {
            logger.error("Error while retrieving public key ring for " + email, ex);
            return (null);
        }
    }

    /**
     * Get the public keyring associated with the id.  Can be long or Hex String (8 or 16 byte)
     * @param req_id
     * @param userid
     * @param cid
     * @return
     */
    public PGPPublicKeyRing getPublicKeyRingById(Object req_id, int userid, int cid) {
        try {
        	String command = "";
        	String hexid = "";
        	DbCommand com;
        	if (req_id.getClass() == String.class) {
        		command = "select email, cid, keyid, local FROM PGPKeys WHERE hexid = ?";
        		hexid = (String) req_id;
        		if (hexid.length() > 8) {  // If this is a 16 byte key, then lets convert it back to the long id
        			req_id = Core.getLongPGP(hexid);
        		}
        	}
        	if (req_id.getClass() == Long.class) {
        		command = "select email, cid, keyid, local FROM PGPKeys WHERE id = ?";
        		hexid = Core.PGPKeyId((Long) req_id);
        	}
        	DbQuery db = new DbQuery();
        	com = new DbCommand(command);
    		com.addVariables(req_id);
        	db.readOG(com);
        	if (db.next()) {
        		String email = db.rs.getString("email");
        		int key_cid = db.rs.getInt("cid");
        		long keyid = db.rs.getLong("keyid");
        		boolean local = db.rs.getBit("local");
        		if (local) {  // If local, we just use the one we have
        			GuardKeys key = getKeysFromEmail(email, keyid);
        			db.close();
        			return (key.pubring);
        		} else {
        			db.close();
        			if (cid == 0) {
                        return (null);
                    }
        			return (getUploadedPublicKeyRingById (hexid, userid, cid));
        		}
        	}
        	db.close();
        	if (cid == 0) {
                return (null);
            }
        	return (getUploadedPublicKeyRingById (hexid, userid, cid));
        } catch (Exception ex) {
            logger.error("Error while retrieving public key ring for " + req_id, ex);
            return (null);
        }
    }

    private PGPPublicKeyRing getUploadedPublicKeyRingById (String req_id, int userid, int cid) throws Exception {
    	String command = "SELECT PGPPublic FROM og_pgp_keys WHERE cid = ? AND (userid = ? OR share_level = 1) AND ids LIKE ?";
		DbCommand com2 = new DbCommand(command);
		com2.addVariables(cid);
		com2.addVariables(userid);
		com2.addVariables("%" + req_id + "%");
		DbQuery db2 = new DbQuery();
		db2.readFromDB(com2, cid);
		if (db2.next()) {
			GuardKeys key = new GuardKeys();
			key.setPGPPublicKeyRing(db2.rs.getString("PGPPublic"));
			db2.close();
			return(key.pubring);
		}
		db2.close();
		return (null);
    }

    public static void addEmail(String email, int id, int cid, int shard) throws Exception {
        String command = "INSERT INTO og_email VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE id = ?, cid = ?, db = ?";
        DbCommand com = new DbCommand(command);
        com.addVariables(email);
        com.addVariables(id);
        com.addVariables(cid);
        com.addVariables(shard);
        com.addVariables(id);
        com.addVariables(cid);
        com.addVariables(shard);
        DbQuery db = new DbQuery();
        db.writeOxGuard(com);
        db.close();
    }
}
