package com.openexchange.guard.cron;

import java.util.Date;
import java.util.TimerTask;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.database.repositories.DeletedKeyRepository;

/**
 * A job which resets the "exposed" flag of deleted keys which have been exposed longer since a given time
 */
public class ResetExposedKeysJob extends TimerTask {

    private static Logger logger = LoggerFactory.getLogger(ResetExposedKeysJob.class);
    private final int exposedKeyDurationInHours;

    /**
     * Constructor
     * @param exposedKeyDurationInHours the amount of hours specifying how long a deleted and exposed key will be marked as "exposed"
     */
    public ResetExposedKeysJob(int exposedKeyDurationInHours) {
        this.exposedKeyDurationInHours = exposedKeyDurationInHours;
    }

    /**
     * Calculating boundary date for resetting exposed keys
     * @param the duration in hours how long a key can be marked as "exposed"
     * @return the boundary date before exposed keys can be reseted to "not exposed"
     */
    private Date CalculateBoundaryDate(int exposedDurationHours) {
        DateTime now = new DateTime();
        Date before =  now.minusHours(exposedDurationHours).toDate();
        return before;
    }

    @Override
    public void run() {
        try {
            //Getting the boundary date before which we reset the exposed flag for exposed keys
            Date before = CalculateBoundaryDate(this.exposedKeyDurationInHours);

            //Resetting the exposed flag
            logger.debug(String.format("Unexposing exposed/deleted keys which have been exposed before %s ...",before.toString()));
            int updatedKeyCount = new DeletedKeyRepository().setAllUnexposed(before);
            logger.debug(String.format("Unexposed %d exposed/deleted keys which have been exposed before %s",updatedKeyCount,before.toString()));

        } catch (Exception e) {
            logger.error("Error while doing schedued cleanup of exposed keys", e);
        }
    }
}
