/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.guard.config;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import javax.crypto.Cipher;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.net.util.SubnetUtils;
import org.apache.commons.net.util.SubnetUtils.SubnetInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ch.qos.logback.classic.Level;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.dbpool.ContextDBConnection;
import com.openexchange.guard.dbpool.Manager;
import com.openexchange.guard.encr.EncrLib;

public class Config {
	private static Logger logger = LoggerFactory.getLogger(Config.class);

	public static int version = 7;  // Version for upgrader

	public static String serverVersion = "n/a";  // Value of server version from Java Manifest

    public static Level logLevel = Level.ALL;

    public static Level jettyLogLevel = Level.ERROR;

    public enum LogType {
        udp, file, terminal
    }

    public final static String fileName = "etc/guard.properties";

    final static String PassFile = "oxguardpass";

    public static String mysqlIP = "";

    public static String ogMySqlReadIP = null;

    public static String ogMySqlIP = "";
    
    public static String ogShardReadIP = null;
    
    public static String ogShardIP = null;

    public static String mysqlUsername = "";

    public static String readmysqlUsername = null;

    public static String readmyqlPassword = null;

    public static String mysqlPassword = "";

    public static String storageType = "file";

    public static String configdbName = "configdb";

    public static String rootFileDir = "/var/spool/open-xchange/guard/uploads";

	public static String S3Endpoint = "";

	public static String S3BucketName = "";

    public static String S3Region = "us-west-2";

    public static String S3AccessKey = "";

    public static String S3SecretKey = "";

    public static String OXBackend = "";

    public static String OXBackendPath = "/ajax/";

    public static int OXBackendIdleTime = 62;  // Close idle connections to backend after time in seconds

    public static int keyCacheCheckInterval = 29;   // Time between check of RSA pool count in seconds

    public static String OXBackendUser = "";

    public static String OXBackendPassword = "";

    public static String SupportApiUser = "";

    public static String SupportApiPassword = "";

    public static String OXUrl = "";

    public static String externalReaderPath = "/appsuite/api/oxguard/reader/reader.html";

    public static String remoteLog = "";

    public static boolean deleteOnRetract = true;

    public static String logDir = "/var/log/open-xchange/guard";

    public static int remotePort = 517;

    public static LogType logType = LogType.terminal;

    public static int AESKeyLen = 128;

    public static int RSAKeyLen = 2048;

    public static int cachedays = 30;

    public static String codePath = "";

    public static int cronHour = 2;

    public static int newPassLength = 10;

    public static String SMTPServer = "";

    public static String SMTPUsername = "";

    public static String SMTPPassword = "";

    public static int SMTPPort = 25;

    public static int jettyport = 8080;

    public static int sslport = 8443;

    public static boolean usessl = false;

    public static String sslkeystore = "";

    public static String sslkeyname = "";

    public static String sslkeypass = "";

    public static String mpass = "";

    public static String rpass = "";

    public static int max_idle_db = 2;

    public static int max_open_db = 10;

    public static int max_http_connections = 50;	// Max HTTP connections at once to a given route (backend, etc)

    public static int max_total_connections = 50;   // Max total HTTP pooled connections to all destinations

    public static boolean no_recovery = false;

    public static boolean no_delete_recovery = true;  // Users allowed to delete their password recovery info

    public static boolean no_delete_keys = false;  // Users allowed to delete their private keys or not

    public static boolean ok_private = true;

    public static boolean ok_extra_password = true;

    public static int bad_minute_lock = 10;

    public static int bad_password_count = 5;

    public static int max_remote = 100;  // Rate limit, max number of emails received remotely in lockout period

    public static int bad_ip_count = 10;

    public static boolean rsa_cache = true;

    public static int rsa_cache_count = 100;

    public static int rsa_certainty = 256;
    
    public static int key_valid_days = 3650;

    public static int min_password_length = 6;

    public static String default_lang = "en_US";

    public static boolean default_oxguard_enabled = true;

    public static String token = "";

    public static String etcpath = "/opt/open-xchange/guard/etc";

    public static int ox_backend_port = 8009;

    public static boolean backend_ssl = false;

    public static int shard_size = 10000;

    public static String db_prefix = "oxguard_";

    public static boolean demo = false;

    public static String upsellTemplate = "/opt/open-xchange/guard/templates/upsell.html";

    public static String passwordFromAddress = null;  // Address and name for the FROM header for new password emails

    public static String passwordFromName = null;

    public static boolean useStartTls = true;

    public static boolean showStatus = true;

    public static int pgpCacheDays = 7;

    public static String mailResolverUrl = null;  // MailResolverURL, must be defined so when email address is added to the end it will be valid

    public static ArrayList<String> pgpLocations = new ArrayList<String> ();

    public static String mailIdDomain = "";

    public static ArrayList<SubnetInfo> pgpWhitelist = new ArrayList<SubnetInfo>();
    
    public static String apiUserAgent = "Open-Xchange Guard Server";

    /**
     * The amount of hours how long the deleted keys will be stayed "exposed" for downloading
     */
    public static int ExposedKeyDurationInHours = 168 /* 1 Week */;

    /**
     * Parse config file and update variables
     */
    public static void loadConfig() {
        getRoot();
        Path path = Paths.get(codePath + fileName);
        try {
            List<String> lines = Files.readAllLines(path, StandardCharsets.UTF_8);
            for (String line : lines) {
                if (!line.startsWith("#")) {
                    String[] data = line.split("=");
                    if (data.length > 1) {
                        try {
                            String variable = data[0].toLowerCase().trim();
                            switch (variable) {
                            case "com.openexchange.guard.guestsmtpserver":
                                SMTPServer = data[1].trim();
                                break;
                            case "com.openexchange.guard.sslkeystore":
                                sslkeystore = data[1].trim();
                                break;
                            case "com.openexchange.guard.sslport":
                                sslport = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.usessl":
                                if (data[1].trim().toLowerCase().contains("true")) {
                                    usessl = true;
                                } else {
                                    usessl = false;
                                }
                                break;
                            case "com.openexchange.guard.rsacache":
                                if (data[1].trim().toLowerCase().contains("true")) {
                                    rsa_cache = true;
                                } else {
                                    rsa_cache = false;
                                }
                                break;
                            case "com.openexchange.guard.rsacachecount":
                                rsa_cache_count = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.keycachecheckinterval":
                            	keyCacheCheckInterval = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.rsacertainty":
                            	rsa_certainty = Integer.parseInt(data[1].trim());
                            	break;
                            case "com.openexchange.guard.norecovery":
                                if (data[1].trim().toLowerCase().contains("true")) {
                                    no_recovery = true;
                                } else {
                                    no_recovery = false;
                                }
                                break;
                            case "com.openexchange.guard.nodeleteprivate":
                                if (data[1].trim().toLowerCase().contains("true")) {
                                    no_delete_keys = true;
                                } else {
                                    no_delete_keys = false;
                                }
                                break;
                            case "com.openexchange.guard.nodeleterecovery":
                                if (data[1].trim().toLowerCase().contains("true")) {
                                    no_delete_recovery = true;
                                } else {
                                    no_delete_recovery = false;
                                }
                                break;
                            case "com.openexchange.guard.okprivate":
                                if (data[1].trim().toLowerCase().contains("true")) {
                                    ok_private = true;
                                } else {
                                    ok_private = false;
                                }
                                break;
                            case "com.openexchange.guard.okextrapassword":
                                if (data[1].trim().toLowerCase().contains("true")) {
                                    ok_extra_password = true;
                                } else {
                                    ok_extra_password = false;
                                }
                                break;
                            case "com.openexchange.guard.demo":
                                if (data[1].trim().toLowerCase().contains("true")) {
                                    demo = true;
                                } else {
                                    demo = false;
                                }
                                break;
                            case "com.openexchange.guard.deleteonretract":
                            	if (data[1].trim().toLowerCase().contains("true")) {
                            		deleteOnRetract = true;
                            	} else {
                            		deleteOnRetract = false;
                            	}
                            	break;
                            case "com.openexchange.guard.showstatus":
                            	if (data[1].trim().toLowerCase().contains("true")) {
                            		showStatus = true;
                            	} else {
                            		showStatus = false;
                            	}
                            	break;
                            case "com.openexchange.guard.defaultoxguardenabled":
                                if (data[1].trim().toLowerCase().contains("true")) {
                                    default_oxguard_enabled = true;
                                } else {
                                    default_oxguard_enabled = false;
                                }
                                break;
                            case "com.openexchange.guard.newpasslength":
                                newPassLength = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.oxbackendport":
                                ox_backend_port = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.oxbackendpath":
                            	OXBackendPath = data[1].trim();
                            	break;
                            case "com.openexchange.guard.oxbackendidletime":
                            	OXBackendIdleTime = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.backend_ssl":
                            	if (data[1].trim().toLowerCase().contains("true")) {
                            		backend_ssl = true;
                            	} else {
                            		backend_ssl = false;
                            	}
                            	break;
                            case "com.openexchange.guard.minpasswordlength":
                                min_password_length = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.sslkeyname":
                                sslkeyname = data[1].trim();
                                break;
                            case "com.openexchange.guard.sslkeypass":
                                sslkeypass = data[1].trim();
                                break;
                            case "com.openexchange.guard.guestsmtpusername":
                                SMTPUsername = data[1].trim();
                                break;
                            case "com.openexchange.guard.guestsmtppassword":
                                SMTPPassword = data[1].trim();
                                break;
                            case "com.openexchange.guard.guestsmtpport":
                                SMTPPort = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.usestarttls":
                            	if (data[1].trim().toLowerCase().contains("true")) {
                            		useStartTls = true;
                            	} else {
                            		useStartTls = false;
                            	}
                            	break;
                            case "com.openexchange.guard.jettyport":
                                jettyport = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.databaseusername":
                                mysqlUsername = data[1].trim();
                                break;
                            case "com.openexchange.guard.databasepassword":
                                mysqlPassword = data[1].trim();
                                break;
                            case "com.openexchange.guard.configdbname":
                            	configdbName = data[1].trim();
                            	break;
                            case "com.openexchange.guard.readusername":
                                readmysqlUsername = data[1].trim();
                                break;
                            case "com.openexchange.guard.readpassword":
                                readmyqlPassword = data[1].trim();
                                break;
                            case "com.openexchange.guard.configdbhostname":
                                mysqlIP = data[1].trim();
                                break;
                            case "com.openexchange.guard.oxguarddatabasehostname":
                                ogMySqlIP = data[1].trim();
                                break;
                            case "com.openexchange.guard.oxguarddatabaseread":
                                ogMySqlReadIP = data[1].trim();
                                break;
                            case "com.openexchange.guard.oxguardsharddatabase":
                                ogShardIP = data[1].trim();
                                break;
                            case "com.openexchange.guard.oxguardshardread":
                                ogShardReadIP = data[1].trim();
                                break;
                            case "com.openexchange.guard.storage.type":
                                storageType = data[1].trim();
                                break;
                            case "com.openexchange.guard.storage.file.uploaddirectory":
                                rootFileDir = data[1].trim();
                                if (data.length > 2) {
                                    rootFileDir = rootFileDir + ":" + data[2].trim(); // windows file format may have :
                                }
                                break;
                            case "com.openexchange.guard.storage.s3.endpoint":
                                S3Endpoint = data[1].trim();
                                break;
                            case "com.openexchange.guard.storage.s3.bucketname":
                                S3BucketName = data[1].trim();
                                break;
                            case "com.openexchange.guard.storage.s3.region":
                                if (data[1].trim().length() > 0) {
                                	S3Region = data[1].trim();
                                }
                                break;
                            case "com.openexchange.guard.storage.s3.accesskey":
                                S3AccessKey = data[1].trim();
                                break;
                            case "com.openexchange.guard.storage.s3.secretkey":
                                S3SecretKey = data[1].trim();
                                break;
                            case "com.openexchange.guard.restapihostname":
                                OXBackend = data[1].trim();
                                break;
                            case "com.openexchange.guard.restapiusername":
                                OXBackendUser = data[1].trim();
                                break;
                            case "com.openexchange.guard.restapipassword":
                                OXBackendPassword = data[1].trim();
                                break;
                            case "com.openexchange.guard.supportapiusername":
                                SupportApiUser = data[1].trim();
                                break;
                            case "com.openexchange.guard.supportapipassword":
                                SupportApiPassword = data[1].trim();
                                break;
                            case "com.openexchange.guard.upselltemplate":
                            	upsellTemplate = data[1].trim();
                            	break;
                            case "com.openexchange.guard.remoteloghostname":
                                remoteLog = data[1].trim();
                                break;
                            case "com.openexchange.guard.logdirectory":
                                logDir = data[1].trim();
                                File dir = new File(logDir);
                                if (!dir.isDirectory()) {
                                    if (!dir.mkdir()) {
                                        System.out.println("Error creating log directory " + logDir);
                                    }
                                }
                                break;
                            case "com.openexchange.guard.remotelogport":
                                remotePort = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.mailresolverurl":
                            	for (int i = 1; i < data.length; i++) {
                            		if (i == 1) {
                            			mailResolverUrl = data[i].trim();
                            		} else {
                            			mailResolverUrl = mailResolverUrl + "=" + data[i].trim();
                            		}
                            	}
                            	break;
                            case "com.openexchange.guard.cronhour":
                                cronHour = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.cachedays":
                                cachedays = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.pgpcachedays":
                                pgpCacheDays = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.badipcount":
                                bad_ip_count = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.badpasswordcount":
                                bad_password_count = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.badminutelock":
                                bad_minute_lock = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.maxremote":
                            	max_remote = Integer.parseInt(data[1].trim());
                            	break;
                            case "com.openexchange.guard.maxhttpconnections":
                            	max_http_connections = Integer.parseInt(data[1].trim());
                            	break;
                            case "com.openexchange.guard.maxtotalconnections":
                            	max_total_connections = Integer.parseInt(data[1].trim());
                            	break;
                            case "com.openexchange.guard.shardsize":
                            	shard_size = Integer.parseInt(data[1].trim());
                            	break;
                            case "com.openexchange.guard.logtype":
                                if (data[1].contains("udp")) {
                                    logType = LogType.udp;
                                }
                                if (data[1].contains("terminal")) {
                                    logType = LogType.terminal;
                                }
                                if (data[1].contains("file")) {
                                    logType = LogType.file;
                                }
                                break;
                            case "com.openexchange.guard.loglevel":
                                if (data[1].contains("debug")) {
                                    logLevel = Level.DEBUG;
                                }
                                if (data[1].contains("error")) {
                                    logLevel = Level.ERROR;
                                }
                                if (data[1].contains("normal")) {
                                    logLevel = Level.INFO;
                                }
                                break;
                            case "com.openexchange.guard.jettyloglevel":
                            	if (data[1].contains("debug")) {
                                    jettyLogLevel = Level.DEBUG;
                                }
                                if (data[1].contains("error")) {
                                    jettyLogLevel = Level.ERROR;
                                }
                                if (data[1].contains("normal")) {
                                    jettyLogLevel = Level.INFO;
                                }
                                break;
                            case "com.openexchange.guard.aeskeylength":
                                AESKeyLen = Integer.parseInt(data[1].trim());
                                if (AESKeyLen > Cipher.getMaxAllowedKeyLength("AES")) {
                                    logger.error("INVALID AES KEY LENGTH " + AESKeyLen + ", Max is " + Cipher.getMaxAllowedKeyLength("AES"));
                                    System.exit(1);
                                }
                                logger.debug("OK Loading config file");
                                break;
                            case "com.openexchange.guard.rsakeylength":
                                RSAKeyLen = Integer.parseInt(data[1].trim());
                                if (RSAKeyLen > Cipher.getMaxAllowedKeyLength("RSA")) {
                                    logger.error("INVALID RSA KEY LENGTH");
                                    System.exit(1);
                                }
                                break;
                            case "com.openexchange.guard.keyvaliddays":
                                key_valid_days = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.externalemailurl":
                                OXUrl = data[1].trim().replace("http://", "").replace("https://", "");;
                                if (data.length > 2) { // may have port in command line
                                    OXUrl = OXUrl + ":" + data[2].trim();
                                }
                                break;
                            case "com.openexchange.guard.externalreaderpath":
                            	externalReaderPath = data[1].trim().replace("http://", "").replace("https://", "");
                            	if (data.length > 2) { // may have port in command line
                            	    externalReaderPath = externalReaderPath + ":" + data[2].trim();
                                }
                                break;
                            case "com.openexchange.guard.etcpath":
                                etcpath = data[1].trim();
                                break;
                            case "com.openexchange.guard.passwordfromaddress":
                            	passwordFromAddress = data[1].trim();
                            	break;
                            case "com.openexchange.guard.passwordfromname":
                            	passwordFromName = data[1].trim();
                            	break;
                            case "com.openexchange.guard.maxidledb":
                                max_idle_db = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.maxopendb":
                                max_open_db = Integer.parseInt(data[1].trim());
                                break;
                            case "com.openexchange.guard.defaultlanguage":
                                default_lang = data[1].trim();
                                break;
                            case "com.openexchange.guard.mailiddomain":
                                mailIdDomain = data[1].trim();
                                break;
                            case "com.openexchange.guard.apiuseragent":
                                apiUserAgent = data[1].trim();
                                break;
                            case "com.openexchange.guard.publicpgpdirectory":
                            	String urlStrings = data[1].trim();
                            	String [] urls = urlStrings.split(",");
                            	for (String url : urls) {
                            		url = url.trim();
                            		if (url.startsWith("htt") || url.startsWith("hkp")) {
                            			pgpLocations.add(url);
                            		}
                            	}
                            	break;
                            case "com.openexchange.guard.publickeywhitelist":
                            	String list = data[1].trim();
                            	String [] items = list.split(",");
                            	for (String item : items) {
                            		try {
                            			SubnetUtils util = new SubnetUtils(item.trim());
                            			pgpWhitelist.add(util.getInfo());
                            		} catch (Exception e) {
                            			logger.error("Bad IP address in whitelist", e);
                            		}

                            	}
                            case "com.openexchange.guard.exposedkeydurationinhours":
                                    try {
                                        ExposedKeyDurationInHours = Integer.parseInt(data[1].trim());
                                    } catch (Exception e) {
                                        logger.error(
                                            "Error while loading configuration com.openexchange.guard.exposedKeyDurationInHours. Using default value.",
                                            e);

                                    }
                            default:
                            	logger.info("Unknown configuration tag " + variable);
                                break;
                            }

                        } catch (Exception ex) {
                            logger.error("Error while loading config", ex);
                        }
                    }
                }
            }
            if (mysqlUsername.equals("")) {
            	System.out.println("Possible malformed config file, or missing mysql username data");
            }
        } catch (IOException e) {
            logger.error("Error with config file " + path);
            System.exit(1);
        }
        path = Paths.get(codePath + PassFile);
        try {
            List<String> lines = Files.readAllLines(path, StandardCharsets.UTF_8);
            for (String line : lines) {
                if (line.startsWith("MC")) {
                    mpass = line.substring(2);
                }
                if (line.startsWith("RC")) {
                    rpass = line.substring(2);
                }
            }
        } catch (Exception ex) {
            logger.error("Unable to load password file: " + ex.toString());
        }

    }

    public static void loadMysqlConfigData() {
        ContextDBConnection dbcon = null;
        try {

            DbCommand com = new DbCommand("SELECT global_settings.key, global_settings.value FROM global_settings");
            DbQuery db = new DbQuery();
            db.readOG(com);
            while (db.next()) {
                String field = db.rs.getString("key");
                switch (field) {
                case "token":
                    token = db.rs.getString("value");
                    break;
                }
            }
            db.close();
        } catch (Exception ex) {
            logger.error("Error while loading mysql config", ex);
        }
        Manager.closeOG(dbcon);
    }

    public static String getRoot() {
        File file = new File("/opt/open-xchange/guard/" + fileName);
        if (file.exists()) {
            codePath = "/opt/open-xchange/guard/";
            return (codePath);
        }
        String cpath = Config.class.getProtectionDomain().getCodeSource().getLocation().getPath();
        try {
            codePath = URLDecoder.decode(cpath, "UTF-8");
            if (codePath.contains("/")) {
                codePath = codePath.substring(0, codePath.lastIndexOf("/") - 1);
                codePath = codePath.substring(0, codePath.lastIndexOf("/") + 1);
            }
        } catch (UnsupportedEncodingException e1) {
        	logger.error("Error while getting root of file " + fileName, e1);
        }
        return (codePath);
    }

    private static void check() {
        if (mysqlIP.equals("")) {
            loadConfig();
        }
    }

    public static String getMysqlIP() {
        check();
        return (mysqlIP);
    }

    public static String getMysqlIPOxGuard() {
        check();
        return (ogMySqlIP);
    }

    public static String getMysqlPassword() {
        check();
        return (mysqlPassword);
    }

    public static String getMysqlUsername() {
        check();
        return (mysqlUsername);
    }

    public static String getStorageType() {
        check();
        return (storageType);
    }

    public static String getRootFileDirectory() {
        check();
        return (rootFileDir);
    }

    public static String getS3Endpoint() {
    	check();
		return S3Endpoint;
	}

    public static String getS3BucketName() {
    	check();
		return S3BucketName;
	}

	public static String getS3Region() {
		check();
		return S3Region;
	}

	public static String getS3AccessKey() {
		check();
		return S3AccessKey;
	}

	public static String getS3SecretKey() {
		check();
		return S3SecretKey;
	}

    public static String getOxBackend() {
        check();
        return (OXBackend);
    }

    public static String getBasicAuthenticationEncoding() throws UnsupportedEncodingException {
    	String userPassword = OXBackendUser + ":" + OXBackendPassword;
        return new String(Base64.encodeBase64(userPassword.getBytes("UTF-8")));
    }

    public static String getConfigDbConnectionString() {
        return ("jdbc:mysql://" + mysqlIP + "/" + configdbName + "?user=" + mysqlUsername + "&password=" + mysqlPassword);
    }

    public static String getMessageID () {
        if (mailIdDomain.equals("")) {
            if (OXUrl.contains("/")) {
                mailIdDomain = OXUrl.substring(0, OXUrl.indexOf("/"));
            }
            if (mailIdDomain.contains(":")) {
                mailIdDomain = mailIdDomain.substring(0, mailIdDomain.indexOf(":"));
            }
        }
        return ("<" + EncrLib.getUUID() + "@" + mailIdDomain + ">");
    }
}
