/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2011
 */

define('oxguard/mail/sender_compose', ['io.ox/core/extensions', 'gettext!oxguard/mail/sender_compose'],
                              function (ext, gt) {
    
    'use strict';
    
    var timer = null;
    
    ////////////   SENDING
    
    // Get files for upload
    function getFiles(baton, mail, authorization, guestdata, pass, draft) {
        var secure_reply = getReturnId(baton);
        // If needed, add passwords to json data
        if (pass !== undefined) {
            mail.attachpassword = pass.password;
            mail.attachextra = pass.extrapass;
        }
        if (window.FormData === undefined) {
            alert (gt('Unsupported browser'));
        } else {
            var def = $.Deferred();

            var formData = new FormData();
            formData.append('json_0', JSON.stringify(mail));
            if (guestdata !== null) {
                formData.append('guest_message', guestdata.message);
                if (guestdata.pin !== undefined) {
                    formData.append('pin', guestdata.pin);
                }
                
            }

            for (var l = 0; l < mail.data.files.length; l++) {
                formData.append('file' + l, mail.data.files[l]);
            }
            var xhr = new XMLHttpRequest();
            var dest = '/oxguard/mail?action=emailform';
            var url = ox.apiRoot + dest;
            var params = '&session=' + ox.session + '&userid=' + ox.user_id + '&cid=' + ox.context_id +
            '&sentfolder=' + encodeURIComponent(window.oxguarddata.sentfolder) +
            '&draftfolder=' + encodeURIComponent(window.oxguarddata.draftfolder) +
            '&draft=' + (draft ? 'true' : 'false') + 
            '&lang=' + ox.language + (guestdata === null ? '' : ('&guestlanguage=' + guestdata.language)) +
            ((secure_reply === undefined) ? '' : ('&sr=' + secure_reply));
            
                if (baton.model.get('PGPSignature')) {
                    params = params + '&pgpsign=true';
                }
                if (baton.model.get('PGPInline') === 'inline') {
                    params = params + '&pgpinline=true';
                }
                
            xhr.open('POST', url + params);
            
            xhr.onreadystatechange = function () {
                if (xhr.readyState === 4) {
                    if (xhr.status !== 200) {
                        require (['oxguard/core/log'], function (log) {
                            log.fail(xhr, url, params, mail);
                        });
                        if (xhr.status === 503) {
                            def.reject(gt('Service Unavailable'));
                        }
                        def.reject(xhr.responseText);
                    } else {
                        def.resolve(xhr.response);
                    }
                }
            };
            xhr.send(formData);
            return (def);
        }
        
    }
    
   
    // Main send function
    function send(baton, auth, guestdata, pass, draft)
    {
        require (['oxguard/mail/oxguard_mail_compose_core'], function (write_core) {
            write_core.saveToHeader(baton);  // Make sure headers updated
        });
        
        var mail = {
                data : baton.view.model.getMail() // Get the email
        };
        if (!draft && (mail.data.to.length + mail.data.cc.length + mail.data.bcc.length) === 0) {
            require (['io.ox/core/notifications'], function (notifications) {
                notifications.yell('error', gt('Mail has no recipients'));
            });
            return;
        }
        var type = 0;
        if (draft) {
            if (baton.view.model.get('refId') !== undefined) {
                type = 3;
                mail.data.msgref = baton.view.model.get('refId');
            } else type = 4;
        }
        mail.data.sendtype = type;
        var keys = baton.model.get('gkeys');
        var isguest = false;
        for (var i = 0; i < keys.length; i++) {
            if (keys[i].result === 'guest') isguest = true;
        }
        if (isguest && (guestdata === null) && !draft) {
            guestOptions(baton, auth);
            return;
        }
        if (pass === undefined) {
            var passReq = checkPasswordNeeded(baton, mail);
            if (passReq > 0) {
                getPassword(baton, auth, guestdata, passReq);
                return;
            }
        }
        spinRefresh();
        getFiles(baton, mail, auth, guestdata, pass, draft)
        .done(function (result) {
            if (draft) {
                require (['io.ox/core/notifications'], function (notifications) {
                    notifications.yell('success', gt('Message was saved'));
                });
                try {
                    var dat = $.parseJSON(result);
                    baton.view.model.set('refId', dat.data.folder_id + '/' + dat.data.id);
                    saveDraftId(baton);
                } catch (e) {
                    console.log(result);
                    console.log(e);
                }
                baton.view.isSaving = false;
            }
            stopSpinRefresh();
        })
        .fail(function (r) {
            stopSpinRefresh();
            require (['io.ox/core/notifications'], function (notifications) {
                if (draft) {
                    notifications.yell('error', gt('An error occurred saving the email.') + ' ' + r);
                    baton.view.isSaving = false;
                } else
                    notifications.yell('error', gt('An error occurred during sending the encrypted email.') + ' ' + r);
            });
            
        });
        baton.view.stopListening(baton.view.model, 'needsync', baton.view.syncMail);
        baton.view.model.dirty(false);
        if (!draft) baton.view.app.quit();
    }
    
    
    function spinRefresh() {
        var spinner = $('#io-ox-refresh-icon').find('i');
        timer = window.setInterval(function () {
            if (spinner.hasClass('fa-spin-paused')) {
                spinner.removeClass('fa-spin-paused').addClass('fa-spin');
            }
        }, 2000);
        spinner.removeClass('fa-spin-paused').addClass('fa-spin');
    }
    
    function stopSpinRefresh() {
        window.clearInterval(timer);
        $('#io-ox-refresh-icon').find('i').removeClass('fa-spin').addClass('fa-spin-paused');
    }
    
    function getReturnId(baton) {
        try {
            var headers = baton.model.get('headers');
            return (headers['X-OxGuard-ReplyId']);
        } catch (ex) {
            console.log('OG: error parsing headers');
            console.log(ex);
            return (undefined);
        }
    }
    
    function saveDraftId(baton) {
        var header = baton.model.get('headers');
        header['X-OxGuard-Draft'] = baton.app.refId;
        baton.model.set('headers', header);
    }
    
    
    function checkPasswordNeeded(baton) {
        if (baton.model.get('PGPSignature')) return(2);
        // CHECK FOR PGP FILE ATTACHMENTS
        return (0);
    }
    
    
    function getPassword(baton, auth, data, level) {
        require (['io.ox/core/tk/dialogs'], function (dialogs) {
            var dialog = new dialogs.CreateDialog({ width: 490, center: true,  enter : 'ok'});
            dialog.header($('<h4>').text(gt('Password Needed')));
            var help = '';
            if (level === 2) { // PGP signing email
                help = $('<p>' + gt('Please enter your Guard password for signing') + '</p>');
            } else 
                help = $('<p>' + gt('At least one of your attachments are encrypted.  Please provide your Guard Security Password to decrypt before sending') + '</p>');
            var pass = $('<input name="password" id="password" type="password"/>');
            dialog.getBody().append(help).append(pass);
            if (level == 1) { // Attachment with extra password
                var extrapasshelp = $('<p style="padding-top:5px;">' + gt('Extra Password') + '</p>');
                var extra = $('<input name="extrapass" id="atextrapass" type="password"/>');
                dialog.getBody().append(extrapasshelp).append(extra);
            }
            dialog.addPrimaryButton('ok', gt('OK'), 'ok')
            .on('ok', function () {
                var extrap = $('#atextrapass').val();
                var pass = {
                        password : $('#password').val(),
                        extrapass : (extrap === undefined) ? '' : extrap,
                    };
                send(baton, auth, data, pass);
                
            });
            dialog.show(function () {
                pass.focus();
            });
        });
        
    }
    
    // Display options for guest account header (introduction)
    function guestOptions(baton, auth) {

        require (['io.ox/core/tk/dialogs'], function (dialogs) {
            
            var dialog = new dialogs.CreateDialog({ width: 490, center: true,  enter : 'ok'});
            dialog.header($('<h4>').text(gt('Recipient Options')));
            var help = $('<p>' + gt('At least one of your recipients uses a different webmail system.  They will receive an email with a link for them to read this message.  If you like, you can add a personalized message that will appear at the top of this message.  This will help identify the message as coming from you.') + '</p>');
            var messageLabel = $('<label class="oxguard_message_label">' + gt('Message:') + '</label>');
            var message = $('<textarea type="text" id="message" class="og_guest_message"></textarea>');
            var languageLabel = $('<br/><label>' + gt('Automated Email Language') + '</label>');
            var langselect = $('<select class="og_language"></select>');
            // Populate language selector with languages avail
            var languages = window.oxguarddata.lang;
            $.each(languages, function (c, n) {
                langselect.append('<option value="' + c + '">' + n + '</option>');
            });
            dialog.getBody().append(help).append(messageLabel).append(message).append(languageLabel.append(langselect));
            dialog.addPrimaryButton('ok', gt('OK'), 'ok')
            .addButton('cancel', gt('Cancel'), 'cancel')
            .on('ok', function () {
                var data = {
                        message : message.val(),
                        language: langselect.val()
                    };
                
                // Check if requires PIN to be displayed
                require(['oxguard/mail/pin', 'io.ox/core/capabilities'], function (pin, capabilities) {
                    if (capabilities.has('guard-pin')) {
                        pin.showpin()
                        .done(function(e) {
                            data.pin = e;
                            send(baton, auth, data);
                        })
                        .fail(function() {
                            send(baton, auth, data);
                        });
                    } else send(baton, auth, data);
                    
                });   
            });
            dialog.show(function () {
                var urlLang = ox.language;
                if ($('.og_language option[value="' + urlLang + '"]').length > 0) {
                    langselect.val(urlLang);
                } else {
                    langselect.val(urlLang.substring(0, 2));
                }
            });
        });
    }
    
    return {
        sendMail: function sendmail(baton, auth) {
            send(baton, auth, null, undefined, false);
        },
        draftMail: function draftmail(baton, auth) {
            send (baton, auth, null, undefined, true);
        }
    };
    
                              
});