/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2011
 */

define('oxguard/mail/sender', ['io.ox/core/extensions', 'gettext!oxguard/mail/sender'],
                              function (ext, gt) {
    
    var timer = null;
    
    ////////////   SENDING
    
    // Get files for upload
    function getFiles(baton, mail, authorization, guestdata, pass, draft) {
        var view = baton.app.getView();

        var secure_reply = getReturnId(view.form);
        // If needed, add passwords to json data
        if (pass !== undefined) {
            mail.attachpassword = pass.password;
            mail.attachextra = pass.extrapass;
        }
        if (window.FormData === undefined) {
            alert (gt('Unsupported browser'));
        } else {
            var def = $.Deferred();

            var formData = new FormData();
            formData.append('json_0', JSON.stringify(mail));
            if (guestdata !== null) {
                formData.append('guest_message', guestdata.message);
                if (guestdata.pin !== undefined) {
                    formData.append('pin', guestdata.pin);
                }
                
            }

            for (var l = 0; l < mail.files.length; l++) {
                formData.append('file' + l, mail.files[l]);
            }
            var xhr = new XMLHttpRequest();
            var dest = '/oxguard/mail?action=emailform';
            var url = ox.apiRoot + dest;
            var params = '&session=' + ox.session + '&userid=' + ox.user_id + '&cid=' + ox.context_id +
                '&sentfolder=' + encodeURIComponent(window.oxguarddata.sentfolder) +
                '&draftfolder=' + encodeURIComponent(window.oxguarddata.draftfolder) +
                '&draft=' + (draft ? 'true' : 'false') +
                '&lang=' + ox.language + (guestdata === null ? '' : ('&guestlanguage=' + guestdata.language)) +
                ((secure_reply === undefined) ? '' : ('&sr=' + secure_reply));

            if (view.form.find('#PGPSign').is(':checked') && !draft) {
                params = params + '&pgpsign=true';
            }
            if (view.form.find('#PGPinline').is(':checked')) {
                params = params + '&pgpinline=true';
            }

            xhr.open('POST', url + params);
            
            xhr.onreadystatechange = function () {
                if (xhr.readyState === 4) {
                    if (xhr.status !== 200) {
                        require (['oxguard/core/log'], function (log) {
                            log.fail(xhr, url, params, mail);
                        });
                        if (xhr.status === 503) {
                            def.reject(gt('Service Unavailable'));
                        }
                        def.reject(xhr.responseText);
                    } else {
                        def.resolve(xhr.response);
                    }
                }
            };
            xhr.send(formData);
            return (def);
        }
        
    }
    
   
    // Main send function
    function send(baton, auth, guestdata, pass, draft)
    {
        var win = baton.app.getWindow();
        var def = $.Deferred();
        var view = baton.app.getView().form;
        require (['oxguard/mail/oxguard_mail_write_core'], function (write_core) {
            write_core.saveToHeader(view);  // Make sure headers updated
        });
        var mail = baton.app.getMail(); // Get the email
        if (!draft && (mail.data.to.length + mail.data.cc.length + mail.data.bcc.length) === 0) {
            require (['io.ox/core/notifications'], function (notifications) {
                notifications.yell('error', gt('Mail has no recipients'));
            });
            def.reject('no recipients');
            return (def);
        }
        var type = 0;
        if (draft) {
            if (baton.app.refId !== undefined) {
                type = 3;
                mail.data.msgref = baton.app.refId;
            } else type = 4;
        }
        mail.data.sendtype = type;
        if (!draft)
            win.busy();
        if (($('#newguest').length > 0) && (guestdata === null) && !draft) {
            guestOptions(baton, auth);
            return (def);
        }

        if (!draft) {
            if (pass === undefined) {
                var encrAttach = checkPasswordNeeded(view, mail);
                if (encrAttach > 0) {
                    if (getPassword(baton, auth, guestdata, encrAttach)) {
                        return (def);
                    } else {
                        def.reject('cancel');
                        return def;
                    }
                }
                
            }
            win.preQuit();
        }
        spinRefresh();
        getFiles(baton, mail, auth, guestdata, pass, draft)
        .done(function (result) {
            baton.app.dirty(false);
            if (draft) {
                require (['io.ox/core/notifications'], function (notifications) {
                    notifications.yell('success', gt('Message was saved'));
                    def.resolve('ok');
                });
                try {
                    var dat = $.parseJSON(result);
                    baton.app.refId = dat.data.folder_id + '/' + dat.data.id;
                    saveDraftId(baton);
                } catch (e) {
                    console.log(result);
                    console.log(e);
                }
                baton.app.isSaving = false;
            } else {
                baton.app.quit();
            }
            stopSpinRefresh();
        })
        .fail(function (r) {
            stopSpinRefresh();
            def.reject('error');
            win.idle().show();
            require (['io.ox/core/notifications'], function (notifications) {
                if (draft) {
                    notifications.yell('error', gt('An error occurred saving the email.') + ' ' + r);
                    baton.app.isSaving = false;
                } else {
                    console.log(r);
                    if (r === 'Bad password') {
                        notifications.yell('error', gt('Bad Password'));
                    } else
                        notifications.yell('error', gt('An error occurred during sending the encrypted email.') + ' ' + r);
                }
                    
            });
            
        });
        if (draft) {
            return(def);
        }
    }
    
    
    function spinRefresh() {
        var spinner = $('#io-ox-refresh-icon').find('i');
        timer = window.setInterval(function () {
            if (spinner.hasClass('fa-spin-paused')) {
                spinner.removeClass('fa-spin-paused').addClass('fa-spin');
            }
        }, 2000);
        spinner.removeClass('fa-spin-paused').addClass('fa-spin');
    }
    
    function stopSpinRefresh() {
        window.clearInterval(timer);
        $('#io-ox-refresh-icon').find('i').removeClass('fa-spin').addClass('fa-spin-paused');
    }
    
    function getReturnId(view) {
        try {
            var headers = JSON.parse(view.find('input[name="headers"]').val());
            return (headers['X-OxGuard-ReplyId']);
        } catch (ex) {
            console.log('OG: error parsing headers');
            console.log(ex);
            return (undefined);
        }
    }
    
    function saveDraftId(baton) {
        var emailheader = baton.app.getView().form.find('[name="headers"]');
        var headerstring = emailheader.val();
        var header = {};
        if (headerstring !== '') header = JSON.parse(headerstring);
        header['X-OxGuard-Draft'] = baton.app.refId;
        emailheader.val(JSON.stringify(header));
    }
    
    
    function checkPasswordNeeded(view, mail) {
        if (view.find('#PGPGuardOn').is(':checked')) {  // Check if PGP with signing, if so, need password
            if (view.find('#PGPSign').is(':checked')) return (2);
        }
        for (var i = 0; i < mail.data.infostore_ids.length; i++) {
            if (mail.data.infostore_ids[i].filename.indexOf('.pgp') > 0) {
                return (1);
            }
        }
        for (var j = 0; j < mail.files.length; j++) {
            if (mail.files[j].name.indexOf('.pgp') > 0) {
                return (1);
            }
        }
        return (0);
    }
    
    
    function getPassword(baton, auth, data, level) {
        require (['io.ox/core/tk/dialogs'], function (dialogs) {
            var dialog = new dialogs.CreateDialog({ width: 490, center: true,  enter : 'ok'});
            dialog.header($('<h4>').text(gt('Password Needed')));
            var help = '';
            if (level === 2) { // PGP signing email
                help = $('<p>' + gt('Please enter your Guard password for signing') + '</p>');
            } else 
                help = $('<p>' + gt('At least one of your attachments are encrypted.  Please provide your Guard Security Password to decrypt before sending') + '</p>');
            var pass = $('<input name="password" id="password" type="password"/>');
            dialog.getBody().append(help).append(pass);
            dialog.addPrimaryButton('ok', gt('OK'), 'ok')
            .addButton('cancel', gt('Cancel', 'cancel'))
            .on('ok', function () {
                var extrap = $('#atextrapass').val();
                var pass = {
                        password : $('#password').val(),
                        extrapass : (extrap === undefined) ? '' : extrap,
                    };
                send(baton, auth, data, pass);
                return(true);
                
            })
            .on('cancel', function() {
                baton.app.getWindow().idle();
                stopSpinRefresh();
                return (false);
            });
            dialog.show(function () {
                pass.focus();
            });
        });
        
    }
    
    // Display options for guest account header (introduction)
    function guestOptions(baton, auth) {

        require (['io.ox/core/tk/dialogs'], function (dialogs) {
            
            var dialog = new dialogs.CreateDialog({ width: 490, center: true,  enter : 'ok'});
            dialog.header($('<h4>').text(gt('Recipient Options')));
            var help = $('<p>' + gt('At least one of your recipients uses a different webmail system.  They will receive an email with a link for them to read this message.  If you like, you can add a personalized message that will appear at the top of this message.  This will help identify the message as coming from you.') + '</p>');
            var messageLabel = $('<label class="oxguard_message_label">' + gt('Message:') + '</label>');
            var message = $('<textarea type="text" id="message" class="og_guest_message"></textarea>');
            var languageLabel = $('<br/><label>' + gt('Automated Email Language') + '</label>');
            var langselect = $('<select class="og_language"></select>');
            // Populate language selector with languages avail
            var languages = window.oxguarddata.lang;
            $.each(languages, function (c, n) {
                langselect.append('<option value="' + c + '">' + n + '</option>');
            });
            dialog.getBody().append(help).append(messageLabel).append(message).append(languageLabel.append(langselect));
            dialog.addPrimaryButton('ok', gt('OK'), 'ok')
            .addButton('cancel', gt('Cancel'), 'cancel')
            .on('ok', function () {
                var data = {
                        message : message.val(),
                        language: langselect.val()
                    };
                
                // Check if requires PIN to be displayed
                require(['oxguard/mail/pin', 'io.ox/core/capabilities'], function (pin, capabilities) {
                    if (capabilities.has('guard-pin')) {
                        pin.showpin()
                        .done(function(e) {
                            data.pin = e;
                            send(baton, auth, data);
                        })
                        .fail(function() {
                            send(baton, auth, data);
                        });
                    } else send(baton, auth, data);
                    
                });   
            })
            .on('cancel', function () {
                baton.app.getWindow().idle();
                return;
            });
            dialog.show(function () {
                var urlLang = ox.language;
                if ($('.og_language option[value="' + urlLang + '"]').length > 0) {
                    langselect.val(urlLang);
                } else {
                    langselect.val(urlLang.substring(0, 2));
                }
            });
        });
    }
    
    return {
        sendMail: function sendmail(baton, auth) {
            send(baton, auth, null, undefined, false);
        },
        draftMail: function draftmail(baton, auth) {
            return(send (baton, auth, null, undefined, true));
        }
    };
    
                              
});
