/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.server;

import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.mail.*;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.InternetHeaders;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

import net.htmlparser.jericho.Source;

import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.mailcreator.Attachment;
import com.openexchange.guard.translate.GetText;

public class Sender {
	
	private static Logger logger = LoggerFactory.getLogger(Sender.class);

    static private String username = Config.SMTPUsername;

    static private String password = Config.SMTPPassword;

    public boolean send (JsonObject email, ArrayList<Attachment> attachments) {
    	String messageId = "<OxGuard-" + EncrLib.getUUID() + ">";
    	return (send (email, attachments, "", Config.default_lang, messageId));
    }
    
    public boolean send(JsonObject email, ArrayList<Attachment> attachments, String sender, String lang, String messageId) {
        Properties prop = new Properties();
        boolean authenticate = true;
        if (username.length() < 2) {
        	authenticate = false;
        	prop.put("mail.smtp.auth", "false");
        } else 
        	prop.put("mail.smtp.auth", "true");
        prop.put("mail.smtp.host", Config.SMTPServer);
        prop.put("mail.smtp.port", Integer.toString(Config.SMTPPort));
        if (Config.useStartTls) prop.put("mail.smtp.starttls.enable", "true");

        try {
            email = email.get("data").getAsJsonObject();
            String to = email.get("to").getAsString();

            Address[] addrTo = new Address[1];
            addrTo[0] = new InternetAddress(to);  // right now, only one recipient per email
            //     String[] recips = to.split(">,");
        //    for (int i = 0; i < recips.length; i++) {
         //       addrTo[i] = new InternetAddress(recips[i]);
         //   }
            String from =  email.get("from").getAsString();
            InternetAddress fromaddr = new InternetAddress(from);
            if (!sender.equals("")) {
            	fromaddr.setPersonal(GetText.gettext("Sent on behalf of %s", lang, ((fromaddr.getPersonal() == null) ? "" : (fromaddr.getPersonal().replace("\"", "") + " - ")) + fromaddr.getAddress().replace("<", "").replace(">", "")));
            	fromaddr.setAddress(sender);
            }
            String subject = email.get("subject").getAsString();
            String htmlcontent = null;
            String plaincontent = null;
            JsonArray at = email.getAsJsonArray("attachments").getAsJsonArray();
            for (int i = 0; i < at.size(); i++) {
            	JsonObject item = at.get(i).getAsJsonObject();
            	String type = item.get("content_type").getAsString();
            	if (type.contains("html") || type.contains("ALTERN")) {
            		htmlcontent = item.get("content").getAsString();
            	}
            	if (type.contains("text/plain")) {
            		plaincontent = item.get("content").getAsString();
            	}
            }
            
            Session session = null;
            if (authenticate) {
	            session = Session.getInstance(prop, new Authenticator() {
	
	                protected PasswordAuthentication getPasswordAuthentication() {
	                    return new PasswordAuthentication(username, password);
	                }
	            });
            } else {
            	session = Session.getInstance(prop);
            }

            Message message = new MimeMessage(session);
            if (email.get("headers") != null) {
                JsonObject headers = email.get("headers").getAsJsonObject();
                for (Map.Entry<String, JsonElement> h : headers.entrySet()) {
                	message.setHeader(h.getKey(), checkCR(h.getValue().getAsString()));
                }
            }
            InternetAddress replyaddr = new InternetAddress(from);
            message.setHeader("Reply-To", replyaddr.toString());
            message.setRecipients(Message.RecipientType.TO, addrTo);
            ;
            message.setSubject(subject);
            MimeBodyPart multipart = new MimeBodyPart();  // Main message wrapper
            // Create alternative wrapped message
            Multipart alternate = new MimeMultipart("alternative");
            BodyPart pt = new MimeBodyPart();
            if (plaincontent != null) { // If plaintext exists, use
            	pt.setContent(plaincontent, "text/plain; charset=utf-8");
            	alternate.addBodyPart(pt);
            } else {  // else create from html
            	if (htmlcontent != null) {
            		StringReader sr = new StringReader(htmlcontent);
            		Source src = new Source (sr);
            		pt.setContent(src.getRenderer().toString(), "text/plain; charset=utf-8");
            		alternate.addBodyPart(pt);
            		sr.close();
            	}
            }
            MimeMultipart content = new MimeMultipart("mixed");
        	ArrayList <MimeBodyPart> cids = new ArrayList <MimeBodyPart> ();
            if (htmlcontent != null) {
            	try {
            		htmlcontent = checkInlineImages(htmlcontent, cids);
            	} catch (Exception ex) {
            		logger.error("Problem checking for inline mail images ", ex);
            	}
            	if (cids.size() > 0) {
            		MimeMultipart htmlpart = new MimeMultipart("related");
            		BodyPart main = new MimeBodyPart();
		            main.setContent(htmlcontent, "text/html; charset=utf-8");
            		htmlpart.addBodyPart(main);
                    for (int i = 0; i < cids.size(); i++) {
                    	htmlpart.addBodyPart(cids.get(i));
                    }
                    MimeBodyPart htmlbody = new MimeBodyPart();
                    htmlbody.setContent(htmlpart);
                    alternate.addBodyPart(htmlbody);
            	} else {
		            BodyPart main = new MimeBodyPart();
		            main.setContent(htmlcontent, "text/html; charset=utf-8");
		            alternate.addBodyPart(main);
            	}
            }
            // add the alternate to multipart
            multipart.setContent(alternate);
            // email multipart
            
            // add the wrapped alternate email
            content.addBodyPart(multipart);
            
            // add attachments

            if (attachments != null) {
                for (int i = 0; i < attachments.size(); i++) {
                    if (attachments.get(i) != null) {
                        Attachment attachment = attachments.get(i);
                        InternetHeaders header = new InternetHeaders();
                        BodyPart att = new MimeBodyPart();
                        att.setText(new String(attachment.encrContent));
                        att.setHeader("Content-Type", checkCR(attachment.type));
                        header.addHeader("Content-Type", checkCR(attachment.type));
                        header.addHeader("Content-Transfer-Encoding", "base64");
                        att.setFileName(checkCR(attachment.filename));
                        content.addBodyPart(att);
                    }
                }
            }
            message.setContent(content);
            message.setFrom(fromaddr);
            message.setSentDate(new Date());
            message.saveChanges();
            message.setHeader("Message-ID", messageId);
            Transport transport = session.getTransport("smtp");
            transport.connect(Config.SMTPServer, Config.SMTPPort, Config.SMTPUsername, Config.SMTPPassword);
            transport.sendMessage(message, message.getAllRecipients());
            transport.close();
            // Transport.send(message);;

        } catch (Exception ex) {
            logger.error("Error sending email to " + email.get("to").getAsString(), ex);
            return (false);

        }
        return (true);
    }
    
    private String checkCR (String data) {
    	try {
    		return (data.replace("\r", "").replace("\n", ""));
    	} catch (Exception ex) {
    		return(data);
    	}
    }
    
    /**
     * Check for inline data images and convert them to cid images
     * @param html
     * @param content
     * @return
     * @throws MessagingException
     * @throws UnsupportedEncodingException 
     */
    private String checkInlineImages (String html, ArrayList <MimeBodyPart> cids) throws MessagingException, UnsupportedEncodingException {
    	Pattern p = Pattern.compile("<img[^>]*");
    	Matcher m = p.matcher(html);
    	int count = 0;
    	while (m.find()) {
    		String image = m.group();
    		if (image.contains("data:image")) {
    			Pattern p2 = Pattern.compile("src[ ]*=[ ]*\"[^\"]*");
    			Matcher m2 = p2.matcher(image);
    			if (m2.find()) {
    				String data = m2.group();
    				int i = data.indexOf("image");
    				int j = data.indexOf(",");
    				String header = data.substring(i,j);
    				String imagedata = data.substring(j+1).replace("%3D", "=");
    				String[] headers = header.split(";");
    				if (headers.length > 1) {
    					if (headers[1].toLowerCase().trim().equals("base64")) {
    						String cid = EncrLib.getUUID() + "@guard";
    						String replacement = "src=\"cid:" + cid;
    						html = html.replace(data, replacement);
    						String suffix = "";
    						switch (headers[0]) {
	    						case "image/jpeg":
	    							suffix = ".jpg";
	    							break;
	    						case "image/jpg":
	    							suffix = ".jpg";
	    							break;
	    						case "image/png":
	    							suffix = ".png";
	    							break;
	    						case "images/gif":
	    							suffix = ".gif";
	    							break;
    						}
    						InternetHeaders h = new InternetHeaders();
    						h.addHeader("Content-Type", checkCR(headers[0]));
    						h.addHeader("Content-Transfer-Encoding", "base64");
                            MimeBodyPart att = new MimeBodyPart(h, imagedata.getBytes("UTF-8"));
                            att.setDisposition(MimeBodyPart.INLINE);
                            att.setContentID("<" + cid + ">");
                            att.setFileName("att"+ (++count) + suffix);
                            cids.add(att);
    					}
    				}
    			}
    		}
    	}
    	return (html);
    }

}
