/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.server;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.RetrievedItem;
import com.openexchange.guard.encr.Crypto;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.EncryptedObj;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.GuardMissingParameter;
import com.openexchange.guard.file.Fileshare;
import com.openexchange.guard.mailcreator.Attachment;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.server.Settings.permissions;
import com.openexchange.guard.translate.GetText;
import com.openexchange.guard.util.Core;

public class FileHandler {

	private static Logger logger = LoggerFactory.getLogger(FileHandler.class);

    /**
     * For downloading Form type files from client
     * 
     * @param request
     * @param response
     * @throws IOException
     * @throws GuardMissingParameter 
     */
    public void downloadFormFromClient(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie, boolean update) throws IOException, GuardMissingParameter {
        FileItemFactory fif = new DiskFileItemFactory();
        ServletFileUpload sfu = new ServletFileUpload(fif);
        String jsondata = "";
        String itemID = EncrLib.getUUID(); // One itemID for email and it's attachments
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        int folderid = Core.getIntParameter(request, "folder", update); 
        int version = Core.getIntParameter(request, "version", update); // Version of file for updating
        int id = Core.getIntParameter(request, "id", update); // Id of file for updating
        String sessionID = Core.getStringParameter(request, "session", true);
        String auth = Core.getStringParameter(request, "auth");

        if (!Auth.checkPermission(cookie, request, com.openexchange.guard.server.Settings.permissions.FILE)) {
            logger.error("not authorized");
            Core.sendFail(response, "not authorized");
            return;
        }
        Crypto crypt = new Crypto();
        try {
            List files = sfu.parseRequest(request);
            Iterator iterator = files.iterator();
            ArrayList<Attachment> attachments = new ArrayList<Attachment>();// Get list of all attachments
            while (iterator.hasNext()) {
                FileItem fi = (FileItem) iterator.next();
                String fname = fi.getFieldName();
                if (fname.startsWith("json")) {
                    jsondata = fi.getString();
                } else {
                    if (fi.isFormField()) {
                        String value = fi.getString();
                        logger.debug(value);
                    } else {
                        String name = fi.getName();
                        if (name != null) {// Make sure the name of the file exists
                            if (name.trim() != "") {
                                if (name.lastIndexOf("\\") > -1) {
                                    name = name.substring(name.lastIndexOf("\\"));
                                }
                                if (name.lastIndexOf("/") > -1) {
                                    name = name.substring(name.lastIndexOf("/"));
                                }
                                ByteArrayOutputStream out = new ByteArrayOutputStream();
                                InputStream input = fi.getInputStream();
                                IOUtils.copy(input, out);
                                Attachment attach = new Attachment();
                                attach.content = out.toByteArray(); // Populate attachment data
                                String filename = fi.getName();
                                filename = filename + ".grd";
                                attach.filename = filename;
                                attach.type = "grd/" + fi.getContentType().replace("/", "--");
                                attachments.add(attach); // Add to our array of attachments
                                out.close();
                                input.close();
                            }
                        }
                    }
                }
            }
            if (jsondata != "") {
                if (update) {// If updating file, then will be first item, get key, encrypt, and update
                    RetrievedItem parentFile = getUpdateFile(
                        URLDecoder.decode(auth, "UTF-8"),
                        id,
                        folderid,
                        version,
                        request,
                        cookie,
                        sessionID);
                    if (parentFile == null) {
                        sendFail(response, "Error retrieving original file data");
                    }
                    Attachment attach = attachments.get(0);
                    attach.encrContent = crypt.encrItem(
                        userid,
                        cid,
                        null,
                        Base64.encodeBase64String(attach.content),
                        0,
                        2,
                        "",
                        parentFile.itemID,
                        parentFile.contentKey,
                        false,
                        attach.filename);
                    Api ap = new com.openexchange.guard.ox.Api(cookie, sessionID, request.getHeader("User-Agent"));
                    String oxResp = ap.updateFile(attach, jsondata, id, folderid);
                    if (oxResp.toLowerCase().contains("error")) {
                        sendFail(response, oxResp);

                    } else {
                        sendOK(response, oxResp);
                    }
                } else { // If new file, may be multiple. Process each
                    for (int i = 0; i < attachments.size(); i++) {
                        Attachment attach = attachments.get(i);
                        if (attach.encrContent == null) {// Make sure this isn't the email
                            attach.encrContent = crypt.encrItem(userid, cid, null, Base64.encodeBase64String(attach.content), 0, 2, "");
                            attachments.set(i, attach);
                        }
                        if (attach.encrContent == null) {
                            sendFail(response, "Fail");
                            return;
                        }
                        com.openexchange.guard.ox.Api ap = new com.openexchange.guard.ox.Api(
                            cookie,
                            sessionID,
                            request.getHeader("User-Agent"));
                        String params = "files?action=new&filename&session=" + sessionID;

                        String attid = Crypto.getID(attach.encrContent);
                        Gson gson = new GsonBuilder().create();
                        JsonObject json = gson.fromJson(jsondata, JsonObject.class);
                        JsonObject meta = new JsonObject();
                        meta.addProperty("OGID", attid);
                        meta.addProperty("Encrypted", true);
                        meta.addProperty("OwnerCid", cid);
                        meta.addProperty("OwnerId", userid);
                        json.add("meta", meta);
                        String data = ap.filePost(gson.toJson(json), params, attachments);
                        logger.debug(data);
                        // String data =
                        // "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\" \"http://www.w3.org/TR/html4/strict.dtd\"><html><head><META http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\"><script type=\"text/javascript\">(parent.callback_new || window.opener && window.opener.callback_new)({\"timestamp\":1383229034723,\"data\":\"9\"})</script></head></html>";
                        sendOK(response, data);// If OK, send response
                        return;
                    }
                }

            } else {
                sendOK(response, "Fail");
                return;
            }

        } catch (Exception ex) {
            sendOK(response, "Fail");
            logger.error("Error while downloading item " + itemID, ex);
        }
        sendOK(response, "Fail");
    }

    private String getFileJson(String itemId, int folder, String description, int id, int cid) {
        Gson gson = new GsonBuilder().create();
        JsonObject json = new JsonObject();
        JsonObject meta = new JsonObject();
        meta.addProperty("OGID", itemId);
        meta.addProperty("Encrypted", true);
        meta.addProperty("OwnerCid", cid);
        meta.addProperty("OwnerId", id);
        json.add("folder_id", new JsonPrimitive(folder));
        json.addProperty("description", description);
        json.add("meta", meta);
        String data = gson.toJson(json);
        return (data);
    }

    /**
     * Decode and send encrypted file
     * 
     * @param request
     * @param response
     * @param cookie
     * @param send
     * @return
     * @throws IOException
     * @throws GuardMissingParameter 
     */
    public Attachment getFile(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie, boolean send) throws IOException, GuardMissingParameter {
        int itemid = Core.getIntParameter(request, "id", true);
        int folder = Core.getIntParameter(request, "folder", true);
        int version = request.getParameter("version") == null ? -1 : Integer.parseInt(request.getParameter("version"));
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        String lang = Core.getStringParameter(request, "language");
        String auth = Core.getStringParameter(request, "auth");
        String filename = Core.getStringParameter(request, "filename", true);
        String sessionID = Core.getStringParameter(request, "session", true);
        String encrExtra = Core.getStringParameter(request, "encrextra");
        UserData data;
        if (auth.length() < 20) {
            String password = Core.getStringParameter(request, "password", true);
            data = new UserData();
            data.encr_password = password;
            data.userid = userid;
        } else {
            data = new UserData(auth, cookie);
        }
        if (data.userid == -1) {
            sendFail(response, GetText.gettext("Unable to decode file, bad authorization", lang));
            return (null);
        }
        Crypto crypto = new Crypto();
        try {
            Api ap = new Api(cookie, sessionID, request.getHeader("User-Agent"));
            EncryptedObj obj = ap.getInfoStore(itemid, folder, version);
            if (obj.ItemID == null) {
                logger.debug("Not encrypted file or file not found");
                sendFail(response, GetText.gettext("Not encrypted file or file not found", lang));
                return (null);
            }
            String file = crypto.decodeEncrObj(userid, cid, obj, data.encr_password);
            boolean sizematch = (file.length() == obj.length) || (obj.length <= 0);
            if (file != null) {// Data was stored base64, so we need to decode when returned
                if (file.startsWith("EXTRA")) { // Check for extra password
                    EncrLib encr = new EncrLib();
                    String extra = encr.decryptAES(encrExtra, cookie.JSESSIONID, "epass", Config.AESKeyLen);
                    file = encr.decryptAES(file.substring(5), extra, obj.ItemID, obj.keyLength);
                    if (file.length() < 2) {
                        sendOK(response, GetText.gettext("Incorrect password", lang));
                        return (null);
                    }
                }
                if (file.startsWith("{\"status\"")) { // If our email has expired, then just send that
                    sendOK(response, GetText.gettext("Item has expired", lang));
                    return (null);
                }
                if (file.startsWith("{\"Exceeded\" : ")) {
                    sendOK(response, GetText.gettext("Exceeded attempts, try again later", lang));
                    return (null);
                }
                obj.content_type = obj.content_type.replace("grd/", "").replace("--","/");
                if (send) {
                    if ((request.getParameter("download") != null) && (request.getParameter("download")).contains("true")) {
                        saveToDownload(
                            response,
                            obj.content_type,
                            Base64.decodeBase64(file),
                            filename.replace(".grd2", "").replace(".grd", ""));
                    } else {
                        sendOK(response, obj.content_type, Base64.decodeBase64(file));
                    }
                    // saveAttach(response, attdata.content_type, Base64.decode(attach), filename);
                    return (null);
                }
                Attachment decoded = new Attachment();
                decoded.sizeMatch = sizematch;
                decoded.filename = filename;
                decoded.encrContent = Base64.decodeBase64(file);
                decoded.type = obj.content_type;
                return (decoded);

            }
            sendOK(response, GetText.gettext("Incorrect password", lang));
        } catch (Exception e) {
            logger.error("Error while getting file " + folder + "/" + itemid + " for user " + userid, e);
            sendFail(response, "Unable to decode");
        }
        return (null);
    }

    /**
     * Send answer to response stream
     * 
     * @param response
     * @param answer String to put in HTML
     */
    private void sendOK(HttpServletResponse response, String answer) {
        response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            response.getWriter().println(answer);
        } catch (IOException e) {
        	logger.error("Error while writing 'OK' answer to response", e);
        }
    }

    /**
     * Send answer to response stream
     * 
     * @param response
     * @param answer String to put in HTML
     */
    private void sendOK(HttpServletResponse response, String type, byte[] answer) {
        response.setContentType(type);

        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            OutputStream out = response.getOutputStream();
            out.write(answer);
            out.close();
        } catch (IOException e) {
        	logger.error("Error while writing answer to output stream", e);
        }
    }

    private void sendFail(HttpServletResponse response, String answer) {
        response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_NOT_ACCEPTABLE);
        try {
            response.getWriter().println(answer);
        } catch (IOException e) {
        	logger.error("Error while writing 'fail' answer to response", e);
        }

    }

    private void saveToDownload(HttpServletResponse response, String type, byte[] answer, String filename) {
        response.setContentType(type);
        response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            OutputStream out = response.getOutputStream();
            out.write(answer);
            out.close();
        } catch (IOException e) {
        	logger.error("Error while writing to output stream", e);
        }
    }

    public void encryptFile(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws GuardMissingParameter {

            int id = Core.getIntParameter(request, "id", true);
            int folder = Core.getIntParameter(request, "folder", true);
            int version = Core.getIntParameter(request, "version", true);
            int userid = Core.getIntParameter(request, "userid", true);
            int cid = Core.getIntParameter(request, "cid", true);
        try {
            String sessionID = Core.getStringParameter(request, "session", true);

            if (!Auth.checkPermission(cookie, request, permissions.MAIL)) {
                logger.error("not authorized");
                Core.sendFail(response, "not authorized");
                return;
            }

            Api ap = new Api(cookie, request);
            // Get original file json data
            String data = ap.getInfoItemJson(id, folder, version);
            Gson gson = new Gson();
            JsonObject json = gson.fromJson(data, JsonObject.class);
            JsonObject jdata = json.get("data").getAsJsonObject();

            // Create new file and popluate with old json data
            Attachment encrfile = new Attachment();
            encrfile.filename = jdata.get("filename").getAsString();
            encrfile.type = jdata.get("file_mimetype").getAsString();
            version = jdata.get("version").getAsInt();
            String description = (jdata.get("description") == null) ? "" : jdata.get("description").getAsString();
            // Load the file and encrypt
            byte[] filedata = ap.getPlainTextInfoStore(id, folder, version);
            if (filedata == null) {
                sendFail(response, "Unable to get file");
                return;
            }
            Crypto crypt = new Crypto();
            String itemID = EncrLib.getUUID();
            String newjson = getFileJson(itemID, folder, description, userid, cid);
            encrfile.encrContent = crypt.encrItem(userid, cid, null, Base64.encodeBase64String(filedata), 0, 2, "", itemID);
            if (encrfile.encrContent == null) {
                sendFail(response, "Unable to encrypt file");
                return;
            }
            if (encrfile.encrContent.length < 50) {// Simple check to see if file really there
                sendFail(response, "Unable to encrypt file");
                return;
            }
            ArrayList<Attachment> attachments = new ArrayList<Attachment>();
            encrfile.filename = encrfile.filename + ".grd";
            encrfile.type = "grd/" + encrfile.type.replace("/", "--");
            attachments.add(encrfile);

            // Save to ox backend
            String params = "files?action=new&filename=" + URLEncoder.encode(encrfile.filename) + "&session=" + sessionID;
            String resp = ap.filePost(newjson, params, attachments);
            if (resp.contains("error")) {
                sendFail(response, "Unable to save encrypted file " + Core.getErrorDescr(resp));
                return;
            }
            if (resp.contains("timestamp")) {
                ap.deleteItem(id, folder);
                sendOK(response, "OK");
            }
        } catch (Exception ex) {
            logger.error("Error encrypting file", ex);
            sendFail(response, "Unable to encrypt");
        }

    }

    private RetrievedItem getUpdateFile(String auth, int id, int folder, int version, HttpServletRequest request, com.openexchange.guard.server.OxCookie cookie, String sessionID) {

        UserData user = new UserData(auth, cookie);
        if (user.userid < 0) {
        	return (null);
            // bad login
        }
        Api ap = new Api(cookie, request);
        EncryptedObj file = ap.getInfoStore(id, folder, version);
        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        com.openexchange.guard.database.RetrievedItem item;
        try {
            item = acc.getItemID(file.ItemID, user.userid, user.cid);
            item.itemID = file.ItemID;
            GuardKeys keys = acc.getKeys(user.userid, user.cid);

            EncrLib encryptor = new EncrLib();
            String contentkey = encryptor.decryptRSA(item.contentKey, keys.getDecodedPrivate(user.encr_password));
            item.contentKey = contentkey; // Save decoded content key into item
            if (contentkey != null) {
                return (item);
            }
        } catch (Exception ex) {
            logger.error("Error getting updated file " + folder + "/" + id + " in version " + version, ex);

        }
        return (null);
    }

    public void removeEncryption(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws GuardMissingParameter {

    	int itemid = Core.getIntParameter(request, "id", true);
    	int folder = Core.getIntParameter(request, "folder", true);
    	int version = Core.getIntParameter(request, "version", true);
        try {
            Attachment decoded = getFile(request, response, cookie, false);
            if (decoded == null) {
            	sendFail(response, "Bad Password");
            	return;
            }
            if (!decoded.sizeMatch) {
            	sendFail(response, "Bad file size match");
            	return;
            }
            decoded.filename = decoded.filename.replace(".grd2", "").replace(".grd", "");
            String sessionID = request.getParameter("session");
            Api ap = new Api(cookie, request);
            String data = ap.getInfoItemJson(itemid, folder, version);
            Gson gson = new Gson();
            JsonObject json = gson.fromJson(data, JsonObject.class);
            JsonObject jdata = json.get("data").getAsJsonObject();
            String description = (jdata.get("description") == null) ? "" : jdata.get("description").getAsString();
            JsonObject newjson = new JsonObject();
            newjson.add("folder_id", new JsonPrimitive(folder));
            newjson.addProperty("description", description);
            // Create new file and popluate with old json data
            String params = "files?action=new&filename=" + URLEncoder.encode(decoded.filename) + "&session=" + sessionID;
            ArrayList<Attachment> attachments = new ArrayList<Attachment>();
            attachments.add(decoded);
            String resp = ap.filePost(newjson.toString(), params, attachments);
            if (resp.contains("error")) {
                sendFail(response, "Unable to save decoded file " + Core.getErrorDescr(resp));
                return;
            } else {
            	//  OK.  We've decoded and sent.  Let's check the backend has a file the same size before deleting original
            	int newitemid = getID(resp);
            	if (newitemid == 0) { // Returns 0 if the id isn't there
            		sendFail(response, "Unable to save decoded file, bad file size match");
                    return;
            	}
            	// Get the new file data json, which includes file size
            	JsonObject savedFile = ap.getInfoItemJson(newitemid, folder);
            	int savedLength = savedFile.get("data").getAsJsonObject().get("file_size").getAsInt();
            	// If same size, then delete, otherwise keep original
            	if (savedLength == decoded.encrContent.length) {
            		ap.deleteItem(itemid, folder);
                	sendOK (response, "OK");
            	} else {
            		sendFail(response, "Unable to save decoded file");
                    return;
            	}
            }
        } catch (Exception ex) {
            logger.error("Error removing encryption from file " + folder + "/" +itemid + " in version " + version, ex);
            sendFail(response, "Problems decoding");
        }
    }
    
    private int getID (String resp) {
    	try {
    		int i = resp.indexOf("{");
    		int j = resp.lastIndexOf("}");
    		String jsonString = resp.substring(i, j+1);
    		JsonParser parser = new JsonParser();
            JsonObject result = (JsonObject) parser.parse(jsonString);
            return (result.get("data").getAsInt());
    	} catch (Exception ex) {
    		logger.error("Error parsing response", ex);
    		return(0);
    	}
    }

    public void getShares(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) throws GuardMissingParameter {
    	int userid = Core.getIntParameter(request, "userid", true); 
    	int cid = Core.getIntParameter(request, "cid", true);
    	String itemID = Core.getStringParameter(request, "id", true);
        try {
            Fileshare fs = new Fileshare();
            JsonArray json = fs.getShares(itemID, userid, cid);
            if (json == null) {
                com.openexchange.guard.util.Core.sendFail(response, "NOTFOUND");
            } else {
                Gson gson = new Gson();
                String json_string = gson.toJson(json);
                com.openexchange.guard.util.Core.sendOK(response, json_string);
            }
        } catch (Exception ex) {
            logger.error("Error getting shared for user " + userid + " in context " + cid + " and item " + itemID, ex);
        }
    }

    public void getfolder(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) {
        try {
            Api ap = new Api(oxCookie, request);
            String data = ap.getFolder(Integer.parseInt(request.getParameter("id")));
            if (!data.contains("error")) {
                com.openexchange.guard.util.Core.sendOK(response, data);
                return;
            }

        } catch (Exception ex) {
            logger.error("Error getting folder " + request.getParameter("id"), ex);
        }
        com.openexchange.guard.util.Core.sendFail(response, "error");
    }

    public void addFolderShare(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) {
        try {
            Api ap = new Api(oxCookie, request);
            int id = Integer.parseInt(request.getParameter("id"));
            String data = ap.getFolder(id);
            if (data.contains("error")) {
                com.openexchange.guard.util.Core.sendFail(response, "error retrieving folder perm");
                return;
            }
            String[] users = request.getParameter("users").split(",");
            JsonArray permissions = getPermissions(data);
            for (int i = 0; i < users.length; i++) {
                int user = Integer.parseInt(users[i]);
                boolean found = false;
                permloop: for (int j = 0; j < permissions.size(); j++) {
                    if (permissions.get(j).getAsJsonObject().get("entity").getAsInt() == user) {
                        // update
                        int bits = permissions.get(j).getAsJsonObject().get("bits").getAsInt();
                        bits = bits | 257;
                        JsonObject perm = permissions.get(j).getAsJsonObject();
                        perm.addProperty("bits", bits);
                        JsonArray newpermissions = new JsonArray();
                        for (int c = 0; c < permissions.size(); c++) {
                            if (c != j) {
                                newpermissions.add(permissions.get(c).getAsJsonObject());
                            }
                        }
                        newpermissions.add(perm);
                        permissions = newpermissions;
                        found = true;
                        break permloop;
                    }
                }
                if (!found) {
                    JsonObject newperm = new JsonObject();
                    newperm.addProperty("entity", user);
                    newperm.addProperty("bits", 257);
                    newperm.addProperty("group", false);
                    permissions.add(newperm);
                }
            }
            Gson gson = new Gson();
            JsonObject newdata = new JsonObject();
            newdata.add("permissions", permissions);
            String addresult = ap.updateFolder(id, gson.toJson(newdata));
            if (addresult.contains("error")) {
                logger.error("failed to add share");
                logger.error(addresult);
                com.openexchange.guard.util.Core.sendFail(response, "error adding folder perm");
                return;
            }
            com.openexchange.guard.util.Core.sendOK(response, "OK");
            return;
        } catch (Exception ex) {
            logger.error("Error adding folder share", ex);
        }
        com.openexchange.guard.util.Core.sendFail(response, "error");
    }

    private JsonArray getPermissions(String data) {
        Gson gson = new Gson();
        JsonObject json = gson.fromJson(data, JsonObject.class);
        JsonObject jdata = json.get("data").getAsJsonObject();
        JsonArray permissions = jdata.get("permissions").getAsJsonArray();
        return (permissions);
    }

    public void share(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) {
        try {
        	JsonObject jdata = Core.getJSON(request);
            String auth = request.getParameter("auth");
            String password = jdata.get("password").getAsString();
            if (password.equals("undefined"))
                password = null;
            if (auth != null) {
            	if (!auth.equals("null")) {
                    UserData data = new UserData(auth, oxCookie);
                    if (data.ox_password != null) {
                        password = data.ox_password;
                    }
            	}
            }
            if (password == null) {
                com.openexchange.guard.util.Core.sendFail(response, "Fail");
                return;
            }
            int cid = Integer.parseInt(request.getParameter("cid"));
            int userid = Integer.parseInt(request.getParameter("userid"));
            String itemID = request.getParameter("id");
            String name = jdata.get("name").getAsString();
            String email = jdata.get("email").getAsString();
            Fileshare fs = new Fileshare();
            String sharedata = fs.share(itemID, name, email, userid, cid, password);
            if (sharedata == null) {
            	com.openexchange.guard.util.Core.sendFail(response, "Bad Password");
            	logger.info("Bad password for file share");
            	return;
            }
            if (sharedata.equals("nsc")) {
            	com.openexchange.guard.util.Core.sendFail(response, "Not same context");
            	return;
            }
            com.openexchange.guard.util.Core.sendOK(response, sharedata);

        } catch (Exception ex) {
            logger.error("Error sharing", ex);
        }
    }

    public void removeShare(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) {
    	String email = request.getParameter("email");
    	String itemID = request.getParameter("id");
        try {
            Api ap = new Api(oxCookie, request);
            if (!ap.verifyLogin())
                return; // Make sure legit log into UI
            int cid = Integer.parseInt(request.getParameter("cid"));
            int userid = Integer.parseInt(request.getParameter("userid"));
            Fileshare fs = new Fileshare();
            if (fs.removeShare(itemID, userid, cid, email)) {
                com.openexchange.guard.util.Core.sendOK(response, "OK");
            } else {
                com.openexchange.guard.util.Core.sendFail(response, "Fail");
            }

        } catch (Exception ex) {
            logger.error("Error removing share for email " + email + " and item " + itemID, ex);
        }
    }

    public void addExtraPassword(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie oxCookie) throws GuardMissingParameter {
        
    	JsonObject rdata = Core.getJSON(request);
        String auth = Core.getStringParameter(request, "auth");
        String password = Core.getStringFromJson(rdata, "password");
        if (password != null) {
        	if (password.equals("undefined")) password = null;
        }

        if (auth != null) {
            UserData data = new UserData(auth, oxCookie);
            if (data.ox_password != null) {
                password = data.ox_password;
            }
        }
        if (password == null) {
            com.openexchange.guard.util.Core.sendFail(response, "Fail");
            return;
        }
        int userid = Core.getIntFromJson(rdata, "userid", true);
        int cid = Core.getIntFromJson(rdata, "cid", true);
        int id = Core.getIntFromJson(rdata, "id", true);
        int folder = Core.getIntFromJson(rdata, "folder", true);
        int version = Core.getIntFromJson(rdata, "version", true);
        String sessionID = Core.getStringParameter(request, "session", true);
        String extraPass = Core.getStringFromJson(rdata, "extrapass", true);
        if ((extraPass == null) || (extraPass.length() < 3)) {
            com.openexchange.guard.util.Core.sendFail(response, "Fail");
            return;
        }
        try {
            // Pull old encrypted file and it's json data
            Api ap = new Api(oxCookie, request);
            EncryptedObj file = ap.getInfoStore(id, folder, version);
            String data = ap.getInfoItemJson(id, folder, version);
            Gson gson = new Gson();
            JsonObject json = gson.fromJson(data, JsonObject.class);
            JsonObject jdata = json.get("data").getAsJsonObject();

            // Start creating new file
            Attachment encrfile = new Attachment();
            encrfile.filename = jdata.get("filename").getAsString().replace(".grd", ".grd2");
            encrfile.type = jdata.get("file_mimetype").getAsString();
            String description = (jdata.get("description") == null) ? "" : jdata.get("description").getAsString();
            // String newjson = "{\"folder_id\" : \"" + folder + "\",\"description\":\"" + description + "\"}";

            Crypto crypt = new Crypto();
            String decoded = crypt.decodeEncrObj(userid, cid, file, password); // Get base64 encoded decoded item
            if (decoded != null) {
                String itemID = EncrLib.getUUID();
                String newjson = getFileJson(itemID, folder, description, userid ,cid);
                encrfile.encrContent = crypt.encrItem(userid, cid, null, crypt.addExtraPass(decoded, extraPass, itemID), 0, 2, "", itemID);
                String params = "files?action=new&filename=" + URLEncoder.encode(encrfile.filename) + "&session=" + sessionID;
                ArrayList<Attachment> attachments = new ArrayList<Attachment>();
                attachments.add(encrfile);
                String resp = ap.filePost(newjson, params, attachments);
                if (!com.openexchange.guard.util.Core.checkFail(resp)) {
                    ap.deleteItem(id, folder);
                }
                com.openexchange.guard.util.Core.sendOK(response, "OK");
                return;
            }
            com.openexchange.guard.util.Core.sendFail(response, "fail");
        } catch (Exception ex) {
            logger.error("Error adding extra password", ex);
            Core.sendFail(response, "fail");
        }
    }
}
