/*
 *
 *  Copyright (C) 2012 Open-Xchange, Inc. All rights reserved.
 *
 *  This program is proprietary software and licensed to you under Open-Xchange
 *  Inc's Software License Agreement.
 *
 *  In some countries OX, OX Open-Xchange, open-xchange and OXtender as well as
 *  the corresponding Logos OX Open-Xchange and OX are registered trademarks of
 *  the Open-Xchange, Inc. group of companies.
 *
 */

package com.openexchange.documentconverter;

import java.io.File;

/**
 * {@link ICache}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public interface ICache {

    /**
     * creates a new ICacheEntry instance;
     *
     * @param hash The unique hash for the new entry.
     */
    public ICacheEntry createNewEntry(String hash);

    /**
     * Add a new entry to the cache. If there isn't enough free space or free entries, we need to remove
     * one or more of the oldest entries until the entry count and the current size of the cache+the size
     * of the new entry is smaller than the maximum size of the cache
     *
     * @param entry The new cache entry
     * @param force If set to true, add the entry in any case,
     *      regardless of any cache config restrictions (e.g. size or entry count)
     * @return true, if the entry was added to the cache
     */
    public boolean addCacheEntry(ICacheEntry entry, boolean force);

    /**
     * @param entry The cach entry that should be removed.
     * @param force If set to true, remove the entry and all associated resources in any case.
     *              If set to false, the entry is just removed from the cache if it is not locked
     */
    public void removeEntry(ICacheEntry entry, boolean force);

    /**
     * Acquires an entry with a given hash from the cache. After a successful call to <code>acquireEntry</code>
     * the returned resource needs to be released with a call to <code>releaseEntry</code> after processing!
     *
     * @param hash The hash value to search for within the cache.
     * @return The entry with the given hash value or null if no such object exists within the cache.
     */
    /**
     * @param hash The hash value to search for within the cache
     * @param args String parameters, whose usage is depending on the implementation
     * @return
     */
    public ICacheEntry acquireEntry(String hash, String... args);

    /**
     * Releases an entry and all connected resources from a previous call to <code>acquireEntry</code>.
     *
     * @param entry The entry to be released.
     */
    public void releaseEntry(ICacheEntry entry);

    /**
     * @param entry The entry that should be put at the last position of the cache.
     * @return The object with the given hash value or null if no such object exists within the cache.
     */
    public void touchEntry(ICacheEntry entry);

    /**
     * @return The working directory for the cache
     */
    public File getPersistentCacheDirectory();

    /**
     * Removes all entries within the cache and calls the clear method of entries that are not persistent
     */
    public void end();

    /**
    * Tells the cache to perform internal management tasks, e.g. update statistics, cleanup etc.
    */
    public void update();
}
