/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */



package com.openexchange.documentconverter;

import java.util.Arrays;
import java.util.function.Consumer;

//-------------
//- JobStatus -
//-------------

/**
 * {@link JobError}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public enum JobError {

    NONE(JobError.ERRORCODE_NONE, "none"),
    GENERAL(JobError.ERRORCODE_GENERAL, "filterError"),
    TIMEOUT(JobError.ERRORCODE_TIMEOUT, "timeout"),
    PASSWORD(JobError.ERRORCODE_PASSWORD, "passwordProtected"),
    NO_CONTENT(JobError.ERRORCODE_NO_CONTENT, "noContent"),
    MAX_SOURCESIZE(JobError.ERRORCODE_MAX_SOURCESIZE, "maxSourceSizeExceeded"),
    DISPOSED(JobError.ERRORCODE_DISPOSED, "engineDisposed"),
    UNSUPPORTED(JobError.ERRORCODE_UNSUPPORTED, "unsupportedType"),
    OUT_OF_MEMORY(JobError.ERRORCODE_OUT_OF_MEMORY, "outOfMemory"),
    ABORT(JobError.ERRORCODE_ABORT, "abort"),
    PDFTOOL(JobError.ERRORCODE_PDFTOOL, "pdfTool"),
    MAX_QUEUE_COUNT(JobError.ERRORCODE_MAX_QUEUE_COUNT, "maxQueueCount"),
    QUEUE_TIMEOUT(JobError.ERRORCODE_QUEUE_TIMEOUT, "queueTimeout");

    /**
     * @param errorCode
     */
    private JobError(int errorCode, String errorText) {
        m_errorCode = errorCode;
        m_errorText = errorText;
    }

    /**
     * @return The error code id of the enum
     */
    public int errorCode() {
        return m_errorCode;
    }

    /**
     * @return The textual representation of the current error
     */
    public String errorText() {
        return m_errorText;
    }

    /**
     * @param errorCode
     * @return The enum, given the error code id
     */
    public static JobError fromErrorCode(int errorCode) {
        JobError ret = NONE;

        switch (errorCode) {
            case ERRORCODE_NONE:
                ret = NONE;
                break;
            case ERRORCODE_GENERAL:
                ret = GENERAL;
                break;
            case ERRORCODE_TIMEOUT:
                ret = TIMEOUT;
                break;
            case ERRORCODE_PASSWORD:
                ret = PASSWORD;
                break;
            case ERRORCODE_NO_CONTENT:
                ret = NO_CONTENT;
                break;
            case ERRORCODE_MAX_SOURCESIZE:
                ret = MAX_SOURCESIZE;
                break;

            case ERRORCODE_DISPOSED:
                ret = DISPOSED;
                break;

            case ERRORCODE_UNSUPPORTED:
                ret = UNSUPPORTED;
                break;

            case ERRORCODE_OUT_OF_MEMORY:
                ret = JobError.OUT_OF_MEMORY;
                break;

            case ERRORCODE_ABORT:
                ret = JobError.ABORT;
                break;

            case ERRORCODE_PDFTOOL:
                ret = JobError.PDFTOOL;
                break;

            case ERRORCODE_MAX_QUEUE_COUNT:
                ret = JobError.MAX_QUEUE_COUNT;
                break;

            case ERRORCODE_QUEUE_TIMEOUT:
                ret = JobError.QUEUE_TIMEOUT;
                break;

            default:
                ret = GENERAL;
                break;
        }

        return ret;
    }

    /**
     * @param resultCodeNumberObj
     * @return
     */
    public static JobError fromObject(Object resultCodeNumberObj) {
        return ((null != resultCodeNumberObj) && (resultCodeNumberObj instanceof Number)) ?
            fromErrorCode(((Number) resultCodeNumberObj).intValue()) :
                JobError.NONE;
    }

    /**
     * @param action
     */
    public static void forEach(Consumer<? super JobError> action) {
        Arrays.stream(values()).forEach(action);
    }

    // - Members ---------------------------------------------------------------

    private int m_errorCode = ERRORCODE_NONE;

    private String m_errorText = "none";

    // - Constants -------------------------------------------------------------

    // the error codes to be mapped to enums and vice versa
    final private static int ERRORCODE_NONE           = 0x00000000;
    final private static int ERRORCODE_GENERAL        = 0x00000001;
    final private static int ERRORCODE_TIMEOUT        = 0x00000002;
    final private static int ERRORCODE_PASSWORD       = 0x00000004;
    final private static int ERRORCODE_NO_CONTENT     = 0x00000008;
    final private static int ERRORCODE_MAX_SOURCESIZE = 0x00000010;
    final private static int ERRORCODE_DISPOSED       = 0x00000020;
    final private static int ERRORCODE_UNSUPPORTED    = 0x00000040;
    final private static int ERRORCODE_OUT_OF_MEMORY  = 0x00000080;
    final private static int ERRORCODE_ABORT          = 0x00000100;
    final private static int ERRORCODE_PDFTOOL        = 0x00000200;
    final private static int ERRORCODE_MAX_QUEUE_COUNT= 0x00000400;
    final private static int ERRORCODE_QUEUE_TIMEOUT  = 0x00000800;
}
