/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */


package com.openexchange.documentconverter.client.json;

import java.io.File;
import java.util.UUID;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONObject;
import com.openexchange.ajax.container.ByteArrayInputStreamClosure;
import com.openexchange.ajax.container.FileHolder;
import com.openexchange.ajax.fileholder.IFileHolder;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.documentconverter.DocumentConverterManager;
import com.openexchange.documentconverter.FileIdManager;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.JobError;
import com.openexchange.documentconverter.JobPriority;
import com.openexchange.documentconverter.MutableWrapper;
import com.openexchange.documentconverter.NonNull;
import com.openexchange.documentconverter.PDFExtractor;
import com.openexchange.documentconverter.client.impl.ClientConfig;
import com.openexchange.documentconverter.client.impl.ClientManager;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link ConvertDocumentAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
class ConvertDocumentAction extends DocumentConverterAction {

    /**
     * Initializes a new {@link ConvertDocumentAction}.
     *
     * @param services
     */
    public ConvertDocumentAction(final ServiceLookup services, final ClientManager manager, final FileIdManager fileIdManager, final ClientConfig clientConfig) {
        super(services, manager, fileIdManager, clientConfig);
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.ajax.requesthandler.AJAXActionService#perform(com.openexchange.ajax.requesthandler.AJAXRequestData,
     * com.openexchange.tools.session.ServerSession)
     */
    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) {
        final PDFExtractor pdfExtractor = m_manager.getPDFExtractor();
        final IDocumentConverter manager = m_services.getService(IDocumentConverter.class);

        if ((null != pdfExtractor) && (null != request)  && (null != session) && (null != manager)) {
            String convertAction = StringUtils.lowerCase(request.getParameter("convert_action"));

            if (null != convertAction) {
                IFileHolder resultFileHolder = null;
                JSONObject jsonResult = new JSONObject();
                final MutableWrapper<Boolean> documentDecrypted = new MutableWrapper<>(Boolean.FALSE);
                final MutableWrapper<JobError> jobError = new MutableWrapper<>(JobError.GENERAL);

                if ("beginconvert".equals(convertAction)) {
                    performBeginConvert(
                        request,
                        session,
                        manager,
                        JobPriority.fromString(request.getParameter("convert_priority")),
                        documentDecrypted,
                        jsonResult,
                        jobError);
                } else if ("getpage".equals(convertAction)) {
                    resultFileHolder = performGetPage(
                        request,
                        session,
                        manager,
                        jsonResult,
                        jobError);
                } else if ("endconvert".equals(convertAction)) {
                    performEndConvert(
                        request,
                        session,
                        manager,
                        jsonResult,
                        jobError);
                }

                return getRequestResult(request, resultFileHolder, jsonResult, jobError.get());
            }
        }

        return new AJAXRequestResult(null, FORMAT_JSON);
    }

    /**
     * @param request
     * @param session
     * @param manager
     * @param convertPriority
     * @param jsonResult
     * @param jobError
     */
    private void performBeginConvert(@NonNull AJAXRequestData request, @
        NonNull ServerSession session,
        @NonNull IDocumentConverter manager,
        @NonNull JobPriority convertPriority,
        @NonNull MutableWrapper<Boolean> documentDecrypted,
        @NonNull JSONObject jsonResult,
        @NonNull MutableWrapper<JobError> jobError) {

        final MutableWrapper<String> jobId = new MutableWrapper<>(UUID.randomUUID().toString());
        final File pdfResultFile = getPDFFile(request, session, false, documentDecrypted, jobId, jobError);

        if (null != pdfResultFile) {
            try {
                // resource is held with a configured timeout of AUTO_CLEANUP_TIMEOUT_MILLIS
                m_fileIdManager.lock(jobId.get(), m_clientConfig.AUTO_CLEANUP_TIMEOUT_MILLIS);

                final PDFExtractor pdfExtractor = m_manager.getPDFExtractor();

                if (null != pdfExtractor) {
                    final int pageCount = pdfExtractor.getPageCount(pdfResultFile);

                    if (pageCount > 0) {
                        jsonResult.put("jobID", jobId.get());
                        jsonResult.put("pageCount", pageCount);

                        jobError.set(JobError.NONE);
                    }
                } else {
                    DocumentConverterManager.logError("DC client Ajax #beginConvert handler has no valid PDFExtractor to begin conversion of PDF file: " + jobId);
                    jobError.set(JobError.GENERAL);
                }
            } catch (Exception e) {
                jobError.set(JobError.GENERAL);
                DocumentConverterManager.logExcp(e);
            } finally {
                m_fileIdManager.unlock(jobId.get());
            }
        }
    }

    /**
     * @param request
     * @param session
     * @param manager
     * @param jsonResult
     * @param jobError
     * @return
     */
    private IFileHolder performGetPage(@NonNull AJAXRequestData request, @NonNull ServerSession session, @NonNull IDocumentConverter manager, @NonNull JSONObject jsonResult, @NonNull MutableWrapper<JobError> jobError) {
        IFileHolder resultFileHolder = null;
        final String jobId = request.getParameter("job_id");
        final String targetFormat = request.getParameter("target_format");
        final String pageNumberStr = request.getParameter("page_number");
        final int pageNumber = (null != pageNumberStr) ? Integer.parseInt(pageNumberStr) : 0;
        final boolean isPngTarget = "png".equalsIgnoreCase(targetFormat);
        final boolean isJpegTarget = !isPngTarget && ("jpg".equalsIgnoreCase(targetFormat) || "jpeg".equalsIgnoreCase(targetFormat));
        final boolean isSVGTarget = !isPngTarget && !isJpegTarget && "svg".equalsIgnoreCase(targetFormat);

        jobError.set(JobError.GENERAL);

        if (StringUtils.isNotBlank(jobId) && (pageNumber > 0) && (isPngTarget || isJpegTarget|| isSVGTarget)) {
            File tmpOutputFile = null;

            try {
                m_fileIdManager.lock(jobId);

                final PDFExtractor pdfExtractor = m_manager.getPDFExtractor();
                final File pdfInputFile = m_fileIdManager.getDataFile(jobId);

                if (null == pdfInputFile) {
                    DocumentConverterManager.logError("DC client Ajax #getPage handler got no input file for PDF document. The input file has been removed by auto cleanup handler in the meantime: " + jobId);
                    jobError.set(JobError.NO_CONTENT);
                } else if (null == pdfExtractor) {
                    DocumentConverterManager.logError("DC client Ajax #getPage handler has no valid PDFExtractor to get page from PDF file : " + jobId);
                    jobError.set(JobError.GENERAL);
                } else {
                    final String targetWidthStr = request.getParameter("target_width");
                    final String targetHeightStr = request.getParameter("target_height");
                    final int pixelWidth = new Integer((null != targetWidthStr) ? targetWidthStr : "-1").intValue();
                    final int pixelHeight = new Integer((null != targetHeightStr) ? targetHeightStr : "-1").intValue();

                    tmpOutputFile = m_manager.createTempFile(jobId);

                    m_manager.getPDFExtractor().extractToGraphic(
                        pdfInputFile,
                        tmpOutputFile,
                        targetFormat,
                        pageNumber,
                        pixelWidth,
                        pixelHeight,
                        "contain");

                    // prepare Ajax result object
                    resultFileHolder = new FileHolder(
                        new ByteArrayInputStreamClosure(FileUtils.readFileToByteArray(tmpOutputFile)),
                        tmpOutputFile.length(),
                        isPngTarget ? "image/png" : (isSVGTarget ? "image/svg+xml" : "image/jpeg"),
                            new StringBuilder(32).append("pdfpage_").
                            append(pageNumber).
                            append(isPngTarget ? ".png" : (isSVGTarget ? ".svg" : ".jpg")).toString());

                    request.setFormat(FORMAT_FILE);
                    request.setProperty("target_width", Integer.valueOf(pixelWidth));
                    request.setProperty("target_height", Integer.valueOf(pixelHeight));

                    jobError.set(JobError.NONE);
                }
            } catch (Exception e) {
                DocumentConverterManager.logExcp(e);
            } finally {
                FileUtils.deleteQuietly(tmpOutputFile);
                m_fileIdManager.unlock(jobId);
            }
        }

        return resultFileHolder;
    }

    /**
     * @param request
     * @param session
     * @param manager
     * @param jsonResult
     * @param jobError
     */
    private void performEndConvert(@NonNull AJAXRequestData request, @NonNull ServerSession session, @NonNull IDocumentConverter manager, @NonNull JSONObject jsonResult, @NonNull MutableWrapper<JobError> jobError) {
        final String jobId = request.getParameter("job_id");

        if (StringUtils.isNotBlank(jobId)) {
            try {
                m_fileIdManager.lock(jobId);
                m_fileIdManager.delete(jobId);
            } finally {
                jobError.set(JobError.NONE);
                m_fileIdManager.unlock(jobId);
            }
        } else {
            jobError.set(JobError.NO_CONTENT);
        }
    }
}
