/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */


package com.openexchange.documentconverter.client.json;

import java.io.File;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONObject;
import com.openexchange.ajax.container.ByteArrayInputStreamClosure;
import com.openexchange.ajax.container.FileHolder;
import com.openexchange.ajax.fileholder.IFileHolder;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.JobError;
import com.openexchange.documentconverter.JobPriority;
import com.openexchange.documentconverter.NonNull;
import com.openexchange.documentconverter.PDFExtractor;
import com.openexchange.documentconverter.client.impl.ClientManager;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link ConvertDocumentAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
class ConvertDocumentAction extends DocumentConverterAction {

    /**
     * Initializes a new {@link ConvertDocumentAction}.
     *
     * @param services
     */
    public ConvertDocumentAction(ServiceLookup services) {
        super(services);
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.ajax.requesthandler.AJAXActionService#perform(com.openexchange.ajax.requesthandler.AJAXRequestData,
     * com.openexchange.tools.session.ServerSession)
     */
    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) {
        final IDocumentConverter manager = m_services.getService(IDocumentConverter.class);
        final File pdfExtractorExecutable = ClientManager.get().getClientConfig().PDFEXTRACTOR_EXECUTABLE_PATH;

        if ((null != request)  && (null != session) && (null != manager) && (null != pdfExtractorExecutable)) {
            String convertAction = StringUtils.lowerCase(request.getParameter("convert_action"));

            if (null != convertAction) {
                IFileHolder resultFileHolder = null;
                JSONObject jsonResult = new JSONObject();
                JobError jobError[] = { JobError.GENERAL };

                if (convertAction.equals("beginconvert")) {
                    performBeginConvert(request, session, manager, JobPriority.fromString(request.getParameter("convert_priority")), jsonResult, jobError);
                } else if (convertAction.equals("getpage")) {
                    resultFileHolder = performGetPage(request, session, manager, jsonResult, jobError);
                } else if (convertAction.equals("endconvert")) {
                    performEndConvert(request, session, manager, jsonResult, jobError);
                }

                return getRequestResult(request, resultFileHolder, jsonResult, jobError[0]);
            }
        }

        return new AJAXRequestResult(null, "json");
    }

    /**
     * @param request
     * @param session
     * @param manager
     * @param convertPriority
     * @param jsonResult
     * @param jobError
     */
    private void performBeginConvert(@NonNull AJAXRequestData request, @NonNull ServerSession session, @NonNull IDocumentConverter manager, @NonNull JobPriority convertPriority, @NonNull JSONObject jsonResult, @NonNull JobError[] jobError) {
        final ManagedFile managedResultFile = getPDFFile(request, session, false, jobError);

        if (null != managedResultFile) {
            final FileIdLocker fileIdLocker = FileIdLocker.get(m_managedFileManagement);
            final String resultFileId = managedResultFile.getID();

            try {
                fileIdLocker.lock(resultFileId);

                final int pageCount = getPDFPageCount(managedResultFile.getFile());

                if (pageCount > 0) {
                    jsonResult.put("jobID", resultFileId);
                    jsonResult.put("pageCount", pageCount);

                    fileIdLocker.holdManagedFile(resultFileId);
                    jobError[0] = JobError.NONE;
                }
            } catch (Exception e) {
                jobError[0] = JobError.GENERAL;
                logException(e);
            } finally {
                fileIdLocker.unlock(resultFileId);
            }

        } else {
            jobError[0] = JobError.GENERAL;
        }
    }

    /**
     * @param request
     * @param session
     * @param manager
     * @param jsonResult
     * @param jobError
     * @return
     */
    private IFileHolder performGetPage(@NonNull AJAXRequestData request, @NonNull ServerSession session, @NonNull IDocumentConverter manager, @NonNull JSONObject jsonResult, @NonNull JobError[] jobError) {
        IFileHolder resultFileHolder = null;
        final String jobIdStr = request.getParameter("job_id");
        final String targetFormat = request.getParameter("target_format");
        final String pageNumberStr = request.getParameter("page_number");
        final int pageNumber = (null != pageNumberStr) ? Integer.parseInt(pageNumberStr) : 0;
        final boolean isPngTarget = "png".equalsIgnoreCase(targetFormat);
        final boolean isJpegTarget = !isPngTarget && ("jpg".equalsIgnoreCase(targetFormat) || "jpeg".equalsIgnoreCase(targetFormat));

        if ((null != jobIdStr) && (pageNumber > 0) && (isPngTarget || isJpegTarget)) {
            final FileIdLocker fileIdLocker = FileIdLocker.get(m_managedFileManagement);
            File tmpOutputFile = null;

            try {
                fileIdLocker.lock(jobIdStr);

                final PDFExtractor pdfExtractor = getPDFExtractor();
                final ManagedFile pdfInputFile = getManagedFile(jobIdStr);

                if (null == pdfInputFile) {
                    logException(new RuntimeException("DC Ajax handler got no managed file for PDF document: " + jobIdStr));
                }

                if (null != pdfExtractor) {
                    final String targetWidthStr = request.getParameter("target_width");
                    final String targetHeightStr = request.getParameter("target_height");
                    final int pixelWidth = new Integer((null != targetWidthStr) ? targetWidthStr : "-1").intValue();
                    final int pixelHeight = new Integer((null != targetHeightStr) ? targetHeightStr : "-1").intValue();

                    tmpOutputFile = File.createTempFile(jobIdStr, ".tmp", pdfExtractor.getWorkDir());

                    pdfExtractor.extractToGraphic(
                        pdfInputFile.getFile(),
                        tmpOutputFile,
                        targetFormat,
                        pageNumber,
                        pixelWidth,
                        pixelHeight,
                        "contain");

                    // prepare Ajax result object
                    resultFileHolder = new FileHolder(
                        new ByteArrayInputStreamClosure(FileUtils.readFileToByteArray(tmpOutputFile)),
                            tmpOutputFile.length(),
                            isPngTarget ? "image/png" : "image/jpeg",
                            new StringBuilder(32).append("pdfpage_").
                                append(pageNumber).
                                append(isPngTarget ? ".png" : ".jpg").toString());

                    request.setFormat("file");
                    request.setProperty("target_width", Integer.valueOf(pixelWidth));
                    request.setProperty("target_height", Integer.valueOf(pixelHeight));

                    jobError[0] = JobError.NONE;
                }
            } catch (Exception e) {
                jobError[0] = JobError.GENERAL;
                logException(e);
            } finally {
                FileUtils.deleteQuietly(tmpOutputFile);
                fileIdLocker.unlock(jobIdStr);
            }
        }

        return resultFileHolder;
    }

    /**
     * @param request
     * @param session
     * @param manager
     * @param jsonResult
     * @param jobError
     */
    private void performEndConvert(@NonNull AJAXRequestData request, @NonNull ServerSession session, @NonNull IDocumentConverter manager, @NonNull JSONObject jsonResult, @NonNull JobError[] jobError) {
        final String jobIdStr = request.getParameter("job_id");

        if (StringUtils.isNotBlank(jobIdStr)) {
            final FileIdLocker fileIdLocker = FileIdLocker.get(m_managedFileManagement);

            try {
                fileIdLocker.lock(jobIdStr);
                fileIdLocker.unholdManagedFile(jobIdStr);
            } finally {
                fileIdLocker.unlock(jobIdStr);
                jobError[0] = JobError.NONE;
            }
        } else {
            jobError[0] = JobError.NO_CONTENT;
        }
    }
}
