/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    .
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2016 OX Software GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;


/**
 * {@link DataSourceFile}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.6.0
 */
public class DataSourceFile implements Serializable {

    private static final long serialVersionUID = 1L;

    /**
     * Initializes a new {@link DataSourceFile}.
     */
    public DataSourceFile(@NonNull File sourceFile) {
        super();
        m_sourceFile = sourceFile;
    }

    /**
     * @return The data source file
     */
    public File getSourceFile() {
        return m_sourceFile;
    }

    /**
     * @return The byte array containing the source file content after reading
     */
    public byte[] getDataSinkBuffer() {
        return m_dataSinkBuffer;
    }

    /**
     * @param objInputStm
     * @throws IOException
     * @throws ClassNotFoundException
     */
    private synchronized void readObject(ObjectInputStream objInputStm) throws IOException, ClassNotFoundException {
        m_sourceFile = null;
        m_dataSinkBuffer = null;

        if (null != objInputStm) {
            Object readObj = null;

            objInputStm.defaultReadObject();

            if ((null != (readObj = objInputStm.readObject())) && (readObj instanceof Integer)) {
                final int size = ((Integer) readObj).intValue();

                if (size > 0) {
                    final BufferedInputStream bufferedInputStm = new BufferedInputStream(objInputStm);
                    byte[] readBuffer = new byte[8192];
                    int readBytes = 0;
                    int destOffset = 0;

                    m_dataSinkBuffer = new byte[size];

                    try {
                        while ((readBytes = bufferedInputStm.read(readBuffer)) > -1) {
                            System.arraycopy(readBuffer, 0, m_dataSinkBuffer, destOffset, readBytes);
                            destOffset += readBytes;
                        }
                    } catch (IndexOutOfBoundsException e) {
                        DocumentConverterManager.logExcp(e);
                    }
                }
            }
        }
    }

    /**
     * @param objOutputStm
     * @throws IOException
     */
    private synchronized void writeObject(ObjectOutputStream objOutputStm) throws IOException {
        if (null != objOutputStm) {
            final Integer fileSize = new Integer((null != m_sourceFile) && (m_sourceFile.canRead()) ? (int) m_sourceFile.length() : 0);

            // write members by default
            objOutputStm.defaultWriteObject();

            // write file size as Long object
            objOutputStm.writeObject(fileSize);

            // write content of data file, if readable and size > 0
            if (fileSize.longValue() > 0) {
                InputStream dataInputStm = null;

                try {
                    dataInputStm = FileUtils.openInputStream(m_sourceFile);
                    IOUtils.copy(dataInputStm, objOutputStm);
                } catch (IOException e) {
                    DocumentConverterManager.logExcp(e);
                } finally {
                    IOUtils.closeQuietly(dataInputStm);
                }
            }
        }
    }

    // - Members ---------------------------------------------------------------

    private File m_sourceFile;

    private transient byte[] m_dataSinkBuffer = null;
}
