/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter.client.json;

import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Timer;
import java.util.TimerTask;
import com.openexchange.documentconverter.DocumentConverterManager;
import com.openexchange.documentconverter.NonNull;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.filemanagement.ManagedFileManagement;

/**
 * {@link FileIdLocker}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.4
 */
public class FileIdLocker {

    /**
     * Initializes a new {@link FileIdLocker}.
     */
    private FileIdLocker() {
    }

    /**
     * Initializes a new {@link FileIdLocker}.
     */
    private FileIdLocker(@NonNull ManagedFileManagement fileManagement) {
        this.fileManagement = fileManagement;

        implStartTimer();
    }

    /**
     * @param fileManagement
     * @return
     */
    public synchronized static FileIdLocker get(@NonNull ManagedFileManagement fileManagement) {
        if (null == fileIdLocker) {
            fileIdLocker = new FileIdLocker(fileManagement);
        }

        return fileIdLocker;

    }

    /**
     * @param fileId
     */
    public void lock(final String fileId) {
        if (null != fileId) {
            FileIdLock fileIdLock = null;

            synchronized (this.fileIdSet) {
                fileIdLock = implGetFileIdLock(fileId);

                if (null == fileIdLock) {
                    this.fileIdSet.add(fileIdLock = new FileIdLock(fileId));
                } else {
                    fileIdLock.updateTimestamp();
                }
            }

            fileIdLock.lock();
        }
    }

    /**
     * @param fileId
     */
    public void unlock(final String fileId) {
        if (null != fileId) {
            synchronized (this.fileIdSet) {
                final FileIdLock fileIdLock = implGetFileIdLock(fileId);

                if (null != fileIdLock) {
                    // update timestamp (touch) and move entry to last position in list
                    fileIdLock.updateTimestamp();

                    if (this.fileIdSet.remove(fileIdLock)) {
                        this.fileIdSet.add(fileIdLock);
                    }

                    // finally unlock entry
                    fileIdLock.unlock();
                }
            }
        }
    }

    /**
     * @param fileManagement
     */
    public void holdManagedFile(final String fileId) {
        if (null != fileId) {
            synchronized (this.fileIdSet) {
                final FileIdLock fileIdLock = implGetFileIdLock(fileId);

                if (null != fileIdLock) {
                    fileIdLock.holdManagedFile();
                    implTouchManagedFile(fileIdLock);
                }
            }
        }
    }

    /**
     * @param fileManagement
     */
    public void unholdManagedFile(final String fileId) {
        if (null != fileId) {
            synchronized (this.fileIdSet) {
                final FileIdLock fileIdLock = implGetFileIdLock(fileId);

                if (null != fileIdLock) {
                    fileIdLock.unholdManagedFile(this.fileManagement);
                }
            }
        }
    }

    /**
     * !!! <b>Method is not synchronized. Synchronization on <i>this.fileIdSet</i>
     *     <u>has</u> to been done in calling method </b>!!!
     *
     * @param fileId
     * @return
     */
    protected FileIdLock implGetFileIdLock(@NonNull final String fileId) {
        final Iterator<FileIdLock> iter = this.fileIdSet.iterator();

        while (iter.hasNext()) {
            final FileIdLock curFileIdLock = iter.next();

            if (curFileIdLock.getFileId() == fileId) {
                return curFileIdLock;
            }
        }

        return null;
    }

    /**
     * @param fileIdLock
     */
    protected void implTouchManagedFile(@NonNull final FileIdLock fileIdLock) {
        if (null != this.fileManagement) {
            try {
                final ManagedFile managedFile = this.fileManagement.getByID(fileIdLock.getFileId());

                if (null != managedFile) {
                    managedFile.touch();
                }
            } catch (Exception e) {
                DocumentConverterManager.logExcp(e);
            }
        }
    }

    /**
     *
     */
    protected void implStartTimer() {
        this.cleanupTimer.scheduleAtFixedRate(new TimerTask() {

            /* (non-Javadoc)
             * @see java.util.TimerTask#run()
             */
            @Override
            public void run() {
                try {
                    final long curTimeMillis = System.currentTimeMillis();
                    final boolean trace = DocumentConverterManager.isLogTrace();
                    int mapCountBefore = 0;

                    if (trace) {
                        mapCountBefore = FileIdLocker.this.fileIdSet.size();
                    }

                    synchronized (FileIdLocker.this.fileIdSet) {
                        Iterator<FileIdLock> iter = FileIdLocker.this.fileIdSet.iterator();

                        while (iter.hasNext()) {
                            final FileIdLock curFileIdLock = iter.next();

                            if (null != curFileIdLock) {
                                // Don't remove entry, if the managed file is to be held, touch instead.
                                if (curFileIdLock.isManagedFileHeld()) {
                                    implTouchManagedFile(curFileIdLock);

                                    if (trace) {
                                        DocumentConverterManager.logTrace("DC FileIdLocker touched ManagedFile: " + curFileIdLock.getFileId());
                                    }

                                } else if (!curFileIdLock.isLocked()) {
                                    // Check for a timeout
                                    if ((curTimeMillis - curFileIdLock.getTimestampMillis()) >= TIMEOUT_MILLIS) {
                                        iter.remove();
                                    }
                                }
                            }
                        }
                    }

                    if (trace) {
                        final int mapCountAfter = FileIdLocker.this.fileIdSet.size();

                        DocumentConverterManager.logTrace(new StringBuilder(128).
                            append("DC FileIdLocker cleanup removed/left entries: ").
                            append((mapCountBefore - mapCountAfter)).append(" / ").
                            append(mapCountAfter).toString());
                    }
                } catch (Throwable e) {
                    DocumentConverterManager.logExcp(new Exception(e));
                }
            }

        }, 0, PERIOD_MILLIS);
    }

    // - Static Members --------------------------------------------------------

    // 6 minutes
    final private static long TIMEOUT_MILLIS = 360 * 1000;

    // 3 minutes
    final private static long PERIOD_MILLIS = 180 * 1000;

    private static FileIdLocker fileIdLocker = null;

    // - Members ---------------------------------------------------------------

    protected ManagedFileManagement fileManagement;

    protected LinkedHashSet<FileIdLock> fileIdSet = new LinkedHashSet<>();

    protected Timer cleanupTimer = new Timer(true);
}
