/*
 *
 *  Copyright (C) 2012 Open-Xchange, Inc. All rights reserved.
 *
 *  This program is proprietary software and licensed to you under Open-Xchange
 *  Inc's Software License Agreement.
 *
 *  In some countries OX, OX Open-Xchange, open-xchange and OXtender as well as
 *  the corresponding Logos OX Open-Xchange and OX are registered trademarks of
 *  the Open-Xchange, Inc. group of companies.
 *
 */

package com.openexchange.documentconverter;

import java.io.File;
import java.io.InputStream;
import java.util.HashMap;

//------------
//- IManager -
//------------

/**
 * {@link IManager}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public interface IManager {

    /**
     * @param backendDescriptors the array of engine descriptors, containing the properties of each single backend engine
     * @param cacheDescriptor contains properties to setup a JobCache
     * @return the JobProcessor to execute single jobs or null in case of failure
     */
    public IJobProcessor init(BackendDescriptor backendDescriptors[], CacheDescriptor cacheDescriptor);

    /**
     * kills all engine instances etc. and leaves the service with a clean status
     */
    public void end();

    /**
     * @param feature the feature that is queried to be supported
     * @return true if the given feature is supported by this IManager instance
     */
    boolean hasFeature(Feature feature);

    /**
     * @return the successfully initialized JobProcessor or null in case of failure
     */
    IJobProcessor getJobProcessor();

    /**
     * @param jobType the export format for the conversion
     * @param jobProperties contains additional properties to customize the job
     * @return the new Job or null in case of failure
     */
    public IJob createServerJob(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties);

    /**
     * @return the instance that offers a UNO interface to the underlying engine backend or null in case of a failure
     */
    public IInstance createServerInstance(BackendType backendType);

    /**
     * @param jobType The job type fiter to be executed.
     * @param jobProperties Input parameter, containing properties to customize the job (e.g. the InputFile)
     * @param resultProperties Output parameter,  containing additional conversion result properties like e.g. the PageCount of the conversion result.
     * @return The InputStream with the conversion result in case of success or null in case of a failure
     */
    /**
     * @param jobType
     * @param jobProperties
     * @param resultProperties
     * @return The InputStream with the conversion result in case of success or null in case of a failure
     */
    public InputStream convert(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties);

    /**
     * @param jobType The job type fiter to be executed.
     * @param documentInputStream The InputStream, containing the source document to be converted.
     * @param jobProperties Input parameter, containing properties to customize the job (e.g. the InputFile)
     * @param resultProperties Output parameter,  containing additional conversion result properties like e.g. the PageCount of the conversion result.
     * @return The job id string to be used in further getConversionPage/endPageConversion jobs or null in case of a failure
     */
    public String beginPageConversion(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties);

    /**
     * @param jobIdStr The job id gotten from a previous call to beginPageConversion
     * @param pageNumber The number of the page to be returned as input stream.
     * @param pageProperties input/output parameter, containing additional properties to customize the job (input) and containing additional
     *            conversion result properties like e.g. the mime type of the conversion result (output).
     * @return The stream the converted page can be read from or null in case of a failure
     */
    public InputStream getConversionPage(String jobIdStr, int pageNumber, HashMap<String, Object> resultProperties);

    /**
     * @param jobIdStr The job id gotten from a previous call to beginPageConversion
     */
    public void endPageConversion(String jobIdStr);

    /**
     * @return the installation path of the underlying DocumentReaderEngine or null if the engine wasn't properly initialized
     */
    public String getEnginePath();


    /**
     * @param svgInputStm The stream to read the SVG file input from
     * @param targetMimeType The target mime type of the rasterized SVG image; currently support are image/png and image/jpeg [in/out]
     * @param targetWidth The target width in pixels of the rasterized SVG image [in/out]
     * @param targetHeight The target height in pixels of the rasterized SVG image [in/out]
     * @param preserveAspectRatio Use the given targetWidth and targetHeight as bounding rectangle to fit the image extents, preserving the default aspect ratio
     * @param targetZoomFactor The zoom factor to apply for the rasterized SVG image;
     *        the zoom factor is applied after any extent set by target width and target height
     * @return The stream the rasterized SVG image can be read from or null in case of a failure
     */
    public InputStream convertSVG(InputStream svgInputStm, String[] targetMimeType, int[] targetPixelWidth, int[] targetPixelHeight, boolean preserveDefaultAspectRatio, float targetZoomFactor);


    /**
     * @return the created temporary File
     * @param sourceFileName if source document file name is set then the temp file will be created with this file name within a new folder
     */
    public File createTempFile(String prefix, String sourceFileName);
}
