/*
 *
 *  Copyright (C) 2012 Open-Xchange, Inc. All rights reserved.
 *
 *  This program is proprietary software and licensed to you under Open-Xchange
 *  Inc's Software License Agreement.
 *
 *  In some countries OX, OX Open-Xchange, open-xchange and OXtender as well as
 *  the corresponding Logos OX Open-Xchange and OX are registered trademarks of
 *  the Open-Xchange, Inc. group of companies.
 *
 */

package com.openexchange.documentconverter;

import java.io.File;
import java.io.InputStream;
import java.util.HashMap;

//------------
//- IManager -
//------------

/**
 * {@link IManager}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public interface IManager {

    /**
     * @param backendDescriptors the array of engine descriptors, containing the properties of each single backend engine
     * @param cacheDescriptor contains properties to setup a JobCache
     * @return the JobProcessor to execute single jobs or null in case of failure
     */
    public IJobProcessor init(BackendDescriptor backendDescriptors[], CacheDescriptor cacheDescriptor);

    /**
     * kills all engine instances etc. and leaves the service with a clean status
     */
    public void end();

    /**
     * @return the successfully initialized JobProcessor or null in case of failure
     */
    IJobProcessor getJobProcessor();

    /**
     * @param jobType the export format for the conversion
     * @param jobProperties contains additional properties to customize the job
     * @return the new Job or null in case of failure
     */
    public IJob createServerJob(String jobType, HashMap<String, Object> jobProperties);

    /**
     * @return the instance that offers a UNO interface to the underlying engine backend
     */
    public IInstance createServerInstance(BackendType backendType);

    /**
     * @param jobType The job type fiter to be executed.
     * @param jobProperties input/output parameter, containing properties to customize the job (e.g. the InputFile) and containing additional
     *            conversion result properties like e.g. the PageCount of the conversion result (output).
     * @return The InputStream with the conversion result in case of success or null.
     */
    public InputStream convert(String jobType, HashMap<String, Object> jobProperties);

    /**
     * @param jobType The job type fiter to be executed.
     * @param documentInputStream The InputStream, containing the source document to be converted.
     * @param jobProperties input/output parameter, containing properties to customize the job (e.g. the InputFile) and containing additional
     *            conversion result properties like e.g. the PageCount of the conversion result (output).
     * @return The job id to be used in further getConversionPage/endPageConversion jobs
     */
    public long beginPageConversion(String jobType, HashMap<String, Object> jobProperties);

    /**
     * @param jobId The job id gotten from a previous call to beginPageConversion
     * @param pageNumber The number of the page to be returned as input stream.
     * @param pageProperties input/output parameter, containing additional properties to customize the job (input) and containing additional
     *            conversion result properties like e.g. the mime type of the conversion result (output).
     * @return The stream the converted page can be read from
     */
    public InputStream getConversionPage(long jobId, int pageNumber, HashMap<String, Object> pageProperties);

    /**
     * @param jobId The job id gotten from a previous call to beginPageConversion
     */
    public void endPageConversion(long jobId);

    /**
     * @return the installation path of the underlying DocumentReaderEngine or null if the engine wasn't properly initialized
     */
    public String getEnginePath();

    /**
     * @return the created temporary File
     */
    public File createTempFile(String prefix);
}
