/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.file.storage;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import com.openexchange.exception.OXException;

/**
 * {@link DefaultWarningsAware} - The default implementation of {@link WarningsAware}.
 * <p>
 * Supposed to be used to add {@code WarningsAware} in a delegate-fashion:
 *
 * <pre>
 *
 * private final WarningsAware warningsAware = new DefaultWarningsAware();
 *
 * &#064;Override
 * public List&lt;OXException&gt; getWarnings() {
 *     return warningsAware.getWarnings();
 * }
 *
 * &#064;Override
 * public List&lt;OXException&gt; getAndFlushWarnings() {
 *     return warningsAware.getAndFlushWarnings();
 * }
 *
 * &#064;Override
 * public void addWarning(OXException warning) {
 *     warningsAware.addWarning(warning);
 * }
 *
 * &#064;Override
 * public void removeWarning(OXException warning) {
 *     warningsAware.removeWarning(warning);
 * }
 * </pre>
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public class DefaultWarningsAware implements WarningsAware {

    private final List<OXException> warnings;

    /**
     * Initializes a new {@link DefaultWarningsAware}.
     */
    public DefaultWarningsAware() {
        this(false);
    }

    /**
     * Initializes a new {@link DefaultWarningsAware}.
     *
     * @param concurrent Whether concurrent access is supposed to be supported or not
     */
    public DefaultWarningsAware(final boolean concurrent) {
        super();
        warnings = concurrent ? new CopyOnWriteArrayList<OXException>() : new ArrayList<OXException>(4);
    }

    @Override
    public List<OXException> getWarnings() {
        return new ArrayList<OXException>(warnings);
    }

    @Override
    public List<OXException> getAndFlushWarnings() {
        final List<OXException> ret = new ArrayList<OXException>(warnings);
        warnings.clear();
        return ret;
    }

    @Override
    public void addWarning(final OXException warning) {
        warnings.add(warning);
    }

    @Override
    public void removeWarning(final OXException warning) {
        warnings.remove(warning);
    }

}
