/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.oauth;

import com.openexchange.java.Strings;

/**
 * {@link HostInfo} - Provides name/IP address as well as the JVM route for the current host.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.10.0
 */
public class HostInfo {

    private final String host;
    private final String route;

    /**
     * Initializes a new {@link HostInfo}.
     *
     * @param host The name or IP address of the current host
     * @param route The JVM route for current host
     */
    public HostInfo(String host, String route) {
        super();
        this.host = host;
        this.route = route;
    }

    /**
     * Gets the name or IP address of the host
     *
     * @return The name or IP address of the host
     */
    public String getHost() {
        return host;
    }

    /**
     * Gets the JVM route
     *
     * @return The JVM route
     */
    public String getRoute() {
        return route;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((host == null) ? 0 : host.hashCode());
        result = prime * result + ((route == null) ? 0 : route.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        HostInfo other = (HostInfo) obj;
        if (host == null) {
            if (other.host != null) {
                return false;
            }
        } else if (!host.equals(other.host)) {
            return false;
        }
        if (route == null) {
            if (other.route != null) {
                return false;
            }
        } else if (!route.equals(other.route)) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder("HostInfo [");
        if (host != null) {
            builder.append("host=").append(host).append(", ");
        }
        if (route != null) {
            builder.append("route=").append(route);
        }
        builder.append("]");
        return builder.toString();
    }

    /**
     * Injects this instance's JVM route into given URL.
     *
     * @param url The URL to inject to
     * @param route The JVM route to inject
     * @return The URL with JVM route injected
     */
    public String injectRoute(String url) {
        return injectRouteIntoUrl(url, route);
    }

    // ---------------------------------------------------------------------------------------

    /**
     * Injects specified JVM route into given URL.
     *
     * @param url The URL to inject to
     * @param route The JVM route to inject
     * @return The URL with JVM route injected
     */
    public static String injectRouteIntoUrl(String url, String route) {
        if (null == url || Strings.isEmpty(route)) {
            return url;
        }
        if (url.indexOf(";jsessionid=") >= 0) {
            // Route already contained
            return url;
        }

        // Check for query string
        int queryStringPos = url.indexOf('?');
        if (queryStringPos > 0) {
            return new StringBuilder(url.substring(0, queryStringPos)).append(";jsessionid=").append(route).append(url.substring(queryStringPos)).toString();
        }
        return new StringBuilder(url).append(";jsessionid=").append(route).toString();
    }

}
