/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mail.mime;

import static com.openexchange.mail.MailServletInterface.mailInterfaceMonitor;
import java.net.SocketException;
import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.mail.Address;
import javax.mail.Folder;
import javax.mail.MessageRemovedException;
import javax.mail.MessagingException;
import javax.mail.SendFailedException;
import javax.mail.Store;
import javax.mail.internet.AddressException;
import com.openexchange.exception.Category;
import com.openexchange.exception.OXException;
import com.openexchange.log.LogProperties;
import com.openexchange.log.LogProperties.Name;
import com.openexchange.mail.MailExceptionCode;
import com.openexchange.mail.api.MailConfig;
import com.openexchange.mailaccount.MailAccount;
import com.openexchange.session.Session;
import com.openexchange.tools.exceptions.ExceptionUtils;
import com.sun.mail.iap.CommandFailedException;
import com.sun.mail.smtp.SMTPAddressFailedException;
import com.sun.mail.smtp.SMTPSendFailedException;
import com.sun.mail.smtp.SMTPSenderFailedException;

/**
 * {@link OXException} - For MIME related errors.
 * <p>
 * Taken from {@link OXException}:
 * <p>
 * The detail number range in subclasses generated in mail bundles is supposed to start with 2000 and may go up to 2999.
 * <p>
 * The detail number range in subclasses generated in transport bundles is supposed to start with 3000 and may go up to 3999.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public class MimeMailException extends OXException {

    private static final transient org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(MimeMailException.class);

    private static final long serialVersionUID = -3401580182929349354L;

    /**
     * Initializes a new {@link MimeMailException}.
     *
     * @param code
     * @param displayMessage
     * @param displayArgs
     */
    public MimeMailException(final int code, final String displayMessage, final Object... displayArgs) {
        super(code, displayMessage, displayArgs);
    }

    /**
     * Initializes a new {@link MimeMailException}.
     *
     * @param code
     * @param displayMessage
     * @param cause
     * @param displayArgs
     */
    public MimeMailException(final int code, final String displayMessage, final Throwable cause, final Object... displayArgs) {
        super(code, displayMessage, cause, displayArgs);
    }

    /**
     * Handles given instance of {@link MessagingException} and creates an appropriate instance of {@link OXException}
     * <p>
     * This is just a convenience method that simply invokes {@link #handleMessagingException(MessagingException, MailConfig)} with the
     * latter parameter set to <code>null</code>.
     *
     * @param e The messaging exception
     * @return An appropriate instance of {@link OXException}
     */
    public static OXException handleMessagingException(final MessagingException e) {
        return handleMessagingException(e, null, null);
    }

    /**
     * Handles given instance of {@link MessagingException} and creates an appropriate instance of {@link OXException}
     *
     * @param e The messaging exception
     * @param mailConfig The corresponding mail configuration used to add information like mail server etc.
     * @return An appropriate instance of {@link OXException}
     */
    public static OXException handleMessagingException(final MessagingException e, final MailConfig mailConfig) {
        return handleMessagingException(e, mailConfig, mailConfig.getSession());
    }

    /**
     * Handles given instance of {@link MessagingException} and creates an appropriate instance of {@link OXException}
     *
     * @param e The messaging exception
     * @param mailConfig The corresponding mail configuration used to add information like mail server etc.
     * @param session The session providing user information
     * @return An appropriate instance of {@link OXException}
     */
    public static OXException handleMessagingException(final MessagingException e, final MailConfig mailConfig, final Session session) {
        return handleMessagingException(e, mailConfig, session, null);
    }

    private static final String STR_EMPTY = "";

    private static final String ERR_TMP = "temporary error, please try again later";

    private static final String ERR_TMP_FLR = "temporary failure";

    private static final String ERR_AUTH_FAILED = "bad authentication failed";

    private static final String ERR_MSG_TOO_LARGE = "message too large";

    private static final String ERR_QUOTA = "quota";

    /**
     * ConnectionResetException
     */
    private static final String EXC_CONNECTION_RESET_EXCEPTION = "ConnectionResetException";

    private static final Object[] EMPTY_ARGS = new Object[0];

    /**
     * Handles given instance of {@link MessagingException} and creates an appropriate instance of {@link OXException}
     *
     * @param e The messaging exception
     * @param mailConfig The corresponding mail configuration used to add information like mail server etc.
     * @param session The session providing user information
     * @param folder The optional folder
     * @return An appropriate instance of {@link OXException}
     */
    public static OXException handleMessagingException(final MessagingException e, final MailConfig mailConfig, final Session session, final Folder folder) {
        try {
            // Put log properties
            if (null != mailConfig) {
                LogProperties.put(Name.MAIL_ACCOUNT_ID, Integer.valueOf(mailConfig.getAccountId()));
                LogProperties.put(Name.MAIL_HOST, mailConfig.getServer());
                LogProperties.put(Name.MAIL_LOGIN, mailConfig.getLogin());
                if (null != folder) {
                    LogProperties.put(Name.MAIL_FULL_NAME, folder.getFullName());
                }
            }
            // Start examining MessageException
            if (e instanceof MessageRemovedException) {
                // Message has been removed in the meantime
                if (null != folder) {
                    throw MailExceptionCode.MAIL_NOT_FOUND.create(e, "", folder.getFullName());
                }
                throw MailExceptionCode.MAIL_NOT_FOUND_SIMPLE.create(e, new Object[0]);
            }
            if ((e instanceof javax.mail.AuthenticationFailedException) || ((toLowerCase(e.getMessage(), "").indexOf(ERR_AUTH_FAILED) != -1))) {
                // Authentication failed
                if (null != mailConfig && MailAccount.DEFAULT_ID == mailConfig.getAccountId()) {
                    return MimeMailExceptionCode.LOGIN_FAILED.create(e, mailConfig.getServer(), mailConfig.getLogin());
                }
                if ((e.getMessage() != null) && ERR_TMP.equals(toLowerCase(e.getMessage()))) {
                    return MimeMailExceptionCode.LOGIN_FAILED.create(
                        e,
                        mailConfig == null ? STR_EMPTY : mailConfig.getServer(),
                            mailConfig == null ? STR_EMPTY : mailConfig.getLogin());
                }
                if (null != mailConfig && null != session) {
                    return MimeMailExceptionCode.INVALID_CREDENTIALS_EXT.create(
                        e,
                        mailConfig.getServer(),
                        mailConfig.getLogin(),
                        Integer.valueOf(session.getUserId()),
                        Integer.valueOf(session.getContextId()),
                        e.getMessage());
                }
                return MimeMailExceptionCode.INVALID_CREDENTIALS.create(
                    e,
                    mailConfig == null ? STR_EMPTY : mailConfig.getServer(),
                        e.getMessage());
            } else if (e instanceof javax.mail.FolderClosedException) {
                final Folder f = ((javax.mail.FolderClosedException) e).getFolder();
                if (null != mailConfig && null != session) {
                    return MimeMailExceptionCode.FOLDER_CLOSED_EXT.create(
                        e,
                        null == f ? appendInfo(e.getMessage(), folder) : f.getFullName(),
                            mailConfig.getServer(),
                            mailConfig.getLogin(),
                            Integer.valueOf(session.getUserId()),
                            Integer.valueOf(session.getContextId()));
                }
                return MimeMailExceptionCode.FOLDER_CLOSED.create(e, null == f ? appendInfo(e.getMessage(), folder) : f.getFullName());
            } else if (e instanceof javax.mail.FolderNotFoundException) {
                final Folder f = ((javax.mail.FolderNotFoundException) e).getFolder();
                if (null != mailConfig && null != session) {
                    return MimeMailExceptionCode.FOLDER_NOT_FOUND_EXT.create(
                        e,
                        null == f ? appendInfo(e.getMessage(), folder) : f.getFullName(),
                            mailConfig.getServer(),
                            mailConfig.getLogin(),
                            Integer.valueOf(session.getUserId()),
                            Integer.valueOf(session.getContextId()));
                }
                return MimeMailExceptionCode.FOLDER_NOT_FOUND.create(e, null == f ? appendInfo(e.getMessage(), folder) : f.getFullName());
            } else if (e instanceof javax.mail.IllegalWriteException) {
                return MimeMailExceptionCode.ILLEGAL_WRITE.create(e, appendInfo(e.getMessage(), folder));
            } else if (e instanceof javax.mail.MessageRemovedException) {
                return MimeMailExceptionCode.MESSAGE_REMOVED.create(e, appendInfo(e.getMessage(), folder));
            } else if (e instanceof javax.mail.MethodNotSupportedException) {
                return MimeMailExceptionCode.METHOD_NOT_SUPPORTED.create(e, appendInfo(e.getMessage(), folder));
            } else if (e instanceof javax.mail.NoSuchProviderException) {
                return MimeMailExceptionCode.NO_SUCH_PROVIDER.create(e, appendInfo(e.getMessage(), folder));
            } else if (e instanceof javax.mail.internet.ParseException) {
                if (e instanceof javax.mail.internet.AddressException) {
                    final String optRef = ((AddressException) e).getRef();
                    return MimeMailExceptionCode.INVALID_EMAIL_ADDRESS.create(e, optRef == null ? STR_EMPTY : optRef);
                }
                return MimeMailExceptionCode.PARSE_ERROR.create(e, appendInfo(e.getMessage(), folder));
            } else if (e instanceof javax.mail.ReadOnlyFolderException) {
                if (null != mailConfig && null != session) {
                    return MimeMailExceptionCode.READ_ONLY_FOLDER_EXT.create(
                        e,
                        appendInfo(e.getMessage(), folder),
                        mailConfig.getServer(),
                        mailConfig.getLogin(),
                        Integer.valueOf(session.getUserId()),
                        Integer.valueOf(session.getContextId()));
                }
                return MimeMailExceptionCode.READ_ONLY_FOLDER.create(e, appendInfo(e.getMessage(), folder));
            } else if (e instanceof javax.mail.search.SearchException) {
                return MimeMailExceptionCode.SEARCH_ERROR.create(e, appendInfo(e.getMessage(), folder));
            } else if (e instanceof com.sun.mail.smtp.SMTPSenderFailedException) {
                SMTPSenderFailedException failedException = (SMTPSenderFailedException) e;
                if ((failedException.getReturnCode() == 552) || (toLowerCase(failedException.getMessage(), "").indexOf(ERR_MSG_TOO_LARGE) > -1)) {
                    return MimeMailExceptionCode.MESSAGE_TOO_LARGE_EXT.create(failedException, getSmtpInfo(failedException));
                }
                return MimeMailExceptionCode.SEND_FAILED_MSG_EXT_ERROR.create(failedException, failedException.getMessage(), getSmtpInfo(failedException));
            } else if (e instanceof com.sun.mail.smtp.SMTPAddressFailedException) {
                SMTPAddressFailedException failedException = (SMTPAddressFailedException) e;
                if ((failedException.getReturnCode() == 552) || (toLowerCase(failedException.getMessage(), "").indexOf(ERR_MSG_TOO_LARGE) > -1)) {
                    return MimeMailExceptionCode.MESSAGE_TOO_LARGE_EXT.create(failedException, getSmtpInfo(failedException));
                }
                return MimeMailExceptionCode.SEND_FAILED_MSG_EXT_ERROR.create(failedException, failedException.getMessage(), getSmtpInfo(failedException));
            } else if (e instanceof com.sun.mail.smtp.SMTPSendFailedException) {
                final SMTPSendFailedException sendFailedError = (SMTPSendFailedException) e;
                if ((sendFailedError.getReturnCode() == 552) || (toLowerCase(sendFailedError.getMessage(), "").indexOf(ERR_MSG_TOO_LARGE) > -1)) {
                    return MimeMailExceptionCode.MESSAGE_TOO_LARGE_EXT.create(sendFailedError, getSmtpInfo(sendFailedError));
                }
                final Exception nextException = sendFailedError.getNextException();
                if (nextException instanceof com.sun.mail.smtp.SMTPSendFailedException) {
                    final SMTPSendFailedException smtpExc = (SMTPSendFailedException) nextException;
                    final Address[] invalidAddresses = smtpExc.getInvalidAddresses();
                    if (null == invalidAddresses || invalidAddresses.length == 0) {
                        return MimeMailExceptionCode.SEND_FAILED_MSG_EXT_ERROR.create(sendFailedError, sendFailedError.getMessage(), getSmtpInfo(smtpExc));
                    }
                }
                String serverInfo = getSmtpInfo(sendFailedError);
                if (nextException instanceof com.sun.mail.smtp.SMTPAddressFailedException) {
                    serverInfo = getSmtpInfo((com.sun.mail.smtp.SMTPAddressFailedException) nextException);
                }
                final Address[] addrs = sendFailedError.getInvalidAddresses();
                if (null == addrs || addrs.length == 0) {
                    // No invalid addresses available
                    return MimeMailExceptionCode.SEND_FAILED_MSG_ERROR.create(sendFailedError, getSmtpInfo(sendFailedError));
                }
                return MimeMailExceptionCode.SEND_FAILED_EXT.create(sendFailedError, Arrays.toString(addrs), serverInfo);
            } else if (e instanceof javax.mail.SendFailedException) {
                final SendFailedException exc = (SendFailedException) e;
                int returnCode = 250;
                String smtpInfo = null;
                Address[] invalidAddresses = exc.getInvalidAddresses();
                {
                    final Exception nextException = exc.getNextException();
                    if (nextException instanceof com.sun.mail.smtp.SMTPSendFailedException) {
                        com.sun.mail.smtp.SMTPSendFailedException failedError = (com.sun.mail.smtp.SMTPSendFailedException) nextException;
                        smtpInfo = getSmtpInfo(failedError);
                        returnCode = failedError.getReturnCode();
                        if (invalidAddresses == null || invalidAddresses.length == 0) {
                            invalidAddresses = failedError.getInvalidAddresses();
                        }
                    } else if (nextException instanceof com.sun.mail.smtp.SMTPSenderFailedException) {
                        com.sun.mail.smtp.SMTPSenderFailedException failedError = (com.sun.mail.smtp.SMTPSenderFailedException) nextException;
                        smtpInfo = getSmtpInfo(failedError);
                        returnCode = failedError.getReturnCode();
                        if (invalidAddresses == null || invalidAddresses.length == 0) {
                            invalidAddresses = failedError.getInvalidAddresses();
                        }
                    } else if (nextException instanceof com.sun.mail.smtp.SMTPAddressFailedException) {
                        com.sun.mail.smtp.SMTPAddressFailedException failedError = (com.sun.mail.smtp.SMTPAddressFailedException) nextException;
                        smtpInfo = getSmtpInfo(failedError);
                        returnCode = failedError.getReturnCode();
                        if (invalidAddresses == null || invalidAddresses.length == 0) {
                            invalidAddresses = failedError.getInvalidAddresses();
                        }
                    }
                }
                // Message too large?
                if ((returnCode == 552) || (toLowerCase(exc.getMessage(), "").indexOf(ERR_MSG_TOO_LARGE) > -1)) {
                    return MimeMailExceptionCode.MESSAGE_TOO_LARGE_EXT.create(exc, null == smtpInfo ? exc.getMessage() : smtpInfo);
                }
                // Others...
                if (null == invalidAddresses || invalidAddresses.length == 0) {
                    return MimeMailExceptionCode.SEND_FAILED_MSG_ERROR.create(exc, null == smtpInfo ? exc.getMessage() : smtpInfo);
                }
                return MimeMailExceptionCode.SEND_FAILED_EXT.create(exc, Arrays.toString(invalidAddresses), null == smtpInfo ? exc.getMessage() : smtpInfo);
            } else if (e instanceof javax.mail.StoreClosedException) {
                if (null != mailConfig && null != session) {
                    return MimeMailExceptionCode.STORE_CLOSED_EXT.create(e, mailConfig.getServer(), mailConfig.getLogin(), Integer.valueOf(session.getUserId()), Integer.valueOf(session.getContextId()), EMPTY_ARGS);
                }
                return MimeMailExceptionCode.STORE_CLOSED.create(e, EMPTY_ARGS);
            }
            final Exception nextException = e.getNextException();
            if (nextException == null) {
                if (toLowerCase(e.getMessage(), "").indexOf(ERR_QUOTA) != -1) {
                    return MimeMailExceptionCode.QUOTA_EXCEEDED.create(e, getInfo(skipTag(e.getMessage())));
                } else if ("Unable to load BODYSTRUCTURE".equals(e.getMessage())) {
                    return MimeMailExceptionCode.MESSAGE_NOT_DISPLAYED.create(e, EMPTY_ARGS);
                }
                /*
                 * Default case
                 */
                final String message = toLowerCase(e.getMessage());
                if ("failed to load imap envelope".equals(message)) {
                    return MimeMailExceptionCode.MESSAGE_NOT_DISPLAYED.create(e);
                }
                if ("connection failure".equals(e.getMessage())) {
                    return MimeMailExceptionCode.NO_ROUTE_TO_HOST.create(e, mailConfig == null ? STR_EMPTY : mailConfig.getServer());
                }
                return MimeMailExceptionCode.MESSAGING_ERROR.create(e, appendInfo(e.getMessage(), folder));
            }
            /*
             * Messaging exception has a nested exception
             */
            if (nextException instanceof java.net.BindException) {
                return MimeMailExceptionCode.BIND_ERROR.create(e, mailConfig == null ? STR_EMPTY : Integer.valueOf(mailConfig.getPort()));
            } else if (nextException instanceof com.sun.mail.iap.ConnectionException) {
                mailInterfaceMonitor.changeNumBrokenConnections(true);
                return MimeMailExceptionCode.CONNECT_ERROR.create(
                    e,
                    mailConfig == null ? STR_EMPTY : mailConfig.getServer(),
                        mailConfig == null ? STR_EMPTY : mailConfig.getLogin());
            } else if (nextException instanceof java.net.ConnectException) {
                /*
                 * Most modern IP stack implementations sense connection idleness, and abort the connection attempt, resulting in a
                 * java.net.ConnectionException
                 */
                mailInterfaceMonitor.changeNumTimeoutConnections(true);
                final OXException me =
                    MimeMailExceptionCode.CONNECT_ERROR.create(
                        e,
                        mailConfig == null ? STR_EMPTY : mailConfig.getServer(),
                            mailConfig == null ? STR_EMPTY : mailConfig.getLogin());
                return me;
            } else if (nextException.getClass().getName().endsWith(EXC_CONNECTION_RESET_EXCEPTION)) {
                mailInterfaceMonitor.changeNumBrokenConnections(true);
                return MimeMailExceptionCode.CONNECTION_RESET.create(e, new Object[0]);
            } else if (nextException instanceof java.net.NoRouteToHostException) {
                return MimeMailExceptionCode.NO_ROUTE_TO_HOST.create(e, mailConfig == null ? STR_EMPTY : mailConfig.getServer());
            } else if (nextException instanceof java.net.PortUnreachableException) {
                return MimeMailExceptionCode.PORT_UNREACHABLE.create(
                    e,
                    mailConfig == null ? STR_EMPTY : Integer.valueOf(mailConfig.getPort()));
            } else if (nextException instanceof java.net.SocketException) {
                /*
                 * Treat dependent on message
                 */
                final SocketException se = (SocketException) nextException;
                if ("Socket closed".equals(se.getMessage()) || "Connection reset".equals(se.getMessage())) {
                    mailInterfaceMonitor.changeNumBrokenConnections(true);
                    return MimeMailExceptionCode.BROKEN_CONNECTION.create(e, mailConfig == null ? STR_EMPTY : mailConfig.getServer());
                }
                return MimeMailExceptionCode.SOCKET_ERROR.create(e, new Object[0]);
            } else if (nextException instanceof java.net.UnknownHostException) {
                return MimeMailExceptionCode.UNKNOWN_HOST.create(e, appendInfo(e.getMessage(), folder));
            } else if (nextException instanceof java.net.SocketTimeoutException) {
                mailInterfaceMonitor.changeNumBrokenConnections(true);
                return MimeMailExceptionCode.CONNECT_ERROR.create(
                    e,
                    mailConfig == null ? STR_EMPTY : mailConfig.getServer(),
                        mailConfig == null ? STR_EMPTY : mailConfig.getLogin());
            } else if (nextException instanceof com.openexchange.mail.mime.QuotaExceededException) {
                if (null != mailConfig && null != session) {
                    return MimeMailExceptionCode.QUOTA_EXCEEDED_EXT.create(
                        nextException,
                        mailConfig.getServer(),
                        mailConfig.getLogin(),
                        Integer.valueOf(session.getUserId()),
                        Integer.valueOf(session.getContextId()),
                        appendInfo(getInfo(skipTag(nextException.getMessage())), folder));
                }
                return MimeMailExceptionCode.QUOTA_EXCEEDED.create(nextException, appendInfo(getInfo(skipTag(nextException.getMessage())), folder));
            } else if (nextException instanceof com.sun.mail.iap.CommandFailedException) {
                // Check for in-use error
                final String msg = toLowerCase(nextException.getMessage());
                if (isInUseException(msg)) {
                    // Too many sessions in use
                    if (null != mailConfig && null != session) {
                        return MimeMailExceptionCode.IN_USE_ERROR_EXT.create(
                            nextException,
                            mailConfig.getServer(),
                            mailConfig.getLogin(),
                            Integer.valueOf(session.getUserId()),
                            Integer.valueOf(session.getContextId()),
                            appendInfo(getInfo(skipTag(nextException.getMessage())), folder));
                    }
                    return MimeMailExceptionCode.IN_USE_ERROR.create(nextException, appendInfo(getInfo(skipTag(nextException.getMessage())), folder));
                }
                if (isOverQuotaException(msg)) {
                    // Over quota
                    if (null != mailConfig && null != session) {
                        return MimeMailExceptionCode.QUOTA_EXCEEDED_EXT.create(
                            nextException,
                            mailConfig.getServer(),
                            mailConfig.getLogin(),
                            Integer.valueOf(session.getUserId()),
                            Integer.valueOf(session.getContextId()),
                            appendInfo(getInfo(skipTag(nextException.getMessage())), folder));
                    }
                    return MimeMailExceptionCode.QUOTA_EXCEEDED.create(nextException, appendInfo(getInfo(skipTag(nextException.getMessage())), folder));
                }
                // Regular processing error cause by arbitrary CommandFailedException
                if (null != mailConfig && null != session) {
                    return MimeMailExceptionCode.PROCESSING_ERROR_WE_EXT.create(
                        nextException,
                        mailConfig.getServer(),
                        mailConfig.getLogin(),
                        Integer.valueOf(session.getUserId()),
                        Integer.valueOf(session.getContextId()),
                        appendInfo(getInfo(skipTag(nextException.getMessage())), folder));
                }
                return MimeMailExceptionCode.PROCESSING_ERROR_WE.create(nextException, appendInfo(getInfo(skipTag(nextException.getMessage())), folder));
            } else if (nextException instanceof com.sun.mail.iap.BadCommandException) {
                Category category = MimeMailExceptionCode.PROCESSING_ERROR.getCategory();
                if (toLowerCase(e.getMessage()).indexOf("[inuse]") >= 0) {
                    category = CATEGORY_USER_INPUT;
                }
                if (null != mailConfig && null != session) {
                    return MimeMailExceptionCode.PROCESSING_ERROR_EXT.create(
                        nextException,
                        mailConfig.getServer(),
                        mailConfig.getLogin(),
                        Integer.valueOf(session.getUserId()),
                        Integer.valueOf(session.getContextId()),
                        appendInfo(nextException.getMessage(), folder)).setCategory(category);
                }
                return MimeMailExceptionCode.PROCESSING_ERROR.create(nextException, appendInfo(nextException.getMessage(), folder)).setCategory(category);
            } else if (nextException instanceof com.sun.mail.iap.ProtocolException) {
                Category category = MimeMailExceptionCode.PROCESSING_ERROR.getCategory();
                if (toLowerCase(e.getMessage()).indexOf("[inuse]") >= 0) {
                    category = CATEGORY_USER_INPUT;
                }
                if (null != mailConfig && null != session) {
                    return MimeMailExceptionCode.PROCESSING_ERROR_EXT.create(
                        nextException,
                        mailConfig.getServer(),
                        mailConfig.getLogin(),
                        Integer.valueOf(session.getUserId()),
                        Integer.valueOf(session.getContextId()),
                        appendInfo(nextException.getMessage(), folder)).setCategory(category);
                }
                return MimeMailExceptionCode.PROCESSING_ERROR.create(nextException, appendInfo(nextException.getMessage(), folder)).setCategory(category);
            } else if (nextException instanceof java.io.IOException) {
                if (null != mailConfig && null != session) {
                    return MimeMailExceptionCode.IO_ERROR_EXT.create(
                        nextException,
                        appendInfo(nextException.getMessage(), folder),
                        mailConfig.getServer(),
                        mailConfig.getLogin(),
                        Integer.valueOf(session.getUserId()),
                        Integer.valueOf(session.getContextId()));
                }
                return MimeMailExceptionCode.IO_ERROR.create(nextException, appendInfo(nextException.getMessage(), folder));
            } else if (toLowerCase(e.getMessage(), "").indexOf(ERR_QUOTA) != -1) {
                return MimeMailExceptionCode.QUOTA_EXCEEDED.create(e, getInfo(skipTag(e.getMessage())));
            }
            /*
             * Default case
             */
            return MimeMailExceptionCode.MESSAGING_ERROR.create(nextException, appendInfo(nextException.getMessage(), folder));
        } catch (final Throwable t) {
            ExceptionUtils.handleThrowable(t);
            LOG.warn("", t);
            /*
             * This routine should not fail since it's purpose is wrap a corresponding mail error around specified messaging error
             */
            return MimeMailExceptionCode.MESSAGING_ERROR.create(e, appendInfo(e.getMessage(), folder));
        }
    }

    /**
     * Appends command information to given information string.
     *
     * @param info The information
     * @param folder The optional folder
     * @return The command with optional information appended
     */
    public static String appendInfo(final String info, final Folder folder) {
        if (null == folder) {
            return info;
        }
        final StringBuilder sb = null == info ? new StringBuilder(64) : new StringBuilder(info);
        sb.append(" (folder=\"").append(folder.getFullName()).append('"');
        final Store store = folder.getStore();
        if (null != store) {
            sb.append(", store=\"").append(store.toString()).append('"');
        }
        sb.append(')');
        return sb.toString();
    }

    private static String getInfo(final String info) {
        if (null == info) {
            return info;
        }
        final int pos = toLowerCase(info).indexOf("error message: ");
        return pos < 0 ? info : info.substring(pos + 15);
    }

    private static final Pattern PATTERN_TAG = Pattern.compile("A[0-9]+ (.+)");

    private static String skipTag(final String serverResponse) {
        if (null == serverResponse) {
            return null;
        }
        final Matcher m = PATTERN_TAG.matcher(serverResponse);
        if (m.matches()) {
            return m.group(1);
        }
        return serverResponse;
    }

    private static <E> E lookupNested(final MessagingException e, final Class<E> clazz) {
        if (null == e) {
            return null;
        }

        Exception exception = e.getNextException();
        if (clazz.isInstance(exception)) {
            return clazz.cast(exception);
        }
        return exception instanceof MessagingException ? lookupNested((MessagingException) exception, clazz) : null;
    }

    /**
     * Checks for possible already-exists error.
     */
    public static boolean isAlreadyExistsException(final MessagingException e) {
        if (null == e) {
            return false;
        }
        return isAlreadyExistsException(e.getMessage());
    }

    /**
     * Checks for possible already-exists error.
     */
    public static boolean isAlreadyExistsException(final String msg) {
        if (null == msg) {
            return false;
        }
        final String m = toLowerCase(msg);
        return (m.indexOf("alreadyexists") >= 0);
    }

    /**
     * Checks for possible over-quota error.
     */
    public static boolean isOverQuotaException(final MessagingException e) {
        if (null == e) {
            return false;
        }
        return isOverQuotaException(e.getMessage());
    }

    /**
     * Checks for possible over-quota error.
     */
    public static boolean isOverQuotaException(String msg) {
        if (null == msg) {
            return false;
        }
        final String m = toLowerCase(msg);
        return (m.indexOf("quota") >= 0 || m.indexOf("limit") >= 0);
    }

    /**
     * Checks for possible in-use error.
     */
    public static boolean isInUseException(MessagingException e) {
        if (null == e) {
            return false;
        }
        return isInUseException(toLowerCase(e.getMessage()));
    }

    /**
     * Checks for possible in-use error.
     */
    public static boolean isInUseException(final String msg) {
        if (null == msg) {
            return false;
        }
        return (toLowerCase(msg).indexOf("[inuse]") >= 0);
    }

    /**
     * Checks for possible command-failed error.
     */
    public static boolean isCommandFailedException(MessagingException e) {
        if (null == e) {
            return false;
        }
        CommandFailedException commandFailedError = lookupNested(e, com.sun.mail.iap.CommandFailedException.class);
        return null != commandFailedError;
    }

    private static String getSmtpInfo(SMTPSendFailedException sendFailedError) {
        return null == sendFailedError ? "" : new StringBuilder(64).append(sendFailedError.getReturnCode()).append(" - ").append(sendFailedError.getMessage()).toString();
    }

    private static String getSmtpInfo(SMTPAddressFailedException sendFailedError) {
        return null == sendFailedError ? "" : new StringBuilder(64).append(sendFailedError.getReturnCode()).append(" - ").append(sendFailedError.getMessage()).toString();
    }

    private static String getSmtpInfo(SMTPSenderFailedException sendFailedError) {
        return null == sendFailedError ? "" : new StringBuilder(64).append(sendFailedError.getReturnCode()).append(" - ").append(sendFailedError.getMessage()).toString();
    }

    /** ASCII-wise to lower-case */
    private static String toLowerCase(final CharSequence chars) {
        return toLowerCase(chars, null);
    }

    /** ASCII-wise to lower-case */
    private static String toLowerCase(final CharSequence chars, final String defaultValue) {
        if (null == chars) {
            return defaultValue;
        }
        final int length = chars.length();
        final StringBuilder builder = new StringBuilder(length);
        for (int i = 0; i < length; i++) {
            final char c = chars.charAt(i);
            builder.append((c >= 'A') && (c <= 'Z') ? (char) (c ^ 0x20) : c);
        }
        return builder.toString();
    }

}
