package org.javasimon.jdbcx4;

import org.javasimon.jdbc4.SimonConnection;

import javax.sql.DataSource;
import java.sql.*;
import java.sql.Connection;
import java.lang.reflect.Method;

import org.javasimon.jdbc4.WrapperSupport;

/**
 * Wrapper class for real DataSource implementation, produces standard {@link java.sql.Connection}
 * object.
 * <p/>
 * To use SimonDataSource, <b>MUST</b> properties are:
 * <ul>
 * <li><code>realDataSourceClassName</code> - full qualified classname of real Datasource
 * implementation</li>
 * <li><code>url</code> - JDBC connection URL (no special Simon added tags are needed)</li>
 * <li><code>user</code> - DB user name</li>
 * <li><code>password</code> - DB user name</li>
 * </ul>
 * <b>MAY</b> properties are:
 * <ul>
 * <li><code>prefix</code> - Simon prefix (default: <code>org.javasimon.jdbcx4</code></li>
 * </ul>
 * <p/>
 * As mentioned in package description all <code>getConnection</code> methods
 * just invokes real {@link javax.sql.DataSource} object methods and wraps obtained
 * {@link java.sql.Connection} with {@link org.javasimon.jdbc4.SimonConnection} object.
 * <p/>
 * Real {@link javax.sql.DataSource} is obtained in method {@link #datasource()}. It tries
 * to instantiate real datasource object by property <code>realDataSourceClassName</code>
 * (setters and getters for properties are in {@link AbstractSimonDataSource}) and then sets
 * basic properties (<code>url</code>, <code>user</code>, <code>password</code>).
 *
 * @author Radovan Sninsky
 * @author <a href="mailto:virgo47@gmail.com">Richard "Virgo" Richter</a>
 * @since 2.4
 */
public final class SimonDataSource extends AbstractSimonDataSource implements DataSource {
	private DataSource ds;
	private WrapperSupport<DataSource> wrapperSupport;

	private DataSource datasource() throws SQLException {
		if (ds == null) {
			if (realDataSourceClassName == null || realDataSourceClassName.length() == 0) {
				throw new SQLException("Property realDataSourceClassName is not set");
			}
			Object o;
			try {
				o = Class.forName(realDataSourceClassName).newInstance();
			} catch (Exception e) {
				throw new SQLException(e);
			}
			if (o instanceof DataSource) {
				ds = (DataSource) o;
				try {
					for (Method m : ds.getClass().getMethods()) {
						String methodName = m.getName();
						if (methodName.equalsIgnoreCase("setUser")) {
							m.invoke(ds, user);
						} else if (methodName.equalsIgnoreCase("setPassword")) {
							m.invoke(ds, password);
						} else if (methodName.equalsIgnoreCase("setUrl")) {
							m.invoke(ds, url);
						}
					}
				} catch (Exception e) {
					throw new SQLException(e);
				}
				ds.setLogWriter(logWriter);
				ds.setLoginTimeout(loginTimeout);
				wrapperSupport = new WrapperSupport<DataSource>(ds, DataSource.class);
			} else {
				throw new SQLException("Class in realDataSourceClassName is not a DataSource");
			}
		}
		return ds;
	}

	/**
	 * Attempts to establish a connection with the data source that this {@code DataSource} object represents.
	 *
	 * @return a connection to the data source
	 * @throws java.sql.SQLException if a database access error occurs
	 */
	@Override
	public Connection getConnection() throws SQLException {
		return new SimonConnection(datasource().getConnection(), prefix);
	}

	/**
	 * Attempts to establish a connection with the data source that this {@code DataSource} object represents.
	 *
	 * @param user the database user on whose behalf the connection is being made
	 * @param password the user's password
	 * @return a connection to the data source
	 * @throws java.sql.SQLException if a database access error occurs
	 */
	@Override
	public Connection getConnection(String user, String password) throws SQLException {
		return new SimonConnection(datasource().getConnection(user, password), prefix);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public <T> T unwrap(Class<T> iface) throws SQLException {
		return wrapperSupport.unwrap(iface);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isWrapperFor(Class<?> iface) throws SQLException {
		return wrapperSupport.isWrapperFor(iface);
	}
}
