/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.user.json.actions;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXActionServiceFactory;
import com.openexchange.documentation.annotations.Module;
import com.openexchange.exception.OXException;
import com.openexchange.tools.servlet.AjaxExceptionCodes;

/**
 * {@link UserActionFactory} - Factory for user component.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
@Module(name = "user", description = "Provides access to user information.")
public final class UserActionFactory implements AJAXActionServiceFactory {

    /**
     * The singleton instance.
     */
    private static final UserActionFactory SINGLETON = new UserActionFactory();

    /**
     * Gets the {@link UserActionFactory factory} instance.
     *
     * @return The {@link UserActionFactory factory} instance.
     */
    public static final UserActionFactory getInstance() {
        return SINGLETON;
    }

    /*-
     * Member section
     */

    /**
     * The map to store actions.
     */
    private final Map<String, AJAXActionService> actions;

    /**
     * Initializes a new {@link UserActionFactory}.
     */
    private UserActionFactory() {
        super();
        actions = initActions();
    }

    @Override
    public AJAXActionService createActionService(final String action) throws OXException {
        if (null == action) {
            throw AjaxExceptionCodes.UNKNOWN_ACTION.create( action);
        }
        final AJAXActionService retval = actions.get(action);
        if (null == retval) {
            throw AjaxExceptionCodes.UNKNOWN_ACTION.create( action);
        }
        return retval;
    }

    @Override
    public Collection<? extends AJAXActionService> getSupportedServices() {
        return java.util.Collections.unmodifiableCollection(actions.values());
    }

    /**
     * Initializes the unmodifiable map to stored actions.
     *
     * @return The unmodifiable map with actions stored
     */
    private Map<String, AJAXActionService> initActions() {
        final Map<String, AJAXActionService> tmp = new HashMap<String, AJAXActionService>(12);
        tmp.put(GetAction.ACTION, new GetAction());
        tmp.put(ListAction.ACTION, new ListAction());
        tmp.put(AllAction.ACTION, new AllAction());
        tmp.put(SearchAction.ACTION, new SearchAction());
        tmp.put(UpdateAction.ACTION, new UpdateAction());
        tmp.put(GetAttributeAction.ACTION, new GetAttributeAction());
        tmp.put(SetAttributeAction.ACTION, new SetAttributeAction());
        return Collections.unmodifiableMap(tmp);
    }

}
