/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.database.internal;

import static com.openexchange.database.internal.DBUtils.closeSQLStuff;
import static com.openexchange.java.Autoboxing.I;
import java.io.Serializable;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.caching.Cache;
import com.openexchange.caching.CacheKey;
import com.openexchange.caching.CacheService;
import com.openexchange.database.Assignment;
import com.openexchange.database.AssignmentInsertData;
import com.openexchange.database.ConfigDatabaseService;
import com.openexchange.database.DBPoolingExceptionCodes;
import com.openexchange.database.Databases;
import com.openexchange.exception.OXException;
import gnu.trove.list.TIntList;
import gnu.trove.list.linked.TIntLinkedList;

/**
 * Reads assignments from the database, maybe stores them in a cache for faster access.
 *
 * @author <a href="mailto:marcus@open-xchange.org">Marcus Klein</a>
 */
public final class ContextDatabaseAssignmentImpl implements ContextDatabaseAssignmentService {

    private static final Logger LOG = LoggerFactory.getLogger(ContextDatabaseAssignmentImpl.class);

    private static final String SELECT = "SELECT read_db_pool_id,write_db_pool_id,db_schema FROM context_server2db_pool WHERE server_id=? AND cid=?";
    private static final String INSERT = "INSERT INTO context_server2db_pool (read_db_pool_id,write_db_pool_id,db_schema,server_id,cid) VALUES (?,?,?,?,?)";
    private static final String UPDATE = "UPDATE context_server2db_pool SET read_db_pool_id=?,write_db_pool_id=?,db_schema=? WHERE server_id=? AND cid=?";
    private static final String DELETE = "DELETE FROM context_server2db_pool WHERE cid=? AND server_id=?";
    private static final String CONTEXTS_IN_SCHEMA = "SELECT cid FROM context_server2db_pool WHERE server_id=? AND write_db_pool_id=? AND db_schema=?";
    private static final String CONTEXTS_IN_DATABASE = "SELECT cid FROM context_server2db_pool WHERE read_db_pool_id=? OR write_db_pool_id=?";
    private static final String NOTFILLED = "SELECT schemaname,count FROM contexts_per_dbschema WHERE db_pool_id=? AND count<? ORDER BY count ASC";
    private final ConfigDatabaseService configDatabaseService;

    private static final String CACHE_NAME = "OXDBPoolCache";

    private volatile CacheService cacheService;

    private volatile Cache cache;

    /**
     * Lock for the cache.
     */
    private final Lock cacheLock = new ReentrantLock(true);

    private final LockMech lockMech;

    /**
     * Default constructor.
     */
    public ContextDatabaseAssignmentImpl(final ConfigDatabaseService configDatabaseService, LockMech lockMech) {
        super();
        this.configDatabaseService = configDatabaseService;
        this.lockMech = lockMech;
    }

    @Override
    public AssignmentImpl getAssignment(final int contextId) throws OXException {
        return getAssignment(null, contextId, true);
    }

    private AssignmentImpl getAssignment(Connection con, int contextId, boolean errorOnAbsence) throws OXException {
        CacheService myCacheService = this.cacheService;
        Cache myCache = this.cache;

        // Check cache references
        if (null == myCache || null == myCacheService) {
            // No cache available
            return loadAssignment(con, contextId, errorOnAbsence);
        }

        // Use that cache
        CacheKey key = myCacheService.newCacheKey(contextId, Server.getServerId());
        Object object = myCache.get(key);
        if (object instanceof AssignmentImpl) {
            return (AssignmentImpl) object;
        }

        // Need to load - synchronously!
        cacheLock.lock();
        try {
            AssignmentImpl retval = (AssignmentImpl) myCache.get(key);
            if (null == retval) {
                retval = loadAssignment(con, contextId, errorOnAbsence);
                try {
                    myCache.putSafe(key, retval);
                } catch (OXException e) {
                    LOG.error("Cannot put database assignment into cache.", e);
                }
            }
            return retval;
        } finally {
            cacheLock.unlock();
        }
    }

    private static AssignmentImpl loadAssignment(Connection con, int contextId) throws OXException {
        PreparedStatement stmt = null;
        ResultSet result = null;
        try {
            stmt = con.prepareStatement(SELECT);
            stmt.setInt(1, Server.getServerId());
            stmt.setInt(2, contextId);
            result = stmt.executeQuery();

            if (false == result.next()) {
                return null;
            }
            int pos = 1;
            return new AssignmentImpl(contextId, Server.getServerId(), result.getInt(pos++), result.getInt(pos++), result.getString(pos++));
        } catch (final SQLException e) {
            throw DBPoolingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            closeSQLStuff(result, stmt);
        }
    }

    private AssignmentImpl loadAssignment(Connection conn, int contextId, boolean errorOnAbsence) throws OXException {
        Connection con = conn;
        if (null == con) {
            con = configDatabaseService.getReadOnly();
            try {
                return loadAndCheck(con, contextId, errorOnAbsence);
            } finally {
                configDatabaseService.backReadOnly(con);
            }
        }

        return loadAndCheck(con, contextId, errorOnAbsence);
    }

    private AssignmentImpl loadAndCheck(Connection con, int contextId, boolean errorOnAbsence) throws OXException {
        AssignmentImpl retval = loadAssignment(con, contextId);
        if (errorOnAbsence && null == retval) {
            throw DBPoolingExceptionCodes.RESOLVE_FAILED.create(I(contextId), I(Server.getServerId()));
        }
        return retval;
    }

    private static void writeAssignmentDB(Connection con, Assignment assign, AssignmentImpl oldAssign) throws OXException {
        PreparedStatement stmt = null;
        try {
            stmt = con.prepareStatement(null == oldAssign ? INSERT : UPDATE);
            int pos = 1;
            stmt.setInt(pos++, assign.getReadPoolId());
            stmt.setInt(pos++, assign.getWritePoolId());
            stmt.setString(pos++, assign.getSchema());
            stmt.setInt(pos++, assign.getServerId());
            stmt.setInt(pos++, assign.getContextId());
            int count = stmt.executeUpdate();
            if (1 != count) {
                throw DBPoolingExceptionCodes.INSERT_FAILED.create(I(assign.getContextId()), I(assign.getServerId()));
            }
            Databases.closeSQLStuff(stmt);

            if (null == oldAssign) {
                updateCountTables(con, assign.getWritePoolId(), assign.getSchema(), true);
            } else {
                int oldPoolId = oldAssign.getWritePoolId();
                int newPoolId = assign.getWritePoolId();

                if (oldPoolId != newPoolId) {
                    updateCountTables(con, oldPoolId, oldAssign.getSchema(), false);
                    updateCountTables(con, newPoolId, assign.getSchema(), true);
                } else {
                    String oldSchema = oldAssign.getSchema();
                    String newSchema = assign.getSchema();
                    if (false == oldSchema.equals(newSchema)) {
                        updateSchemaCountTable(con, oldPoolId, oldSchema, false);
                        updateSchemaCountTable(con, oldPoolId, newSchema, true);
                    }
                }
            }
        } catch (SQLException e) {
            throw DBPoolingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            closeSQLStuff(stmt);
        }
    }

    private static void updateCountTables(Connection con, int poolId, String schemaName, boolean increment) throws SQLException {
        PreparedStatement stmt = null;
        try {
            stmt = con.prepareStatement("UPDATE contexts_per_dbpool SET count=count" + (increment ? '+' : '-') + "1 WHERE db_pool_id=?");
            stmt.setInt(1, poolId);
            stmt.executeUpdate();
            Databases.closeSQLStuff(stmt);
            stmt = null;

            updateSchemaCountTable(con, poolId, schemaName, increment);
        } finally {
            closeSQLStuff(stmt);
        }
    }

    private static void updateSchemaCountTable(Connection con, int poolId, String schemaName, boolean increment) throws SQLException {
        PreparedStatement stmt = null;
        try {
            stmt = con.prepareStatement("UPDATE contexts_per_dbschema SET count=count" + (increment ? '+' : '-') + "1 WHERE db_pool_id=? AND schemaname=?");
            stmt.setInt(1, poolId);
            stmt.setString(2, schemaName);
            stmt.executeUpdate();
        } finally {
            closeSQLStuff(stmt);
        }
    }

    @Override
    public void writeAssignment(Connection con, Assignment assign) throws OXException {
        AssignmentImpl oldAssign = assign instanceof AssignmentInsertData ? null : getAssignment(con, assign.getContextId(), false);
        Cache myCache = this.cache;
        if (null != myCache) {
            final CacheKey key = myCache.newCacheKey(assign.getContextId(), assign.getServerId());
            cacheLock.lock();
            try {
                if (null != oldAssign) {
                    myCache.remove(key);
                }
                try {
                    myCache.putSafe(key, new AssignmentImpl(assign));
                } catch (OXException e) {
                    LOG.error("Cannot put database assignment into cache.", e);
                }
            } finally {
                cacheLock.unlock();
            }
        }
        writeAssignmentDB(con, assign, oldAssign);
    }

    private void deleteAssignmentDB(Connection con, int contextId) throws OXException {
        AssignmentImpl assignment = getAssignment(con, contextId, false);

        PreparedStatement stmt = null;
        try {
            stmt = con.prepareStatement(DELETE);
            stmt.setInt(1, contextId);
            stmt.setInt(2, Server.getServerId());
            stmt.executeUpdate();
            closeSQLStuff(stmt);

            updateCountTables(con, assignment.getWritePoolId(), assignment.getSchema(), false);
        } catch (SQLException e) {
            throw DBPoolingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            closeSQLStuff(stmt);
        }
    }

    @Override
    public void invalidateAssignment(int... contextIds) {
        Cache myCache = this.cache;
        if (null != myCache) {
            try {
                int serverId = Server.getServerId();
                if (contextIds != null && contextIds.length > 0) {
                    List<Serializable> keys = new ArrayList<Serializable>(contextIds.length);
                    for (int contextId : contextIds) {
                        keys.add(myCache.newCacheKey(contextId, serverId));
                    }
                    myCache.remove(keys);
                }
            } catch (final OXException e) {
                LOG.error("Error while removing database assignment from cache.", e);
            }
        }
    }

    @Override
    public void deleteAssignment(Connection con, int contextId) throws OXException {
        deleteAssignmentDB(con, contextId);
        invalidateAssignment(contextId);
    }

    @Override
    public int[] getContextsFromSchema(Connection con, int writePoolId, String schema) throws OXException {
        PreparedStatement stmt = null;
        ResultSet rs = null;
        try {
            stmt = con.prepareStatement(CONTEXTS_IN_SCHEMA);
            stmt.setInt(1, Server.getServerId());
            stmt.setInt(2, writePoolId);
            stmt.setString(3, schema);
            rs = stmt.executeQuery();
            final TIntList tmp = new TIntLinkedList();
            while (rs.next()) {
                tmp.add(rs.getInt(1));
            }
            return tmp.toArray();
        } catch (SQLException e) {
            throw DBPoolingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            closeSQLStuff(rs, stmt);
        }
    }

    private static int[] listContexts(Connection con, int poolId) throws OXException {
        final List<Integer> tmp = new LinkedList<Integer>();
        PreparedStatement stmt = null;
        ResultSet result = null;
        try {
            stmt = con.prepareStatement(CONTEXTS_IN_DATABASE);
            stmt.setInt(1, poolId);
            stmt.setInt(2, poolId);
            result = stmt.executeQuery();
            while (result.next()) {
                tmp.add(I(result.getInt(1)));
            }
        } catch (SQLException e) {
            throw DBPoolingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            closeSQLStuff(result, stmt);
        }
        final int[] retval = new int[tmp.size()];
        for (int i = 0; i < tmp.size(); i++) {
            retval[i] = tmp.get(i).intValue();
        }
        return retval;
    }

    @Override
    public int[] getContextsInDatabase(int poolId) throws OXException {
        final Connection con = configDatabaseService.getReadOnly();
        try {
            return listContexts(con, poolId);
        } finally {
            configDatabaseService.backReadOnly(con);
        }
    }

    @Override
    public String[] getUnfilledSchemas(Connection con, int poolId, int maxContexts) throws OXException {
        PreparedStatement stmt = null;
        ResultSet result = null;
        try {
            stmt = con.prepareStatement(NOTFILLED);
            stmt.setInt(1, poolId);
            stmt.setInt(2, maxContexts);
            result = stmt.executeQuery();
            List<String> retval = new LinkedList<String>();
            while (result.next()) {
                String schema = result.getString(1);
                LOG.debug("schema {} is filled with {} contexts.", schema, I(result.getInt(2)));
                retval.add(schema);
            }
            return retval.toArray(new String[retval.size()]);
        } catch (final SQLException e) {
            throw DBPoolingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            closeSQLStuff(result, stmt);
        }
    }

    @Override
    public Map<String, Integer> getContextCountPerSchema(Connection con, int poolId, int maxContexts) throws OXException {
        PreparedStatement stmt = null;
        ResultSet result = null;
        try {
            stmt = con.prepareStatement(NOTFILLED);
            stmt.setInt(1, poolId);
            stmt.setInt(2, maxContexts);
            result = stmt.executeQuery();
            Map<String, Integer> retval = new LinkedHashMap<String, Integer>(32, 0.9F);
            while (result.next()) {
                String schema = result.getString(1);
                Integer count = I(result.getInt(2));
                LOG.debug("schema {} is filled with {} contexts.", schema, count);
                retval.put(schema, count);
            }
            return retval;
        } catch (final SQLException e) {
            throw DBPoolingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            closeSQLStuff(result, stmt);
        }
    }

    @Override
    public void lock(Connection con, int writePoolId) throws OXException {
        PreparedStatement stmt = null;
        try {
            switch (lockMech) {
            case GLOBAL_LOCK: {
                stmt = con.prepareStatement("UPDATE ctx_per_schema_sem SET id=id+1");
                stmt.executeUpdate();
                break;
            }
            case ROW_LOCK: {
                stmt = con.prepareStatement("SELECT 1 FROM dbpool_lock WHERE db_pool_id=? FOR UPDATE");
                stmt.setInt(1, writePoolId);
                stmt.executeQuery();
                break;
            }
            }
        } catch (SQLException e) {
            throw DBPoolingExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            closeSQLStuff(stmt);
        }
    }

    void setCacheService(final CacheService service) {
        this.cacheService = service;
        try {
            this.cache = service.getCache(CACHE_NAME);
        } catch (final OXException e) {
            LOG.error("", e);
        }
    }

    void removeCacheService() {
        this.cacheService = null;
        Cache myCache = this.cache;
        if (null != myCache) {
            try {
                myCache.clear();
            } catch (final OXException e) {
                LOG.error("", e);
            }
            this.cache = null;
        }
    }
}
