/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.tools.images.transformations;

import static com.openexchange.tools.images.ImageTransformationUtility.canRead;
import static com.openexchange.tools.images.ImageTransformationUtility.getImageFormat;
import static com.openexchange.tools.images.ImageTransformationUtility.getImageInformation;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.ImageOutputStream;
import com.drew.imaging.ImageMetadataReader;
import com.drew.imaging.ImageProcessingException;
import com.drew.metadata.Metadata;
import com.openexchange.ajax.container.IFileHolder;
import com.openexchange.ajax.container.ThresholdFileHolder;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.Reloadable;
import com.openexchange.exception.OXException;
import com.openexchange.java.Streams;
import com.openexchange.tools.images.Constants;
import com.openexchange.tools.images.ImageInformation;
import com.openexchange.tools.images.ImageTransformationReloadable;
import com.openexchange.tools.images.ImageTransformationSignaler;
import com.openexchange.tools.images.ImageTransformationUtility;
import com.openexchange.tools.images.ImageTransformations;
import com.openexchange.tools.images.ScaleType;
import com.openexchange.tools.images.TransformedImage;
import com.openexchange.tools.images.osgi.Services;
import com.openexchange.tools.stream.CountingInputStream;
import com.openexchange.tools.stream.CountingInputStream.IOExceptionCreator;
import com.openexchange.tools.stream.UnsynchronizedByteArrayOutputStream;

/**
 * {@link ImageTransformationsImpl}
 *
 * Default {@link ImageTransformations} implementation.
 *
 * @author <a href="mailto:martin.herfurth@open-xchange.com">Martin Herfurth</a>
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 */
public class ImageTransformationsImpl implements ImageTransformations {

    private static org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(ImageTransformationsImpl.class);

    private static volatile Integer waitTimeoutSeconds;
    static int waitTimeoutSeconds() {
        Integer tmp = waitTimeoutSeconds;
        if (null == tmp) {
            synchronized (ImageTransformationsTask.class) {
                tmp = waitTimeoutSeconds;
                if (null == tmp) {
                    int defaultValue = 10;
                    ConfigurationService configService = Services.getService(ConfigurationService.class);
                    if (null == configService) {
                        return defaultValue;
                    }
                    tmp = Integer.valueOf(configService.getIntProperty("com.openexchange.tools.images.transformations.waitTimeoutSeconds", defaultValue));
                    waitTimeoutSeconds = tmp;
                }
            }
        }
        return tmp.intValue();
    }

    private static volatile Long maxSize;
    static long maxSize() {
        Long tmp = maxSize;
        if (null == tmp) {
            synchronized (ImageTransformationsTask.class) {
                tmp = maxSize;
                if (null == tmp) {
                    int defaultValue = 5242880; // 5 MB
                    ConfigurationService configService = Services.getService(ConfigurationService.class);
                    if (null == configService) {
                        return defaultValue;
                    }
                    tmp = Long.valueOf(configService.getIntProperty("com.openexchange.tools.images.transformations.maxSize", defaultValue));
                    maxSize = tmp;
                }
            }
        }
        return tmp.longValue();
    }

    private static volatile Long maxResolution;
    static long maxResolution() {
        Long tmp = maxResolution;
        if (null == tmp) {
            synchronized (ImageTransformationsTask.class) {
                tmp = maxResolution;
                if (null == tmp) {
                    int defaultValue = 12087962; // 4064 x 2704 (11.1 megapixels) + 10%
                    ConfigurationService configService = Services.getService(ConfigurationService.class);
                    if (null == configService) {
                        return defaultValue;
                    }
                    tmp = Long.valueOf(configService.getIntProperty("com.openexchange.tools.images.transformations.maxResolution", defaultValue));
                    maxResolution = tmp;
                }
            }
        }
        return tmp.longValue();
    }

    static {
        ImageTransformationReloadable.getInstance().addReloadable(new Reloadable() {

            @Override
            public void reloadConfiguration(ConfigurationService configService) {
                waitTimeoutSeconds = null;
                maxSize = null;
                maxResolution = null;
            }

            @Override
            public Map<String, String[]> getConfigFileNames() {
                return null;
            }
        });
    }

    private static final IOExceptionCreator IMAGE_SIZE_EXCEEDED_EXCEPTION_CREATOR = new IOExceptionCreator() {

        @Override
        public IOException createIOException(long total, long max) {
            if (total > 0 && max > 0) {
                return new ImageTransformationDeniedIOException(new StringBuilder("Image transformation denied. Size is too big. (current=").append(total).append(", max=").append(max).append(')').toString());
            }
            return new ImageTransformationDeniedIOException("Image transformation denied. Size is too big.");
        }
    };

    private static IOException createResolutionExceededIOException(long maxResolution, int resolution) {
        return new ImageTransformationDeniedIOException(new StringBuilder("Image transformation denied. Resolution is too high. (current=").append(resolution).append(", max=").append(maxResolution).append(')').toString());
    }

    // ------------------------------------------------------------------------------------------------------------------------------ //

    private final TransformationContext transformationContext;
    private final InputStream sourceImageStream;
    private final IFileHolder sourceImageFile;
    private final List<ImageTransformation> transformations;
    private BufferedImage sourceImage;
    private Metadata metadata;
    private boolean compress;
    protected final Object optSource;

    private ImageTransformationsImpl(BufferedImage sourceImage, InputStream sourceImageStream, IFileHolder imageFile, Object optSource) {
        super();
        this.optSource = optSource;

        if (null == imageFile) {
            this.sourceImageStream = sourceImageStream;
            this.sourceImageFile = null;
        } else {
            if (imageFile.repetitive()) {
                this.sourceImageStream = null;
                this.sourceImageFile = imageFile;
            } else {
                try {
                    this.sourceImageStream = imageFile.getStream();
                    this.sourceImageFile = null;
                } catch (OXException e) {
                    throw new IllegalStateException(e.getMessage(), e);
                }
            }
        }

        this.sourceImage = sourceImage;
        this.transformations = new ArrayList<ImageTransformation>();
        this.transformationContext = new TransformationContext();
    }

    /**
     * Initializes a new {@link ImageTransformationsImpl}.
     *
     * @param sourceImage The source image
     * @param optSource The source for this invocation; if <code>null</code> calling {@link Thread} is referenced as source
     */
    public ImageTransformationsImpl(final BufferedImage sourceImage, final Object optSource) {
        this(sourceImage, null, null, optSource);
    }

    /**
     * Initializes a new {@link ImageTransformationsImpl}.
     *
     * @param sourceImage The source image
     * @param optSource The source for this invocation; if <code>null</code> calling {@link Thread} is referenced as source
     */
    public ImageTransformationsImpl(final InputStream sourceImageStream, final Object optSource) {
        this(null, sourceImageStream, null, optSource);
    }

    /**
     * Initializes a new {@link ImageTransformationsTask}.
     *
     * @param imageFile The image file
     * @param optSource The source for this invocation; if <code>null</code> calling {@link Thread} is referenced as source
     */
    public ImageTransformationsImpl(IFileHolder imageFile, Object optSource) {
        this(null, null, imageFile, optSource);
    }

    @Override
    public ImageTransformations rotate() {
        transformations.add(RotateTransformation.getInstance());
        return this;
    }

    @Override
    public ImageTransformations scale(int maxWidth, int maxHeight, ScaleType scaleType) {
        if (maxWidth > Constants.getMaxWidth()) {
            throw new IllegalArgumentException("Width " + maxWidth + " exceeds max. supported width " + Constants.getMaxWidth());
        }
        if (maxHeight > Constants.getMaxHeight()) {
            throw new IllegalArgumentException("Height " + maxHeight + " exceeds max. supported height " + Constants.getMaxHeight());
        }
        transformations.add(new ScaleTransformation(maxWidth, maxHeight, scaleType));
        return this;
    }

    @Override
    public ImageTransformations crop(int x, int y, int width, int height) {
        transformations.add(new CropTransformation(x, y, width, height));
        return this;
    }

    @Override
    public ImageTransformations compress() {
        this.compress = true;
        return this;
    }

    @Override
    public BufferedImage getImage() throws IOException {
        if (false == needsTransformation(null) && null != sourceImage) {
            return sourceImage;
        }
        // Get BufferedImage
        return getImage(null, null);
    }

    @Override
    public byte[] getBytes(String formatName) throws IOException {
        String imageFormat = getImageFormat(formatName);
        return innerGetBytes(imageFormat);
    }

    private byte[] innerGetBytes(final String imageFormat) throws IOException {
        return write(getImage(imageFormat, null), imageFormat);
    }

    @Override
    public InputStream getInputStream(String formatName) throws IOException {
        String imageFormat = getImageFormat(formatName);
        if (false == needsTransformation(imageFormat)) {
            // Nothing to do
            InputStream in = getFileStream(sourceImageFile);
            if (null != in) {
                return in;
            }
            in = sourceImageStream;
            if (null != in) {
                return in;
            }
        }
        // Perform transformations
        byte[] bytes = innerGetBytes(imageFormat);
        return null == bytes ? null : Streams.newByteArrayInputStream(bytes);
    }

    @Override
    public TransformedImage getTransformedImage(String formatName) throws IOException {
        String imageFormat = getImageFormat(formatName);
        BufferedImage bufferedImage = getImage(imageFormat, null);
        return writeTransformedImage(bufferedImage, imageFormat);
    }

    /**
     * Gets a value indicating whether the denoted format name leads to transformations or not.
     *
     * @param formatName The format name
     * @return <code>true</code>, if there are transformations for the target image format, <code>false</code>, otherwise
     */
    private boolean needsTransformation(String formatName) {
        if (false == canRead(formatName)) {
            return false;
        }
        for (ImageTransformation transformation : transformations) {
            if (transformation.supports(formatName)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Gets the resulting image after applying all transformations.
     *
     * @param formatName The image format to use, or <code>null</code> if not relevant
     * @param signaler The optional signaler or <code>null</code>
     * @return The transformed image
     * @throws IOException if an I/O error occurs
     */
    protected BufferedImage getImage(String formatName, ImageTransformationSignaler signaler) throws IOException {
        BufferedImage image = getSourceImage(formatName, signaler);

        if (null != image && image.getHeight() > 3 && image.getWidth() > 3) {
            ImageInformation imageInformation = null != this.metadata ? getImageInformation(this.metadata) : null;
            for (ImageTransformation transformation : transformations) {
                if (transformation.supports(formatName)) {
                    image = transformation.perform(image, transformationContext, imageInformation);
                }
            }
        }
        image = removeTransparencyIfNeeded(image, formatName);
        return image;
    }

    /**
     * Gets the source image, either from the supplied buffered image or the supplied stream, extracting image metadata as needed.
     *
     * @param formatName The format to use, e.g. "jpeg" or "tiff"
     * @param signaler The optional signaler or <code>null</code>
     * @return The source image
     * @throws IOException
     */
    private BufferedImage getSourceImage(String formatName, ImageTransformationSignaler signaler) throws IOException {
        if (null == sourceImage) {
            if (null != sourceImageStream) {
                long maxSize = maxSize();
                long maxResolution = maxResolution();
                sourceImage = needsMetadata(formatName, maxSize, maxResolution) ? readAndExtractMetadataFromStream(sourceImageStream, formatName, maxSize, maxResolution, signaler) : read(sourceImageStream, formatName, signaler);
            } else if (null != sourceImageFile) {
                long maxSize = maxSize();
                long maxResolution = maxResolution();
                sourceImage = needsMetadata(formatName, maxSize, maxResolution) ? readAndExtractMetadataFromFile(sourceImageFile, formatName, maxSize, maxResolution, signaler) : read(getFileStream(sourceImageFile), formatName, signaler);
            }
        }
        return sourceImage;
    }

    /**
     * Gets a value indicating whether additional metadata is required for one of the transformations or not.
     *
     * @param formatName The format to use, e.g. "jpeg" or "tiff"
     * @param maxSize The max. size for an image
     * @param maxResolution The max. resolution for an image
     * @return <code>true</code>, if metadata is needed, <code>false</code>, otherwise
     */
    private boolean needsMetadata(String formatName, long maxSize, long maxResolution) {
        if (maxSize > 0 || maxResolution > 0) {
            // Limitations specified, thus meta-data is needed
            return true;
        }
        if (null == formatName || 0 == formatName.length()) {
            return false;
        }
        for (ImageTransformation transformation : transformations) {
            if (transformation.supports(formatName) && transformation.needsImageInformation()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Writes out an image into a byte-array and wraps it into a transformed image.
     *
     * @param image The image to write
     * @param formatName The format to use, e.g. "jpeg" or "tiff"
     * @return The image data
     * @throws IOException
     */
    private TransformedImage writeTransformedImage(BufferedImage image, String formatName) throws IOException {
        if (null == image) {
            return null;
        }
        DigestOutputStream digestOutputStream = null;
        UnsynchronizedByteArrayOutputStream outputStream = null;
        try {
            outputStream = new UnsynchronizedByteArrayOutputStream(8192);
            digestOutputStream = new DigestOutputStream(outputStream, MessageDigest.getInstance("MD5"));
            if (needsCompression(formatName)) {
                writeCompressed(image, formatName, digestOutputStream, transformationContext);
            } else {
                write(image, formatName, digestOutputStream);
            }

            byte[] imageData = outputStream.toByteArray();
            byte[] md5 = digestOutputStream.getMessageDigest().digest();
            long size = null != imageData ? imageData.length : 0L;
            return new TransformedImageImpl(image.getWidth(), image.getHeight(), size, formatName, imageData, md5, transformationContext.getExpenses());
        } catch (NoSuchAlgorithmException e) {
            throw new IOException(e);
        } finally {
            Streams.close(digestOutputStream, outputStream);
        }
    }

    /**
     * Writes out an image into a byte-array.
     *
     * @param image The image to write
     * @param formatName The format to use, e.g. "jpeg" or "tiff"
     * @return The image data
     * @throws IOException
     */
    private byte[] write(BufferedImage image, String formatName) throws IOException {
        if (null == image) {
            return null;
        }
        UnsynchronizedByteArrayOutputStream outputStream = null;
        try {
            outputStream = new UnsynchronizedByteArrayOutputStream(8192);
            if (needsCompression(formatName)) {
                writeCompressed(image, formatName, outputStream, transformationContext);
            } else {
                write(image, formatName, outputStream);
            }
            return outputStream.toByteArray();
        } finally {
            Streams.close(outputStream);
        }
    }

    private boolean needsCompression(String formatName) {
        return this.compress && null != formatName && "jpeg".equalsIgnoreCase(formatName) || "jpg".equalsIgnoreCase(formatName);
    }

    private static void write(BufferedImage image, String formatName, OutputStream output) throws IOException {
        if (false == ImageIO.write(image, formatName, output)) {
            throw new IOException("no appropriate writer found for " + formatName);
        }
    }

    private static void writeCompressed(BufferedImage image, String formatName, OutputStream output, TransformationContext transformationContext) throws IOException {
        ImageWriter writer = null;
        ImageOutputStream imageOutputStream = null;
        try {
            Iterator<ImageWriter> iter = ImageIO.getImageWritersByFormatName(formatName);
            if (null == iter || false == iter.hasNext()) {
                iter = ImageIO.getImageWritersByMIMEType(formatName);
                if (null == iter || false == iter.hasNext()) {
                    throw new IOException("No image writer for format " + formatName);
                }
            }
            writer = iter.next();
            ImageWriteParam iwp = writer.getDefaultWriteParam();
            adjustCompressionParams(iwp);
            imageOutputStream = ImageIO.createImageOutputStream(output);
            writer.setOutput(imageOutputStream);
            IIOImage iioImage = new IIOImage(image, null, null);
            writer.write(null, iioImage, iwp);
            transformationContext.addExpense(ImageTransformations.LOW_EXPENSE);
        } finally {
            if (null != writer) {
                writer.dispose();
            }
            if (null != imageOutputStream) {
                imageOutputStream.close();
            }
        }
    }

    /**
     * Tries to adjust the default settings on the supplied image write parameters to apply compression, ignoring any
     * {@link UnsupportedOperationException}s that may occur.
     *
     * @param parameters The parameters to adjust for compression
     */
    private static void adjustCompressionParams(ImageWriteParam parameters) {
        try {
            parameters.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
        } catch (UnsupportedOperationException e) {
            LOG.debug("", e);
        }
        try {
            parameters.setProgressiveMode(ImageWriteParam.MODE_DEFAULT);
        } catch (UnsupportedOperationException e) {
            LOG.debug("", e);
        }
        try {
            parameters.setCompressionQuality(0.8f);
        } catch (UnsupportedOperationException e) {
            LOG.debug("", e);
        }
    }

    /**
     * Reads a buffered image from the supplied stream and closes the stream afterwards.
     *
     * @param inputStream The stream to read the image from
     * @param formatName The format name
     * @param signaler The optional signaler or <code>null</code>
     * @return The buffered image
     * @throws IOException
     */
    private BufferedImage read(InputStream inputStream, String formatName, ImageTransformationSignaler signaler) throws IOException {
        try {
            return imageIoRead(inputStream, signaler);
        } catch (final RuntimeException e) {
            LOG.debug("error reading image from stream for {}", formatName, e);
            return null;
        } finally {
            Streams.close(inputStream);
        }
    }

    /**
     * Reads a buffered image from the supplied stream and closes the stream afterwards, trying to extract meta-data information.
     *
     * @param inputStream The stream to read the image from
     * @param formatName The format name
     * @param maxSize The max. size for an image or less than/equal to 0 (zero) for no size limitation
     * @param maxResolution The max. resolution for an image or less than/equal to 0 (zero) for no resolution limitation
     * @param signaler The optional signaler or <code>null</code>
     * @return The buffered image
     * @throws IOException
     */
    private BufferedImage readAndExtractMetadataFromStream(InputStream inputStream, String formatName, long maxSize, long maxResolution, ImageTransformationSignaler signaler) throws IOException {
        ThresholdFileHolder sink = null;
        try {
            sink = new ThresholdFileHolder();
            sink.write(maxSize > 0 ? new CountingInputStream(inputStream, maxSize, IMAGE_SIZE_EXCEEDED_EXCEPTION_CREATOR) : inputStream);

            try {
                metadata = ImageMetadataReader.readMetadata(ImageTransformationUtility.bufferedInputStreamFor(getFileStream(sink)), false);
            } catch (ImageProcessingException e) {
                LOG.debug("error getting metadata for {}", formatName, e);
            }

            if (maxResolution > 0) {
                ImageInformation imageInformation = getImageInformation(metadata);
                if (null != imageInformation) {
                    int resolution = imageInformation.height * imageInformation.width;
                    if (resolution > maxResolution) {
                        throw createResolutionExceededIOException(maxResolution, resolution);
                    }
                }
            }

            File tempFile = sink.getTempFile();
            if (null == tempFile) {
                // Everything held in memory - don't care
                return imageIoRead(getFileStream(sink), signaler);
            }

            BufferedImage bufferedImage = imageIORead(tempFile, signaler);
            sink = null; // Avoid preliminary closing in 'finally' clause
            return bufferedImage;
        } catch (OXException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException) cause;
            }
            throw new IOException("error accessing managed file", e);
        } catch (IllegalArgumentException e) {
            LOG.debug("error reading image from stream for {}", formatName, e);
            return null;
        } finally {
            Streams.close(sink);
        }
    }

    /**
     * Reads a buffered image from the supplied stream and closes the stream afterwards, trying to extract meta-data information.
     *
     * @param imageFile The image file to read from
     * @param formatName The format name
     * @param maxSize The max. size for an image or less than/equal to 0 (zero) for no size limitation
     * @param maxResolution The max. resolution for an image or less than/equal to 0 (zero) for no resolution limitation
     * @param signaler The optional signaler or <code>null</code>
     * @return The buffered image
     * @throws IOException
     */
    private BufferedImage readAndExtractMetadataFromFile(IFileHolder imageFile, String formatName, long maxSize, long maxResolution, ImageTransformationSignaler signaler) throws IOException {
        try {
            if (imageFile.getLength() > maxSize) {
                throw IMAGE_SIZE_EXCEEDED_EXCEPTION_CREATOR.createIOException(imageFile.getLength(), maxSize);
            }

            try {
                metadata = ImageMetadataReader.readMetadata(ImageTransformationUtility.bufferedInputStreamFor(getFileStream(imageFile)), false);
            } catch (ImageProcessingException e) {
                LOG.debug("error getting metadata for {}", formatName, e);
            }

            if (maxResolution > 0) {
                ImageInformation imageInformation = getImageInformation(metadata);
                if (null != imageInformation) {
                    int resolution = imageInformation.height * imageInformation.width;
                    if (resolution > maxResolution) {
                        throw createResolutionExceededIOException(maxResolution, resolution);
                    }
                }
            }

            File tempFile = imageFile instanceof ThresholdFileHolder ? ((ThresholdFileHolder) imageFile).getTempFile() : null;
            if (null == tempFile) {
                // Everything held in memory - don't care
                return imageIoRead(getFileStream(imageFile), signaler);
            }

            // Read from file
            return imageIORead(tempFile, signaler);
        } catch (IllegalArgumentException e) {
            LOG.debug("error reading image from stream for {}", formatName, e);
            return null;
        }
    }

    /**
     * Returns a {@link BufferedImage} as the result of decoding a supplied {@code InputStream}.
     *
     * @param in The input stream to read from
     * @param signaler The optional signaler or <code>null</code>
     * @return The resulting {@code BufferedImage} instance
     * @throws IOException If an I/O error occurs
     */
    private BufferedImage imageIoRead(InputStream in, ImageTransformationSignaler signaler) throws IOException {
        onImageRead(signaler);
        return ImageIO.read(in);
    }

    /**
     * Returns a {@link BufferedImage} as the result of decoding a supplied file.
     *
     * @param file The file to read from
     * @param signaler The optional signaler or <code>null</code>
     * @return The resulting {@code BufferedImage} instance
     * @throws IOException If an I/O error occurs
     */
    private BufferedImage imageIORead(File file, ImageTransformationSignaler signaler) throws IOException {
        onImageRead(signaler);
        return ImageIO.read(file);
    }

    private static void onImageRead(ImageTransformationSignaler signaler) {
        if (null != signaler) {
            try {
                signaler.onImageRead();
            } catch (Exception e) {
                LOG.debug("Signaler could not be called", e);
            }
        }
    }

    /**
     * Removes the transparency from the given image if necessary, i.e. the color model has an alpha channel and the supplied image
     * format is supposed to not support transparency.
     *
     * @param image The image
     * @param formatName The image format name, e.g. "jpeg" or "tiff"
     * @return The processed buffered image, or the previous image if no processing was necessary
     */
    private static BufferedImage removeTransparencyIfNeeded(BufferedImage image, String formatName) {
        if (null != image && null != formatName && false == ImageTransformationUtility.supportsTransparency(formatName)) {
            ColorModel colorModel = image.getColorModel();
            if (null != colorModel && colorModel.hasAlpha()) {
                BufferedImage targetImage = new BufferedImage(image.getWidth(), image.getHeight(), BufferedImage.TYPE_INT_RGB);
                Graphics2D graphics = targetImage.createGraphics();
                graphics.drawImage(image, 0, 0, Color.WHITE, null);
                graphics.dispose();
                return targetImage;
            }
        }
        return image;
    }

    /**
     * Gets the {@code InputStream} from specified image file.
     *
     * @param imageFile The image file
     * @return The input stream
     * @throws IOException If input stream cannot be returned
     */
    private static InputStream getFileStream(IFileHolder imageFile) throws IOException {
        if (null == imageFile) {
            return null;
        }
        try {
            return imageFile.getStream();
        } catch (OXException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException) cause;
            }
            throw null == cause ? new IOException(e.getMessage(), e) : new IOException(cause.getMessage(), cause);
        }
    }

}
