/*  Copyright (c) 2006-2007, Vladimir Nikic
    All rights reserved.
	
    Redistribution and use of this software in source and binary forms, 
    with or without modification, are permitted provided that the following 
    conditions are met:
	
    * Redistributions of source code must retain the above
      copyright notice, this list of conditions and the
      following disclaimer.
	
    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the
      following disclaimer in the documentation and/or other
      materials provided with the distribution.
	
    * The name of HtmlCleaner may not be used to endorse or promote 
      products derived from this software without specific prior
      written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
    AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
    ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
    LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
    CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
    ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
    POSSIBILITY OF SUCH DAMAGE.
	
    You can contact Vladimir Nikic by sending e-mail to
    nikic_vladimir@yahoo.com. Please include the word "HtmlCleaner" in the
    subject line.
*/

package org.htmlcleaner;

import java.io.IOException;
import java.io.Writer;
import java.util.Iterator;
import java.util.List;

/**
 * <p>Simple XML serializer - creates resulting XML without indenting lines.</p>
 */
public class SimpleXmlSerializer extends XmlSerializer {

	public SimpleXmlSerializer(CleanerProperties props) {
		super(props);
	}

	protected void serializeContentToken(ContentNode item, TagNode tagNode, Writer writer) throws IOException {
        String content = item.getContent();
        String trimmed = content.trim();
        boolean dontEscape = dontEscape(tagNode);                        
        if (trimmed.endsWith(SAFE_END_CDATA)) {
            int pos = content.lastIndexOf(SAFE_END_CDATA);
            String ending = content.substring(pos);
            if (dontEscape) {
                writer.write( content.substring(0, pos).replaceAll("]]>", "]]&gt;") );
            } else {
                if (trimmed.startsWith(BEGIN_CDATA)) {
                    int actualStart = content.indexOf(BEGIN_CDATA) + BEGIN_CDATA.length();
                    writer.write(content.substring(0, actualStart));
                    writer.write( escapeXml(content.substring(actualStart, pos)));
                } else {
                    writer.write( escapeXml(content.substring(0, pos)));
                }
            }
            writer.write(ending);
        } else {
            if (dontEscape)
                writer.write( content.replaceAll("]]>", "]]&gt;") );
            else {
                writer.write( escapeXml(content) );
            }
        }       
    }

    @Override
    protected void serialize(TagNode tagNode, Writer writer) throws IOException {
        serializeOpenTag(tagNode, writer, false);

        List tagChildren = tagNode.getAllChildren();
        if ( !isMinimizedTagSyntax(tagNode) ) {
            Iterator childrenIt = tagChildren.iterator();
            while ( childrenIt.hasNext() ) {
                Object item = childrenIt.next();
                if (item != null) {
                    if ( item instanceof ContentNode ) {
                        serializeContentToken((ContentNode)item, tagNode, writer);
                    } else {
                        ((BaseToken)item).serialize(this, writer);
                    }
                }
            }

            serializeEndTag(tagNode, writer, false);
        }
    }

}