package org.scribe.builder.api;

import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.scribe.exceptions.OAuthException;
import org.scribe.extractors.AccessTokenExtractor;
import org.scribe.model.OAuthConfig;
import org.scribe.model.OAuthConstants;
import org.scribe.model.OAuthRequest;
import org.scribe.model.Response;
import org.scribe.model.Token;
import org.scribe.model.Verb;
import org.scribe.model.Verifier;
import org.scribe.oauth.OAuth20ServiceImpl;
import org.scribe.oauth.OAuthService;
import org.scribe.utils.OAuthEncoder;
import org.scribe.utils.Preconditions;

/**
 * SurDoc OAuth2.0
 * Released under the same license as scribe (MIT License)
 *
 * This code borrows from and modifies changes made by @yincrash
 * @author th0rb3n
 */
public class SurDocApi extends DefaultApi20 {

    private static final String AUTHORIZE_URL = "https://open.surdoc.com/oauth/oauth/authorize?response_type=code&client_id=%s&redirect_uri=%s";
    private static final String SCOPED_AUTHORIZE_URL = AUTHORIZE_URL + "&scope=%s";
    private static final String SUFFIX_OFFLINE = "&" + OAuthConstants.ACCESS_TYPE + "=" + OAuthConstants.ACCESS_TYPE_OFFLINE
            + "&" + OAuthConstants.APPROVAL_PROMPT + "=" + OAuthConstants.APPROVAL_PROMPT_FORCE;

    /** The <code>"access_token": &lt;token&gt;</code> pattern */
    static final Pattern PATTERN_ACCESS_TOKEN = Pattern.compile("\"access_token\" *: *\"([^&\"]+)\"");

    /** The <code>"refresh_token": &lt;token&gt;</code> pattern */
    static final Pattern PATTERN_REFRESH_TOKEN = Pattern.compile("\"refresh_token\" *: *\"([^&\"]+)\"");

    /** The <code>"expires_in": &lt;number&gt;</code> pattern */
    static final Pattern PATTERN_EXPIRES = Pattern.compile("\"expires_in\" *: *([0-9]+)");

    @Override
    public String getAccessTokenEndpoint() {
        return "https://open.surdoc.com/oauth/oauth/token";
    }

    @Override
    public AccessTokenExtractor getAccessTokenExtractor() {
        return new AccessTokenExtractor() {

            @Override
            public Token extract(String response) {
                Preconditions.checkEmptyString(response, "Response body is incorrect. Can't extract a token from an empty string");

                Matcher matcher = PATTERN_ACCESS_TOKEN.matcher(response);
                if (matcher.find())
                {
                    String token = OAuthEncoder.decode(matcher.group(1));
                    String refreshToken = "";
                    Matcher refreshMatcher = PATTERN_REFRESH_TOKEN.matcher(response);
                    if (refreshMatcher.find()) {
                        refreshToken = OAuthEncoder.decode(refreshMatcher.group(1));
                    }
                    Date expiry = null;
                    Matcher expiryMatcher = PATTERN_EXPIRES.matcher(response);
                    if (expiryMatcher.find())
                    {
                        int lifeTime = Integer.parseInt(OAuthEncoder.decode(expiryMatcher.group(1)));
                        expiry = new Date(System.currentTimeMillis() + lifeTime * 1000);
                    }
                    Token result = new Token(token, refreshToken, expiry, response);
                    return result;
                }
                else
                {
                    throw new OAuthException("Response body is incorrect. Can't extract a token from this: '" + response + "'", null);
                }
            }
        };
    }

    @Override
    public String getAuthorizationUrl(OAuthConfig config) {
        // Append scope if present
        if (config.hasScope()) {
            String format = config.isOffline() ? SCOPED_AUTHORIZE_URL + SUFFIX_OFFLINE : SCOPED_AUTHORIZE_URL;
            return String.format(format, config.getApiKey(),
                    OAuthEncoder.encode(config.getCallback()),
                    OAuthEncoder.encode(config.getScope()));
        } else {
            String format = config.isOffline() ? AUTHORIZE_URL + SUFFIX_OFFLINE : AUTHORIZE_URL;
            return String.format(format, config.getApiKey(),
                    OAuthEncoder.encode(config.getCallback()));
        }
    }

    @Override
    public Verb getAccessTokenVerb() {
        return Verb.POST;
    }

    @Override
    public OAuthService createService(OAuthConfig config) {
        return new SurDocApiService(this, config);
    }

    public static class SurDocApiService extends OAuth20ServiceImpl {

        private final DefaultApi20 api;
        private final OAuthConfig config;

        SurDocApiService(DefaultApi20 api, OAuthConfig config) {
            super(api, config);
            this.api = api;
            this.config = config;
        }

        /**
         * Checks possible expiration for specified access token.
         * <p>
         * If the protected resource request does not include authentication credentials or does not contain an access token that enabled
         * access to the protected resource, Box sets the <code>WWW-Authenticate</code> response header field
         * <p>
         * If the protected resource request included an access token and failed authentication, Box sets the "error" attribute to provide
         * the client with the reason why the access request was declined. Box also includes the "error-description" attribute to provide
         * developers a human-readable explanation that is not meant to be displayed to end users.
         * <p>
         * <table>
         * <thead>
         * <tr>
         * <th>HTTP Status</th>
         * <th>Error Code</th>
         * <th>What happened</th>
         * </tr>
         * </thead>
         * <tr>
         * <td><code>400</code></td>
         * <td>invalid_request</td>
         * <td>The request is missing a required parameter, includes an unsupported parameter or parameter value, repeats the same
         * parameter, uses more than one method for including an access token, or is otherwise malformed</td>
         * </tr>
         * <tr>
         * <td><code>401</code></td>
         * <td>invalid_token</td>
         * <td>The access token provided is expired, revoked, malformed or invalid for other reasons. The client may request a new access
         * token and retry the protected resource request</td>
         * </tr>
         * <tr>
         * <td><code>403</code></td>
         * <td>insufficient_scope</td>
         * <td>The request requires higher privileges than provided by the access token</td>
         * </tr>
         * </table>
         *
         * @param accessToken The access token to validate
         * @return <code>true</code> if expired; otherwise <code>false</code> if valid
         */
        public boolean isExpired(String accessToken) {
            OAuthRequest request = new OAuthRequest(Verb.GET, "https://api.box.com/2.0/users/me");
            request.addHeader("Authorization", "Bearer " + accessToken);
            Response response = request.send();
            if (response.getCode() == 401 || response.getCode() == 400) {
                // 401  unauthorized
                return true;
            }

            return false;
        }

        @Override
        public Token getAccessToken(Token requestToken, Verifier verifier) {
            OAuthRequest request = new OAuthRequest(api.getAccessTokenVerb(), api.getAccessTokenEndpoint());
            switch (api.getAccessTokenVerb()) {
                case POST:
                    request.addBodyParameter(OAuthConstants.CLIENT_ID, config.getApiKey());
                    // API Secret is optional
                    if (config.getApiSecret() != null && config.getApiSecret().length() > 0) {
                        request.addBodyParameter(OAuthConstants.CLIENT_SECRET, config.getApiSecret());
                    }
                    if (requestToken == null) {
                        request.addBodyParameter(OAuthConstants.CODE, verifier.getValue());
                        request.addBodyParameter(OAuthConstants.REDIRECT_URI, config.getCallback());
                        request.addBodyParameter(OAuthConstants.GRANT_TYPE, OAuthConstants.GRANT_TYPE_AUTHORIZATION_CODE);
                    } else {
                        request.addBodyParameter(OAuthConstants.REFRESH_TOKEN, requestToken.getSecret());
                        request.addBodyParameter(OAuthConstants.GRANT_TYPE, OAuthConstants.GRANT_TYPE_REFRESH_TOKEN);
                    }
                    break;
                case GET:
                default:
                    request.addQuerystringParameter(OAuthConstants.CLIENT_ID, config.getApiKey());
                    // API Secret is optional
                    if (config.getApiSecret() != null && config.getApiSecret().length() > 0) {
                        request.addQuerystringParameter(OAuthConstants.CLIENT_SECRET, config.getApiSecret());
                    }
                    request.addQuerystringParameter(OAuthConstants.CODE, verifier.getValue());
                    request.addQuerystringParameter(OAuthConstants.REDIRECT_URI, config.getCallback());
                    if(config.hasScope()) {
                        request.addQuerystringParameter(OAuthConstants.SCOPE, config.getScope());
                    }
            }
            Response response = request.send();
            return api.getAccessTokenExtractor().extract(response.getBody());
        }
    }

}
