/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.ant.tasks;

import static com.openexchange.obs.api.Tools.allPublishedAndSomePackageUnresolved;
import static com.openexchange.obs.api.Tools.getFirstPackageForState;
import static com.openexchange.obs.api.Tools.isProjectBuilding;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import com.openexchange.obs.api.Architecture;
import com.openexchange.obs.api.BuildServiceClient;
import com.openexchange.obs.api.BuildServiceException;
import com.openexchange.obs.api.Code;
import com.openexchange.obs.api.ObsPackage;
import com.openexchange.obs.api.PackageStatus;
import com.openexchange.obs.api.Repository;
import com.openexchange.obs.api.Result;
import com.openexchange.obs.api.Tools;

/**
 * @author choeger
 */
public class Wait4Project extends Task {

    private String url;
    private String login;
    private String password;

    private String projectNames[];
    private String repositories[];
    private HashSet<String> packageNames;

    private String failedProject;
    private String failedRepository;
    private String failedPackage;
    private String failedArchProperty;

    private String packageNamesProperty;

    private int sleep = 1000;

    public Wait4Project() {
        super();
    }

    public final void setSleep(final int sleep) {
        this.sleep = sleep;
    }

    public void setProject(final String projectNames) {
        if (projectNames.length() != 0) {
            this.projectNames = projectNames.split(",", 0);
        }
    }

    public final void setRepositories(final String repositories) {
        if (repositories.length() != 0) {
            this.repositories = repositories.split(",", 0);
        }
    }

    public final void setPackageNames(final String packageNames) {
        if (packageNames.length() != 0) {
            this.packageNames = new HashSet<String>(Arrays.asList(packageNames.split(",", 0)));
        }
    }

    public final void setLogin(final String login) {
        this.login = login;
    }

    public final void setPassword(final String password) {
        this.password = password;
    }

    public final void setUrl(final String url) {
        this.url = url;
    }

    public void setFailProjectName(final String failProjectName) {
        this.failedProject = failProjectName;
    }

    public void setFailRepositoryName(final String failRepositoryName) {
        this.failedRepository = failRepositoryName;
    }

    public void setFailPackageName(final String failPackageName) {
        this.failedPackage = failPackageName;
    }

    public final void setFailedArchProperty(String failedArchProperty) {
        this.failedArchProperty = failedArchProperty;
    }

    public void setPackageNamesProperty(final String packageNamesProperty) {
        this.packageNamesProperty = packageNamesProperty;
    }

    @Override
    public void execute() throws BuildException {
        neccessaryPropertiesCheck();
        try {
            final BuildServiceClient bsc = new BuildServiceClient(url, login, password);
            // Wait 180 seconds to get an initial value of packages to wait for
            int counter = 0;
            while (checkPackageNamesProperty() && counter < 180 && packageNames == null) {
                Thread.sleep(sleep);
                counter++;
            }
            if (packageNames == null) {
                log("Checking build status for all packages.", Project.MSG_INFO);
            } else {
                log("Checking build status of the following packages: " + packageNames, Project.MSG_ERR);
            }
            // endless loop until build finished
            Map<String, Result[]> tmpResults = new HashMap<String, Result[]>();
            boolean didNeverBuild = true;
            do {
                tmpResults.clear();
                Thread.sleep(sleep);
                checkPackageNamesProperty();
                for (final String projectName : projectNames) {
                    HashSet<String> currentPackages = checkCurPackages(bsc, projectName);
                    final Result[] projectResults = bsc.getProjectStatus(projectName, currentPackages);
                    checkResultsForState(projectResults, projectName, Code.FAILED);
                    tmpResults.put(projectName, projectResults);
                }
                if (isProjectBuilding(combine(tmpResults))) {
                    didNeverBuild = false;
                }
            } while ((isProjectBuilding(combine(tmpResults)) || didNeverBuild) && !allPublishedAndSomePackageUnresolved(combine(tmpResults)));
            logUnresolvedPackage(tmpResults);
            for (final String projectName : projectNames) {
                checkResultsForState(combine(tmpResults), projectName, Code.UNRESOLVABLE);
            }
        } catch (IOException e) {
            throw new BuildException(e.getMessage(), e);
        } catch (BuildServiceException e) {
            throw new BuildException(e.getMessage(), e);
        } catch (InterruptedException e) {
            throw new BuildException(e.getMessage(), e);
        }
    }

    private void logUnresolvedPackage(Map<String, Result[]> results) {
        if (allPublishedAndSomePackageUnresolved(combine(results))) {
            for (Entry<String, Result[]> entry : results.entrySet()) {
                String projectName = entry.getKey();
                for (Result result : entry.getValue()) {
                    for (PackageStatus status : result.getStatuses()) {
                        if (Code.UNRESOLVABLE == status.getCode()) {
                            log("Project " + projectName + " has unresolved packages.", Project.MSG_ERR);
                            Repository repository = result.getRepository();
                            log("Repository: " + repository.getName() + ", Architecture: " + result.getRepository().getArch(), Project.MSG_ERR);
                            log("Package " + status.getPackageName() + " can not be built. Reason: " + status.getDetails());
                            throw new BuildException("Build on build service failed.");
                        }
                    }
                }
            }
        }
    }

    private static Result[] combine(Map<String, Result[]> map) {
        List<Result> retval = new ArrayList<Result>();
        for (Result[] results : map.values()) {
            retval.addAll(Arrays.asList(results));
        }
        return retval.toArray(new Result[retval.size()]);
    }

    private void neccessaryPropertiesCheck() throws BuildException {
        if (null == url || 0 == url.length()) {
            throw new BuildException("URL of build service is not defined.");
        }
        if (null == login || 0 == login.length()) {
            throw new BuildException("User for authenticating in build service is not defined.");
        }
        if (null == password || 0 == password.length()) {
            throw new BuildException("Password for authenticating in build service is not defined.");
        }
        if (null == projectNames || 0 == projectNames.length) {
            throw new BuildException("Project in build service is not defined.");
        }
        if (null == repositories || 0 == repositories.length) {
            throw new BuildException("Distribution in build service is not defined.");
        }
        if (sleep < 1000) {
            throw new BuildException("Sleep delay for polling build service must be greater than 1000.");
        }
    }

    private HashSet<String> checkCurPackages(final BuildServiceClient bsc, final String projectName) throws BuildServiceException {
        final HashSet<String> currentPackages = new HashSet<String>();
        if (packageNames != null) {
            // creates intersect list of packageNames in this project and the given list of packageNames
            for (ObsPackage obsp : bsc.getPackageList(projectName)) {
                currentPackages.add(obsp.getPackageName());
            }
            currentPackages.retainAll(packageNames);
        }
        return currentPackages;
    }

    private void checkResultsForState(Result[] results, String projectName, Code... codes) throws BuildException, BuildServiceException {
        Result result = Tools.getFirstResultWithState(results, codes);
        if (null == result) {
            return;
        }
        PackageStatus pack = getFirstPackageForState(result.getStatuses(), codes);
        if (null == pack) {
            throw new BuildException("One or more packages has " + Code.join(',', codes) + " inside result for repository " + result.getRepository().toString() + ", but could not determine which one.");
        }
        setFailProps(projectName, result, pack);
    }

    private void setFailProps(String projectName, Result result, PackageStatus firstFailedPackage) throws BuildException, BuildServiceException {
        if (null != failedProject) {
            log("Project: " + failedProject + '=' + projectName, Project.MSG_ERR);
            getProject().setInheritedProperty(failedProject, projectName);
        } else {
            log("Project: " + projectName, Project.MSG_ERR);
        }
        if (null != failedRepository) {
            log("Repository: " + failedRepository + '=' + result.getRepository().getName(), Project.MSG_ERR);
            getProject().setInheritedProperty(failedRepository, result.getRepository().getName());
        } else {
            log("Repository: " + result.getRepository().getName(), Project.MSG_ERR);
        }
        if (null != failedArchProperty) {
            log("Architecture: " + failedArchProperty + '=' + result.getRepository().getArch().toString(), Project.MSG_ERR);
            getProject().setInheritedProperty(failedArchProperty, Architecture.single(result.getRepository().getArch()).name());
        } else {
            log("Architecture: " + result.getRepository().getArch().toString(), Project.MSG_ERR);
        }
        if (null != failedPackage) {
            log("Package: " + failedPackage + '=' + firstFailedPackage.getPackageName(), Project.MSG_ERR);
            getProject().setInheritedProperty(failedPackage, firstFailedPackage.getPackageName());
        } else {
            log("Package: " + firstFailedPackage.getPackageName(), Project.MSG_ERR);
        }
        throw new BuildException("Build on build service failed.");
    }

    /**
     * Will check the current state of packageNamesProperty
     * @return <code>false</code> if packageNamesProperty is <code>null</code> and <code>true</code> if it is <code>not null</code>
     */
    private boolean checkPackageNamesProperty() {
        if (packageNamesProperty == null) {
            return false;
        }
        if (getProject().getProperty(packageNamesProperty) == null) {
            return true;
        }
        String packageNamesString = getProject().getProperty(packageNamesProperty);
        if (packageNamesString.length() != 0) {
            this.packageNames = new HashSet<String>(Arrays.asList(packageNamesString.split(",", 0)));
        }
        return true;
    }
}
