/*
 * Copyright 2007 Netflix, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.oauth;

import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Properties of one User of an OAuthConsumer. Properties may be added freely,
 * e.g. to support extensions.
 * 
 * @author John Kristian
 */
public class OAuthAccessor implements Cloneable, Serializable {

    private static final long serialVersionUID = 5590788443138352999L;

    public final OAuthConsumer consumer;
    private final Map<String, Object> properties;
    public volatile String requestToken;
    public volatile String accessToken;
    public volatile String tokenSecret;

    /**
     * Initializes a new {@link OAuthAccessor}.
     * 
     * @param consumer The associated consumer
     */
    public OAuthAccessor(final OAuthConsumer consumer) {
        super();
        properties = new ConcurrentHashMap<String, Object>();
        this.consumer = consumer;
        this.requestToken = null;
        this.accessToken = null;
        this.tokenSecret = null;
    }


    @Override
    public OAuthAccessor clone() {
        try {
            return (OAuthAccessor) super.clone();
        } catch (final CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Gets the property associated with given name.
     * 
     * @param name The name
     * @return The value or <code>null</code> if absent
     */
    @SuppressWarnings("unchecked")
    public <V> V getProperty(final String name) {
        try {
            return (V) properties.get(name);
        } catch (final ClassCastException e) {
            return null;
        }
    }

    /**
     * Sets specified property.
     * <p>
     * If value is <code>null</code>, a remove is performed.
     * 
     * @param name The name
     * @param value The value (possibly <code>null</code>)
     */
    public void setProperty(final String name, final Object value) {
        if (null == value) {
            properties.remove(name);
        } else {
            properties.put(name, value);
        }
    }

    /**
     * Gets the properties associated with this accessor.
     * 
     * @return The properties' iterator
     */
    public Iterator<Map.Entry<String, Object>> getProperties() {
        return OAuth.unmodifiableIterator(properties.entrySet().iterator());
    }

    /**
     * Construct a request message containing the given parameters but no body.
     * Don't send the message, merely construct it. The caller will ordinarily
     * send it, for example by calling OAuthClient.invoke or access.
     * 
     * @param method
     *            the HTTP request method. If this is null, use the default
     *            method; that is getProperty("httpMethod") or (if that's null)
     *            consumer.getProperty("httpMethod") or (if that's null)
     *            OAuthMessage.GET.
     */
    public OAuthMessage newRequestMessage(String method, final String url, final Collection<? extends Map.Entry> parameters,
            final InputStream body) throws OAuthException, IOException, URISyntaxException {
        if (method == null) {
            method = (String) this.getProperty("httpMethod");
            if (method == null) {
                method = (String) this.consumer.getProperty("httpMethod");
                if (method == null) {
                    method = OAuthMessage.GET;
                }
            }
        }
        final OAuthMessage message = new OAuthMessage(method, url, parameters, body);
        message.addRequiredParameters(this);
        return message;
    }

    public OAuthMessage newRequestMessage(final String method, final String url, final Collection<? extends Map.Entry> parameters)
            throws OAuthException, IOException, URISyntaxException {
        return newRequestMessage(method, url, parameters, null);
    }

}
