/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.oauth.impl.internal.groupware;

import static com.openexchange.database.Databases.closeSQLStuff;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.update.PerformParameters;
import com.openexchange.tools.update.Column;
import com.openexchange.tools.update.Tools;

/**
 * {@link OAuthSetScopeColumnNotNullTask} - Updates the <code>oauthAccounts.scope</code> column
 * and makes it NON NULL.
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 * @since v8.0.0
 */
public class OAuthSetScopeColumnNotNullTask extends AbstractOAuthUpdateTask {

    /**
     * Initialises a new {@link OAuthSetScopeColumnNotNullTask}.
     */
    public OAuthSetScopeColumnNotNullTask() {
        super();
    }

    @Override
    public String[] getDependencies() {
        return new String[] { OAuthAddScopeColumnTask.class.getName() };
    }

    @Override
    void innerPerform(Connection connection, PerformParameters performParameters) throws OXException, SQLException {
        adjustScopes(connection);
        Tools.modifyColumns(connection, CreateOAuthAccountTable.TABLE_NAME, new Column("scope", "varchar(767) NOT NULL"));
    }

    /**
     * Sets empty scopes where the scope is NULL
     *
     * @param con The writea-ble connection
     * @throws SQLException if an SQL error is occurred
     */
    private void adjustScopes(Connection con) throws SQLException {
        PreparedStatement stmt = null;
        try {
            stmt = con.prepareStatement("UPDATE " + CreateOAuthAccountTable.TABLE_NAME + " SET scope='' WHERE scope IS NULL");
            stmt.executeUpdate();
        } finally {
            closeSQLStuff(stmt);
        }
    }
}
