/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 Open-Xchange GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.ant.tasks;

import java.io.File;
import java.io.IOException;
import java.util.Properties;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.util.FileUtils;
import com.openexchange.ant.Tools;

/**
 * Tries to determine the correct build script for the given project and copies it.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class CopyBuildScript extends Task {

    private File bundle;
    private File clt;
    private File js;
    private File appsuite;
    private File microService;
    private File toProject;
    private String property;

    public CopyBuildScript() {
        super();
    }

    public void setBundle(File bundle) {
        this.bundle = bundle;
    }

    public void setClt(File clt) {
        this.clt = clt;
    }

    public void setJs(File js) {
        this.js = js;
    }

    public void setAppsuite(File appsuite) {
        this.appsuite = appsuite;
    }

    public void setMicroService(File microService) {
        this.microService = microService;
    }

    public void setToProject(File toProject) {
        this.toProject = toProject;
    }

    public void setProperty(String property) {
        this.property = property;
    }

    @Override
    public void execute() throws BuildException {
        if (null == toProject) {
            throw new BuildException("Please define path to project. (toProject property)");
        }
        if (!toProject.exists() || !toProject.isDirectory()) {
            throw new BuildException("A project directory with name \"" + toProject.getAbsolutePath() + "\" does not exist.");
        }
        final File projectFile = new File(toProject, ".project");
        final String projectContent;
        if (projectFile.exists() && projectFile.isFile() && projectFile.canRead()) {
            projectContent = Tools.readFile(projectFile);
        } else {
            projectContent = "";
        }
        final File buildPropertiesFile = new File(toProject, "ox-build.properties");
        final Properties buildProps;
        if (buildPropertiesFile.exists() && buildPropertiesFile.isFile() && buildPropertiesFile.canRead()) {
            buildProps = Tools.readProperties(buildPropertiesFile);
        } else {
            buildProps = new Properties();
        }
        final File buildXMLFile = new File(toProject, "build.xml");
        final boolean subBuildXML = buildXMLFile.exists();
        final String buildContent = subBuildXML ? Tools.readFile(buildXMLFile) : "";
        final String destinationName;
        final File sourceName;
        if (projectContent.contains("org.eclipse.pde.PluginNature")) {
            sourceName = bundle;
            destinationName = subBuildXML ? bundle.getName() : "build.xml";
        } else if (projectContent.contains("org.eclipse.jdt.core.javanature") && "microservice".equals(buildProps.getProperty("type"))) {
            sourceName = microService;
            destinationName = subBuildXML ? microService.getName() : "build.xml";
        } else if (projectContent.contains("org.eclipse.jdt.core.javanature")) {
            sourceName = clt;
            destinationName = subBuildXML ? clt.getName() : "build.xml";
        } else if (new File(toProject, "Gruntfile.js").isFile()) {
            sourceName = appsuite;
            destinationName = subBuildXML ? appsuite.getName() : "build.xml";
        } else if (projectContent.contains("org.eclipse.wst.jsdt.core.jsNature")) {
            sourceName = js;
            destinationName = subBuildXML ? js.getName() : "build.xml";
        } else if (subBuildXML && !buildContent.contains("<import file=\"build-")) {
            sourceName = null;
            destinationName = null;
        } else {
            throw new BuildException("Buildsystem does not know how to build project \"" + toProject + "\".");
        }
        if (null == sourceName) {
            log("Using provided build.xml to build project " + toProject + ".");
        } else {
            log("Using " + sourceName + " to build project " + toProject + ".");
            try {
                FileUtils.getFileUtils().copyFile(sourceName, new File(toProject, destinationName), null, true, true);
            } catch (IOException e) {
                throw new BuildException(e.getMessage(), e);
            }
            getProject().setInheritedProperty(property, destinationName);
        }
    }
}
