/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.gdpr.dataexport.impl.utils;

import static com.openexchange.java.Autoboxing.I;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.text.DateFormat;
import java.util.Iterator;
import java.util.Locale;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;
import java.util.zip.Deflater;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveInputStream;
import org.apache.commons.io.IOUtils;
import org.xml.sax.SAXParseException;
import com.openexchange.ajax.container.ThresholdFileHolder;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.exception.OXException;
import com.openexchange.filestore.FileStorage;
import com.openexchange.gdpr.dataexport.DataExportDiagnosticsReport;
import com.openexchange.gdpr.dataexport.DataExportStorageService;
import com.openexchange.gdpr.dataexport.DataExportTask;
import com.openexchange.gdpr.dataexport.Message;
import com.openexchange.gdpr.dataexport.SpoolingOptions;
import com.openexchange.gdpr.dataexport.impl.DataExportStrings;
import com.openexchange.gdpr.dataexport.impl.DataExportUtility;
import com.openexchange.i18n.tools.StringHelper;
import com.openexchange.java.Streams;
import com.openexchange.java.util.UUIDs;
import com.openexchange.server.ServiceLookup;
import com.openexchange.uploaddir.UploadDirService;

/**
 * {@link ChunkedZippedOutputStream} - Writes result file as a ZIP archive in demanded chunks to file storage
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.10.3
 */
public class ChunkedZippedOutputStream {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(ChunkedZippedOutputStream.class);

    /**
     * The buffer size of 64K.
     */
    private static final int BUFFER_SIZE = DataExportUtility.BUFFER_SIZE;

    private final DataExportStorageService storageService;
    private final DataExportTask task;
    private final ServiceLookup services;
    private final FileStorage destinationFileStorage;
    private final FileStorage fileStorageToWriteTo;
    private final long maxFileSize;
    private final boolean useZip64;
    private final File spoolingDirectory;

    private boolean cleanedUp;
    private int currentChunkNumber;
    private long currentSize;
    private ZippedFileStorageOutputStream zipOutputStream;

    /**
     * Initializes a new {@link ChunkedZippedOutputStream}.
     *
     * @param task The task to create results for
     * @param fileStorage The destination file storage
     * @param storageService The storage to use
     * @param useZip64 Whether to use ZIP64 format when building the ZIP archive
     * @param spoolingOptions The spooling options to use
     * @param services The service look-up
     * @throws OXException If initialization fails
     */
    public ChunkedZippedOutputStream(DataExportTask task, FileStorage fileStorage, DataExportStorageService storageService, boolean useZip64, SpoolingOptions spoolingOptions, ServiceLookup services) throws OXException {
        super();
        this.task = task;
        this.maxFileSize = task.getArguments().getMaxFileSize();
        this.destinationFileStorage = fileStorage;
        if (spoolingOptions.isEnabled()) {
            File rootDir = spoolingOptions.getDirectory();
            if (rootDir == null) {
                UploadDirService uploadDirService = services.getServiceSafe(UploadDirService.class);
                rootDir = uploadDirService.getUploadDir();
            }

            File dir = new File(rootDir, "open-xchange-tmpexportresultfilespool-" + UUIDs.getUnformattedStringFromRandom());
            if (!dir.mkdir()) {
                throw OXException.general("Failed to create temporory directory for spooling: " + dir.getPath());
            }

            LOG.info("Created spooling directory ({}) for result files of data export {} of user {} in context {}", dir.getPath(), UUIDs.getUnformattedStringObjectFor(task.getId()), I(task.getUserId()), I(task.getContextId()));
            this.fileStorageToWriteTo = new TmpFileFileStorage(dir);
            this.spoolingDirectory = dir;
        } else {
            this.fileStorageToWriteTo = fileStorage;
            this.spoolingDirectory = null;
        }
        this.storageService = storageService;
        this.useZip64 = useZip64;
        this.services = services;
        cleanedUp = false;
    }

    /**
     * Cleans-up this instance's results.
     *
     * @throws OXException If an Open-Xchange error occurs
     */
    public synchronized void cleanUp() throws OXException {
        storageService.deleteResultFiles(task.getId(), task.getUserId(), task.getContextId());
        cleanedUp = true;
    }

    /**
     * Closes this instance & returns collected file storage locations.
     *
     * @throws IOException If an I/O error occurs
     * @throws OXException If an Open-Xchange error occurs
     */
    public synchronized void close() throws IOException, OXException {
        if (cleanedUp) {
            return;
        }

        if (zipOutputStream != null) {
            closeStream(true);
        }
    }

    /**
     * Adds the diagnostics report to this instance.
     *
     * @param report The messages of the diagnostics report
     * @param locale The locale
     * @throws IOException If an I/O error occurs
     */
    public synchronized void addDiagnostics(DataExportDiagnosticsReport report, Locale locale) throws IOException {
        if (report == null || report.isEmpty()) {
            return;
        }

        if (cleanedUp) {
            throw new IllegalStateException("Already cleaned-up");
        }

        if (zipOutputStream == null) {
            // Not yet initialized
            constructNewStream();
        }

        // Create archive entry
        StringHelper stringHelper = StringHelper.valueOf(locale);
        ZipArchiveEntry entry = new ZipArchiveEntry(DataExportUtility.prepareEntryName(stringHelper.getString(DataExportStrings.DIAGNOSTICS_FILE_PREFIX) + ".txt", storageService.getConfig()));
        zipOutputStream.putArchiveEntry(entry);

        StringBuilder messageBuilder = new StringBuilder(128);
        byte[] delimiter = "\r\n\r\n---------------------------------------\r\n\r\n".getBytes(StandardCharsets.UTF_8);
        long size = 0;
        Iterator<Message> it = report.iterator();

        size += writeBytesAndReturnLength(formatMessage(it.next(), messageBuilder, locale).getBytes(StandardCharsets.UTF_8));
        while (it.hasNext()) {
            size += writeBytesAndReturnLength(delimiter);
            size += writeBytesAndReturnLength(formatMessage(it.next(), messageBuilder, locale).getBytes(StandardCharsets.UTF_8));
        }
        entry.setSize(size);

        // Complete the entry
        zipOutputStream.closeArchiveEntry();
        zipOutputStream.flush();
    }

    private int writeBytesAndReturnLength(byte[] bytes) throws IOException {
        zipOutputStream.write(bytes);
        return bytes.length;
    }

    /**
     * Formats given message ready for being added to report.
     *
     * @param message The message
     * @param messageBuilder The string builder
     * @param locale The locale
     * @return The formatted string
     */
    private static String formatMessage(Message message, StringBuilder messageBuilder, Locale locale) {
        messageBuilder.setLength(0);

        // Header
        messageBuilder.append(DateFormat.getDateInstance(DateFormat.LONG, locale).format(message.getTimeStamp()));
        messageBuilder.append(" at ");
        messageBuilder.append(DateFormat.getTimeInstance(DateFormat.SHORT, locale).format(message.getTimeStamp()));
        messageBuilder.append(" - ");
        messageBuilder.append(message.getModuleId());
        messageBuilder.append("\r\n");

        // Message
        messageBuilder.append(message.getMessage());
        messageBuilder.append("\r\n");

        return messageBuilder.toString();
    }

    /**
     * Adds the ZIP entries from specified ZIP input stream to this instance.
     *
     * @param zipIn The ZIP input stream to get ZIP entries from
     * @throws IOException If an I/O error occurs
     * @throws OXException If an Open-Xchange error occurs
     */
    public synchronized void addEntriesFrom(ZipArchiveInputStream zipIn) throws IOException, OXException {
        if (zipIn == null) {
            return;
        }

        if (cleanedUp) {
            throw new IllegalStateException("Already cleaned-up");
        }

        ThresholdFileHolder tmp = null;
        try {
            for (ZipArchiveEntry entry; (entry = zipIn.getNextEntry()) != null;) {
                tmp = createOrReset(tmp);
                long entrySize = IOUtils.copy(zipIn, tmp.asOutputStream(), BUFFER_SIZE);

                if (zipOutputStream == null) {
                    // Not yet initialized
                    constructNewStream();
                } else {
                    if ((maxFileSize > 0) && ((currentSize + entrySize) > maxFileSize)) {
                        // Max. file size would be exceeded
                        closeStream(false);
                        constructNewStream();
                    }
                }

                currentSize += entrySize;
                entry.setSize(entrySize);
                zipOutputStream.putArchiveEntry(entry);
                IOUtils.copy(tmp.getStream(), zipOutputStream, BUFFER_SIZE);
                zipOutputStream.closeArchiveEntry();
                zipOutputStream.flush();
            }
        } finally {
            Streams.close(tmp);
        }
    }

    private static ThresholdFileHolder createOrReset(ThresholdFileHolder fileHolder) {
        if (fileHolder == null) {
            return new ThresholdFileHolder(false);
        }

        fileHolder.reset();
        return fileHolder;
    }

    private void closeStream(boolean finalClose) throws IOException, OXException {
        // Complete the ZIP file
        zipOutputStream.flush();
        zipOutputStream.finish();
        zipOutputStream.close();

        // Acquire file storage location
        String fileStorageLocation = zipOutputStream.awaitFileStorageLocation();

        // Check if a spooling file storage was used
        if (fileStorageLocation != null && spoolingDirectory != null) {
            // Transfer to destination file storage
            String oldFileStorageLocation = fileStorageLocation;
            fileStorageLocation = transferSpooledFileWithRetry(oldFileStorageLocation, 5);
            DataExportUtility.deleteQuietly(oldFileStorageLocation, fileStorageToWriteTo);
            if (finalClose) {
                DataExportUtility.removeQuietly(fileStorageToWriteTo);
                LOG.info("Removed spooling directory ({}) for result files of data export {} of user {} in context {}", spoolingDirectory.getPath(), UUIDs.getUnformattedStringObjectFor(task.getId()), I(task.getUserId()), I(task.getContextId()));
            }
        }

        // Remember result file
        storageService.addResultFile(fileStorageLocation, ++currentChunkNumber, currentSize, task.getId(), task.getUserId(), task.getContextId());
    }

    private void constructNewStream() {
        zipOutputStream = ZippedFileStorageOutputStream.createDefaultZippedFileStorageOutputStream(fileStorageToWriteTo, Deflater.DEFAULT_COMPRESSION, useZip64);
        currentSize = 0;
    }

    private String transferSpooledFileWithRetry(String file, int maxRetries) throws OXException {
        String newFileStorageLocation = null;
        int retryCount = 0;
        do {
            try {
                newFileStorageLocation = transferSpooledFile(file);
            } catch (Exception e) {
                if (retryCount++ >= maxRetries || ExceptionUtils.isNoneOf(e, IOException.class, SAXParseException.class)) {
                    LOG.info("Failed to write file {} to destination storage.", file, e);
                    throw e;
                }

                // Retry using exponential back-off...
                LOG.info("Could not write file {} to destination storage on {}. attempt due to a retry-able error (\"{}\"). Retrying...", file, I(retryCount), e.getMessage());
                exponentialBackoffWait(retryCount, 1000L);
            }
        } while (newFileStorageLocation == null);
        return newFileStorageLocation;
    }

    private String transferSpooledFile(String oldFileStorageLocation) throws OXException {
        InputStream spoolData = fileStorageToWriteTo.getFile(oldFileStorageLocation);
        try {
            return destinationFileStorage.saveNewFile(spoolData);
        } finally {
            Streams.close(spoolData);
        }
    }

    /**
     * Performs a wait according to exponential back-off strategy.
     * <pre>
     * (retry-count * base-millis) + random-millis
     * </pre>
     *
     * @param retryCount The current number of retries
     * @param baseMillis The base milliseconds
     */
    private static void exponentialBackoffWait(int retryCount, long baseMillis) {
        long nanosToWait = TimeUnit.NANOSECONDS.convert((retryCount * baseMillis) + ((long) (Math.random() * baseMillis)), TimeUnit.MILLISECONDS);
        LockSupport.parkNanos(nanosToWait);
    }

}
