/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 1997-2018 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://oss.oracle.com/licenses/CDDL+GPL-1.1
 * or LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package javax.mail;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

/**
 * Clients use a FetchProfile to list the Message attributes that
 * it wishes to prefetch from the server for a range of messages.<p>
 *
 * Messages obtained from a Folder are light-weight objects that
 * typically start off as empty references to the actual messages.
 * Such a Message object is filled in "on-demand" when the appropriate
 * get*() methods are invoked on that particular Message. Certain
 * server-based message access protocols (Ex: IMAP) allow batch
 * fetching of message attributes for a range of messages in a single
 * request. Clients that want to use message attributes for a range of
 * Messages (Example: to display the top-level headers in a headerlist)
 * might want to use the optimization provided by such servers. The
 * <code>FetchProfile</code> allows the client to indicate this desire
 * to the server. <p>
 *
 * Note that implementations are not obligated to support
 * FetchProfiles, since there might be cases where the backend service
 * does not allow easy, efficient fetching of such profiles. <p>
 *
 * Sample code that illustrates the use of a FetchProfile is given
 * below:
 * <blockquote>
 * <pre>
 *
 *  Message[] msgs = folder.getMessages();
 *
 *  FetchProfile fp = new FetchProfile();
 *  fp.add(FetchProfile.Item.ENVELOPE);
 *  fp.add("X-mailer");
 *  folder.fetch(msgs, fp);
 *
 * </pre></blockquote><p>
 *
 * @see javax.mail.Folder#fetch
 * @author John Mani
 * @author Bill Shannon
 */

public class FetchProfile {

    /**
     * This inner class is the base class of all items that
     * can be requested in a FetchProfile. The items currently
     * defined here are <code>ENVELOPE</code>, <code>CONTENT_INFO</code>
     * and <code>FLAGS</code>. The <code>UIDFolder</code> interface
     * defines the <code>UID</code> Item as well. <p>
     *
     * Note that this class only has a protected constructor, therby
     * restricting new Item types to either this class or subclasses.
     * This effectively implements a enumeration of allowed Item types.
     *
     * @see UIDFolder
     */
    public static class Item {

        /**
         * This is the Envelope item. <p>
         *
         * The Envelope is an aggregration of the common attributes
         * of a Message. Implementations should include the following
         * attributes: From, To, Cc, Bcc, ReplyTo, Subject and Date.
         * More items may be included as well. <p>
         *
         * For implementations of the IMAP4 protocol (RFC 2060), the
         * Envelope should include the ENVELOPE data item. More items
         * may be included too.
         */
        public static final Item ENVELOPE = new Item("ENVELOPE");

        /**
         * This item is for fetching information about the
         * content of the message. <p>
         *
         * This includes all the attributes that describe the content
         * of the message. Implementations should include the following
         * attributes: ContentType, ContentDisposition,
         * ContentDescription, Size and LineCount. Other items may be
         * included as well.
         */
        public static final Item CONTENT_INFO = new Item("CONTENT_INFO");

        /**
         * SIZE is a fetch profile item that can be included in a
         * <code>FetchProfile</code> during a fetch request to a Folder.
         * This item indicates that the sizes of the messages in the specified
         * range should be prefetched. <p>
         *
         * @since   JavaMail 1.5
         */
        public static final Item SIZE = new Item("SIZE");

        /**
         * This is the Flags item.
         */
        public static final Item FLAGS = new Item("FLAGS");

        private final String name;

        /**
         * Constructor for an item.  The name is used only for debugging.
         *
         * @param   name    the item name
         */
        protected Item(String name) {
            this.name = name;
        }

        /**
         * Gets the item's name
         *
         * @return The item name
         */
        public String name() {
            return name;
        }

        /**
         * Include the name in the toString return value for debugging.
         */
	    @Override
        public String toString() {
            return getClass().getName() + "[" + name + "]";
        }
    } // End of class Item

    // -----------------------------------------------------------------------------------

    private final Set<Item> specials; // specials
    private final Map<String, String> headers; // set of header names

    /**
     * Create an empty FetchProfile.
     */
    public FetchProfile() {
        specials = new LinkedHashSet<Item>();
        headers = new LinkedHashMap<String, String>();
    }

    /**
     * Add the given special item as one of the attributes to
     * be prefetched.
     *
     * @param   item        the special item to be fetched
     * @see FetchProfile.Item#ENVELOPE
     * @see FetchProfile.Item#CONTENT_INFO
     * @see FetchProfile.Item#FLAGS
     */
    public void add(Item item) {
        specials.add(item);
    }

    /**
     * Add the specified header-field to the list of attributes
     * to be prefetched.
     *
     * @param   headerName  header to be prefetched
     */
    public void add(String headerName) {
        headers.put(asciiLowerCase(headerName), headerName);
    }

    /**
     * Returns true if the fetch profile contains the given special item.
     *
     * @param   item    the Item to test
     * @return true if the fetch profile contains the given special item
     */
    public boolean contains(Item item) {
        return specials.contains(item);
    }

    /**
     * Returns true if the fetch profile contains the given header name.
     *
     * @param   headerName  the header to test
     * @return  true if the fetch profile contains the given header name
     */
    public boolean contains(String headerName) {
        return null == headerName ? false : headers.containsKey(asciiLowerCase(headerName));
    }

    /**
     * Returns true if the fetch profile contains any of the given special items.
     *
     * @param items the Items to test
     * @return true if the fetch profile contains any of the given special items
     */
    public boolean contains(Item... items) {
        if (null != items) {
            for (int i = items.length; i-- > 0;) {
                if (specials.contains(items[i])) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Returns true if the fetch profile contains any of the given header names.
     *
     * @param headerNames the headers to test
     * @return true if the fetch profile contains any of the given header names
     */
    public boolean contains(String... headerNames) {
        if (null != headerNames) {
            for (int i = headerNames.length; i-- > 0;) {
                if (headers.containsKey(asciiLowerCase(headerNames[i]))) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Get the items set in this profile.
     *
     * @return      items set in this profile
     */
    public Item[] getItems() {
        int size = specials.size();
        return size == 0 ? new Item[0] : specials.toArray(new Item[size]);
    }

    /**
     * Get the names of the header-fields set in this profile.
     *
     * @return      headers set in this profile
     */
    public String[] getHeaderNames() {
        int size = headers.size();
        return size == 0 ? new String[0] : headers.values().toArray(new String[size]);
    }


    // @formatter:off
    private static char[] lowercases = {
        '\000', '\001', '\002', '\003', '\004', '\005', '\006', '\007', '\010', '\011', '\012', '\013', '\014', '\015', '\016', '\017',
        '\020', '\021', '\022', '\023', '\024', '\025', '\026', '\027', '\030', '\031', '\032', '\033', '\034', '\035', '\036', '\037',
        '\040', '\041', '\042', '\043', '\044', '\045', '\046', '\047', '\050', '\051', '\052', '\053', '\054', '\055', '\056', '\057',
        '\060', '\061', '\062', '\063', '\064', '\065', '\066', '\067', '\070', '\071', '\072', '\073', '\074', '\075', '\076', '\077',
        '\100', '\141', '\142', '\143', '\144', '\145', '\146', '\147', '\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
        '\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167', '\170', '\171', '\172', '\133', '\134', '\135', '\136', '\137',
        '\140', '\141', '\142', '\143', '\144', '\145', '\146', '\147', '\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
        '\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167', '\170', '\171', '\172', '\173', '\174', '\175', '\176', '\177' };
    // @formatter:on

    /**
     * Fast lower-case conversion.
     *
     * @param s The string
     * @return The lower-case string
     */
    @SuppressWarnings("null")
    private static String asciiLowerCase(String s) {
        if (null == s) {
            return null;
        }

        char[] c = null;
        int i = s.length();

        // look for first conversion
        while (i-- > 0) {
            char c1 = s.charAt(i);
            if (c1 <= 127) {
                char c2 = lowercases[c1];
                if (c1 != c2) {
                    c = s.toCharArray();
                    c[i] = c2;
                    break;
                }
            }
        }

        while (i-- > 0) {
            if (c[i] <= 127) {
                c[i] = lowercases[c[i]];
            }
        }

        return c == null ? s : new String(c);
    }
}
