/*
* @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
* @license AGPL-3.0
*
* This code is free software: you can redistribute it and/or modify
* it under the terms of the GNU Affero General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU Affero General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public License
* along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
* 
* Any use of the work other than as authorized under this license or copyright law is prohibited.
*
*/

package com.openexchange.tools;

import java.util.Set;
import com.google.common.collect.ImmutableSet;
import com.openexchange.ajax.requesthandler.annotation.restricted.RestrictedAction.Type;
import com.openexchange.folderstorage.ContentType;
import com.openexchange.folderstorage.SystemContentType;
import com.openexchange.folderstorage.addressbook.AddressDataContentType;
import com.openexchange.folderstorage.database.contentType.CalendarContentType;
import com.openexchange.folderstorage.database.contentType.ContactsContentType;
import com.openexchange.folderstorage.database.contentType.InfostoreContentType;
import com.openexchange.folderstorage.database.contentType.TaskContentType;
import com.openexchange.folderstorage.mail.contentType.DraftsContentType;
import com.openexchange.folderstorage.mail.contentType.MailContentType;
import com.openexchange.folderstorage.mail.contentType.SentContentType;
import com.openexchange.folderstorage.mail.contentType.SpamContentType;
import com.openexchange.folderstorage.mail.contentType.TrashContentType;
import com.openexchange.groupware.modules.Module;
import com.openexchange.oauth.provider.exceptions.OAuthInsufficientScopeException;
import com.openexchange.oauth.provider.resourceserver.OAuthAccess;

/**
 * {@link OAuthContentTypes} - A static helper class to deal with OAuth content types
 * <code>calendar</code>, <code>contacts</code>, <code>files</code>, <code>mail</code> and <code>tasks</code>.
 *
 * @author <a href="mailto:anna.ottersbach@open-xchange.com">Anna Schuerholz</a>
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public final class OAuthContentTypes {

    /**
     * Gets the {@link ContentType}s valid for the given read oauth scope
     *
     * @param scope The read oauth scope
     * @return A set of valid {@link ContentType}s
     */
    public static Set<ContentType> contentTypesForReadScope(String scope) {
        return contentTypesForModule(Module.getForName(Type.READ.getModule(scope)));
    }

    /**
     * Gets the {@link ContentType}s valid for the given write oauth scope
     *
     * @param scope The write oauth scope
     * @return A set of valid {@link ContentType}s
     */
    public static Set<ContentType> contentTypesForWriteScope(String scope) {
        return contentTypesForModule(Module.getForName(Type.WRITE.getModule(scope)));
    }

    /**
     * Returns the a set with the content types that pass to the specified module
     * 
     * @param module The module
     * @return The content types
     */
    private static Set<ContentType> contentTypesForModule(Module module) {
        if (module == null) {
            return ImmutableSet.of();
        }
        switch (module) {
            case CONTACTS:
                return ImmutableSet.of(AddressDataContentType.getInstance(), ContactsContentType.getInstance());
            case CALENDAR:
                return ImmutableSet.of(CalendarContentType.getInstance(), com.openexchange.folderstorage.calendar.contentType.CalendarContentType.getInstance());
            case TASK:
                return ImmutableSet.of((ContentType) TaskContentType.getInstance());
            case MAIL:
                return ImmutableSet.of(MailContentType.getInstance(), DraftsContentType.getInstance(), SentContentType.getInstance(), TrashContentType.getInstance(), SpamContentType.getInstance());
            case INFOSTORE:
                return ImmutableSet.of((ContentType) InfostoreContentType.getInstance());
            default:
                return ImmutableSet.of();
        }
    }

    /**
     * Checks whether read operations are permitted for the given folder content type and OAuth oauthAccess.
     *
     * @param contentType The content type
     * @param oauthAccess The oauthAccess
     * @return <code>true</code> if read operations are permitted
     * @throws OAuthInsufficientScopeException If read operations are permitted.
     */
    public static boolean mayReadViaOAuthRequest(ContentType contentType, OAuthAccess oauthAccess) throws OAuthInsufficientScopeException {
        if (contentType == SystemContentType.getInstance()) {
            return true;
        }
        String scope = OAuthContentTypes.readScopeForContentType(contentType);
        if (scope != null && oauthAccess.getScope().has(scope)) {
            return true;
        }
        throw new OAuthInsufficientScopeException(scope);
    }

    /**
     * Checks whether read operations are permitted for the given folder content types and OAuth oauthAccess.
     *
     * @param contentTypes The content types
     * @param oauthAccess The oauthAccess
     * @return <code>true</code> if read operations are permitted
     * @throws OAuthInsufficientScopeException If read operations are permitted.
     */
    public static boolean mayReadViaOAuthRequest(Set<ContentType> contentTypes, OAuthAccess oauthAccess) throws OAuthInsufficientScopeException {
        for (ContentType contentType : contentTypes) {
            mayReadViaOAuthRequest(contentType, oauthAccess);
        }
        return true;
    }

    /**
     * Checks whether write operations are permitted for the given folder content type and OAuth oauthAccess.
     *
     * @param contentType The content type
     * @param oauthAccess The oauthAccess
     * @return <code>true</code> if write operations are permitted
     * @throws OAuthInsufficientScopeException If write operations are permitted.
     */
    public static boolean mayWriteViaOAuthRequest(ContentType contentType, OAuthAccess oauthAccess) throws OAuthInsufficientScopeException {
        String scope = OAuthContentTypes.writeScopeForContentType(contentType);
        if (scope != null && oauthAccess.getScope().has(scope)) {
            return true;
        }
        throw new OAuthInsufficientScopeException(scope);
    }

    /**
     * Checks whether write operations are permitted for the given folder content types and OAuth oauthAccess.
     *
     * @param contentTypes The content types
     * @param oauthAccess The oauthAccess
     * @return <code>true</code> if write operations are permitted
     * @throws OAuthInsufficientScopeException If write operations are permitted.
     */
    public static boolean mayWriteViaOAuthRequest(Set<ContentType> contentTypes, OAuthAccess oauthAccess) throws OAuthInsufficientScopeException {
        for (ContentType contentType : contentTypes) {
            mayWriteViaOAuthRequest(contentType, oauthAccess);
        }
        return true;
    }

    /**
     * 
     * Gets the scope consisting of the prefix <code>read_</code> and the content type.
     *
     * @param contentType The content type.
     * @return The scope.
     */
    public static String readScopeForContentType(ContentType contentType) {
        if (contentType == AddressDataContentType.getInstance() || contentType == ContactsContentType.getInstance()) {
            return Type.READ.getScope(Module.CONTACTS.getName());
        } else if (contentType == CalendarContentType.getInstance()) {
            return Type.READ.getScope(Module.CALENDAR.getName());
        } else if (contentType == com.openexchange.folderstorage.calendar.contentType.CalendarContentType.getInstance()) {
            return Type.READ.getScope(Module.CALENDAR.getName());
        } else if (contentType == TaskContentType.getInstance()) {
            return Type.READ.getScope(Module.TASK.getName());
        } else if (contentType == MailContentType.getInstance() || contentType == DraftsContentType.getInstance() || contentType == SentContentType.getInstance() || contentType == SpamContentType.getInstance() || contentType == TrashContentType.getInstance()) {
            return Type.READ.getScope(Module.MAIL.getName());
        } else if (contentType == InfostoreContentType.getInstance()) {
            return Type.READ.getScope(Module.INFOSTORE.getName());
        }

        return null;
    }

    /**
     * 
     * Gets the scope consisting of the prefix <code>write_</code> and the content type.
     *
     * @param contentType The content type.
     * @return The scope.
     */
    public static String writeScopeForContentType(ContentType contentType) {
        if (contentType == AddressDataContentType.getInstance() || contentType == ContactsContentType.getInstance()) {
            return Type.WRITE.getScope(Module.CONTACTS.getName());
        } else if (contentType == CalendarContentType.getInstance()) {
            return Type.WRITE.getScope(Module.CALENDAR.getName());
        } else if (contentType == com.openexchange.folderstorage.calendar.contentType.CalendarContentType.getInstance()) {
            return Type.WRITE.getScope(Module.CALENDAR.getName());
        } else if (contentType == TaskContentType.getInstance()) {
            return Type.WRITE.getScope(Module.TASK.getName());
        } else if (contentType == MailContentType.getInstance() || contentType == DraftsContentType.getInstance() || contentType == SentContentType.getInstance() || contentType == SpamContentType.getInstance() || contentType == TrashContentType.getInstance()) {
            return Type.WRITE.getScope(Module.MAIL.getName());
        } else if (contentType == InfostoreContentType.getInstance()) {
            return Type.WRITE.getScope(Module.INFOSTORE.getName());
        }

        return null;
    }

    /**
     * Initialises a new {@link OAuthContentTypes}.
     */
    private OAuthContentTypes() {
        throw new IllegalStateException("Utility class");
    }
}
