/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mail.mime;

import static com.openexchange.java.Strings.asciiLowerCase;
import static com.openexchange.java.Strings.isEmpty;
import static com.openexchange.java.Strings.toUpperCase;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.StringTokenizer;
import java.util.regex.Pattern;
import javax.mail.internet.AddressException;
import javax.mail.internet.FatalAddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeUtility;
import javax.mail.internet.idn.IDNA;
import org.apache.commons.validator.routines.EmailValidator;
import org.hazlewood.connor.bottema.emailaddress.EmailAddressCriteria;
import org.hazlewood.connor.bottema.emailaddress.EmailAddressParser;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.Interests;
import com.openexchange.config.Reloadable;
import com.openexchange.config.Reloadables;
import com.openexchange.java.InterruptibleCharSequence;
import com.openexchange.java.InterruptibleCharSequence.InterruptedRuntimeException;
import com.openexchange.java.Strings;
import com.openexchange.java.util.MsisdnCheck;
import com.openexchange.mail.config.MailProperties;
import com.openexchange.mail.config.MailReloadable;
import com.openexchange.mail.mime.utils.EmailAddressInfo;
import com.openexchange.mail.mime.utils.EmailAddressParserWatcher;
import com.openexchange.mail.mime.utils.MimeMessageUtility;
import com.openexchange.server.services.ServerServiceRegistry;

/**
 * {@link QuotedInternetAddress} - A quoted version of {@link InternetAddress} originally written by <b>Bill Shannon</b> and <b>John
 * Mani</b>. Moreover this class supports <a href="http://en.wikipedia.org/wiki/Punycode">punycode</a>.
 * <p>
 * Quotes are added to encoded personal names to maintain them when converting to mail-safe version. Parental {@link InternetAddress} class
 * ignores quotes when when converting to mail-safe version:
 * <p>
 * <code>``"M&uuml;ller,&nbsp;Jan"&nbsp;&lt;mj@foo.de&gt;''</code><br>
 * is converted to<br>
 * <code>``=?UTF-8?Q?M=C3=BCller=2C_Jan?=&nbsp;&lt;mj@foo.de&gt;''</code>
 * <p>
 * This class maintains the quotes in mail-safe version:
 * <p>
 * <code>``"M&uuml;ller,&nbsp;Jan"&nbsp;&lt;mj@foo.de&gt;''</code><br>
 * is converted to<br>
 * <code>``=?UTF-8?Q?=22M=C3=BCller=2C_Jan=22?=&nbsp;&lt;mj@foo.de&gt;''</code>
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public final class QuotedInternetAddress extends InternetAddress {

    private static final long serialVersionUID = -2523736473507495692L;

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(QuotedInternetAddress.class);

    private static final boolean IGNORE_BOGUS_GROUP_NAME = getBooleanSystemProperty("mail.mime.address.ignorebogusgroupname", true);

    private static boolean getBooleanSystemProperty(String name, boolean def) {
        return Boolean.parseBoolean(System.getProperty(name, def ? "true" : "false"));
    }

    private static volatile Boolean preferSimpleAddressParsing;
    private static boolean preferSimpleAddressParsing() {
        Boolean tmp = preferSimpleAddressParsing;
        if (null == tmp) {
            synchronized (QuotedInternetAddress.class) {
                tmp = preferSimpleAddressParsing;
                if (null == tmp) {
                    boolean defaultValue = true;
                    ConfigurationService service = ServerServiceRegistry.getInstance().getService(ConfigurationService.class);
                    if (null == service) {
                        return defaultValue;
                    }

                    tmp = Boolean.valueOf(service.getBoolProperty("com.openexchange.mail.preferSimpleAddressParsing", defaultValue));
                    preferSimpleAddressParsing = tmp;
                }
            }
        }
        return tmp.booleanValue();
    }

    private static volatile Boolean keepQuotesInEncodedPersonal;
    private static boolean keepQuotesInEncodedPersonal() {
        Boolean tmp = keepQuotesInEncodedPersonal;
        if (null == tmp) {
            synchronized (QuotedInternetAddress.class) {
                tmp = keepQuotesInEncodedPersonal;
                if (null == tmp) {
                    // A workaround to fix bug 14050 (fat clients interpreting a comma as address separator), which became obsolete in the meantime
                    // Therefore assuming 'false' as default here
                    boolean defaultValue = false;
                    ConfigurationService service = ServerServiceRegistry.getInstance().getService(ConfigurationService.class);
                    if (null == service) {
                        return defaultValue;
                    }

                    tmp = Boolean.valueOf(service.getBoolProperty("com.openexchange.mail.keepQuotesInEncodedPersonal", defaultValue));
                    keepQuotesInEncodedPersonal = tmp;
                }
            }
        }
        return tmp.booleanValue();
    }

    private static volatile Boolean checkTopLevelDomainOnAddressValidation;
    private static boolean checkTopLevelDomainOnAddressValidation() {
        Boolean tmp = checkTopLevelDomainOnAddressValidation;
        if (null == tmp) {
            synchronized (QuotedInternetAddress.class) {
                tmp = checkTopLevelDomainOnAddressValidation;
                if (null == tmp) {
                    boolean defaultValue = false;
                    ConfigurationService service = ServerServiceRegistry.getInstance().getService(ConfigurationService.class);
                    if (null == service) {
                        return defaultValue;
                    }

                    tmp = Boolean.valueOf(service.getBoolProperty("com.openexchange.mail.checkTopLevelDomainOnAddressValidation", defaultValue));
                    checkTopLevelDomainOnAddressValidation = tmp;
                }
            }
        }
        return tmp.booleanValue();
    }

    static {
        MailReloadable.getInstance().addReloadable(new Reloadable() {

            @Override
            public void reloadConfiguration(ConfigurationService configService) {
                preferSimpleAddressParsing = null;
                keepQuotesInEncodedPersonal = null;
                checkTopLevelDomainOnAddressValidation = null;
            }

            @Override
            public Interests getInterests() {
                return Reloadables.interestsForProperties(
                    "com.openexchange.mail.preferSimpleAddressParsing",
                    "com.openexchange.mail.keepQuotesInEncodedPersonal",
                    "com.openexchange.mail.checkTopLevelDomainOnAddressValidation");
            }
        });
    }

    /**
     * Validates specified E-Mail address.
     *
     * @param address The E-Mail address to validate
     * @throws AddressException If E-Mail address seems to be invalid
     * @see #isValid(String)
     */
    public static void validate(String address) throws AddressException {
        if (doIsValid(address, true) == false) {
            throw new AddressException("E-Mail address is invalid: " + address);
        }
    }

    /**
     * Checks for validity of specified E-Mail address.
     *
     * @param address The E-Mail address to validate
     * @return <code>true</code> if E-Mail address seems to be valid; otherwise <code>false</code> for invalid
     */
    public static boolean isValid(String address) {
        try {
            return doIsValid(address, false);
        } catch (AddressException e) {
            LOG.debug("Considering E-Mail address ``{}'' as invalid", address, e);
            return false;
        }
    }

    /**
     * Checks for validity of specified E-Mail address.
     *
     * @param address The E-Mail address to validate
     * @return <code>true</code> if E-Mail address seems to be valid; otherwise <code>false</code> for invalid
     * @throws AddressException If E-Mail address seems to be invalid and <code>rethrowAddressException</code> has been set to <code>true</code>
     */
    private static boolean doIsValid(String address, boolean rethrowAddressException) throws AddressException {
        if (Strings.isEmpty(address)) {
            // Empty addresses are not valid...
            return false;
        }

        try {
            QuotedInternetAddress addr = new QuotedInternetAddress(address);
            addr.validate();
            if (checkTopLevelDomainOnAddressValidation() && EmailValidator.getInstance().isValid(addr.getAddress()) == false) {
                // Invalid or missing top-level domain
                return false;
            }
            return true;
        } catch (AddressException e) {
            if (rethrowAddressException) {
                throw e;
            }
            LOG.debug("Considering E-Mail address ``{}'' as invalid", address, e);
            return false;
        }
    }

    /**
     * Converts given array of {@link InternetAddress} to quoted addresses
     *
     * @param addrs The addresses to convert
     * @return The quoted addresses
     * @throws AddressException If conversion fails
     */
    public static InternetAddress[] toQuotedAddresses(InternetAddress[] addrs) throws AddressException {
        if (null == addrs) {
            return null;
        }
        final InternetAddress[] ret = new InternetAddress[addrs.length];
        for (int i = 0; i < ret.length; i++) {
            ret[i] = new QuotedInternetAddress(addrs[i]);
        }
        return ret;
    }

    /**
     * Parse the given comma separated sequence of addresses into {@link InternetAddress} objects. Addresses must follow RFC822 syntax.
     *
     * @param addresslist A comma separated address strings
     * @return An array of {@link InternetAddress} objects
     * @exception AddressException If the parse failed
     */
    public static InternetAddress[] parse(String addresslist) throws AddressException {
        return parse(addresslist, true);
    }

    /**
     * Parse the given sequence of addresses into {@link InternetAddress} objects. If <code>strict</code> is false, simple email addresses
     * separated by spaces are also allowed. If <code>strict</code> is true, many (but not all) of the RFC822 syntax rules are enforced. In
     * particular, even if <code>strict</code> is true, addresses composed of simple names (with no "@domain" part) are allowed. Such
     * "illegal" addresses are not uncommon in real messages.
     * <p>
     * Non-strict parsing is typically used when parsing a list of mail addresses entered by a human. Strict parsing is typically used when
     * parsing address headers in mail messages.
     *
     * @param addresslist A comma separated address strings
     * @param strict <code>true</code> to enforce RFC822 syntax; otherwise <code>false</code>
     * @return An array of {@link InternetAddress} objects
     * @exception AddressException If the parse failed
     */
    public static InternetAddress[] parse(String addresslist, boolean strict) throws AddressException {
        if (preferSimpleAddressParsing()) {
            return parseSimple(addresslist, strict);
        }

        return parse0(addresslist, strict);
    }

    private static InternetAddress[] parse0(String addresslist, boolean strict) throws AddressException {
        try {
            return parse(addresslist, strict, false, false);
        } catch (FatalAddressException e) {
            throw e;
        } catch (AddressException e) {
            return parse(addresslist, strict, false, true);
        }
    }

    /**
     * Parse the given sequence of addresses into {@link InternetAddress} objects. If <code>strict</code> is false, the full syntax rules
     * for individual addresses are not enforced. If <code>strict</code> is true, many (but not all) of the RFC822 syntax rules are
     * enforced.
     * <p>
     * To better support the range of "invalid" addresses seen in real messages, this method enforces fewer syntax rules than the
     * <code>parse</code> method when the strict flag is false and enforces more rules when the strict flag is true. If the strict flag is
     * false and the parse is successful in separating out an email address or addresses, the syntax of the addresses themselves is not
     * checked.
     *
     * @param addresslist A comma separated address strings
     * @param strict <code>true</code> to enforce RFC822 syntax; otherwise <code>false</code>
     * @return An array of {@link InternetAddress} objects
     * @exception AddressException If the parse failed
     */
    public static InternetAddress[] parseHeader(String addresslist, boolean strict) throws AddressException {
        if (preferSimpleAddressParsing()) {
            return parseSimple(addresslist, strict);
        }

        try {
            return parse(addresslist, strict, true, true);
        } catch (AddressException e) {
            return parse(addresslist, strict, true, false);
        }
    }

    private static InternetAddress[] parseSimple(String str, boolean strict) throws AddressException {
        if (str.length() <= 0) {
            return new InternetAddress[0];
        }

        String[] addrs = Strings.splitByCommaNotInQuotes(str);
        List<InternetAddress> l = new ArrayList<InternetAddress>(addrs.length);
        StringBuilder fullB = null; // Init when needed
        for (int i = 0; i < addrs.length; i++) {
            String addr = addrs[i];
            if (addr.lastIndexOf('<') < 0 && addr.indexOf("=?") >= 0) {
                addr = init(MimeMessageUtility.decodeMultiEncodedHeader(addr));
            } else if (addr.indexOf("'?= <") > 0) { // NOSONARLINT
                // Expect something like: =?utf-8?Q?...'?= <jane@doe.org>
                String tmp = init(MimeMessageUtility.decodeMultiEncodedHeader(addr));

                // Check if personal part is surrounded by single-quotes
                if (tmp != null && tmp.startsWith("'")) {
                    int pos = tmp.indexOf("' <");
                    if (pos > 0) {
                        // Replace with double-quotes
                        addr = new StringBuilder(tmp.length()).append('"').append(tmp.substring(1, pos)).append("\" <").append(tmp.substring(pos + 3)).toString();
                    }
                }
            }

            boolean added = false;
            if (addr != null && addrs.length > i + 1 && addr.indexOf('@') < 0) {
                // Appears no address at all. Check for unquoted personal with comma like: "Doe, Jane <jane.doe@nowehere.com>"
                if (fullB == null) {
                    fullB = new StringBuilder(str.length());
                } else {
                    fullB.setLength(0);
                }
                int j = i;
                do {
                    if (fullB.length() > 0) {
                        fullB.append(", ");
                    }
                    fullB.append(addr);
                    j++;
                    addr = addrs[j];
                } while (addr != null && addrs.length > j + 1 && addr.indexOf('@') < 0);
                if (addr != null && addr.indexOf('@') >= 0) {
                    fullB.append(", ");
                    fullB.append(addr);
                    String full = fullB.toString();
                    int addrStart = full.indexOf('<');
                    if (addrStart > 0 && full.length() > addrStart + 1) {
                        int addrEnd = full.lastIndexOf('>');
                        if (addrEnd > 0 && addrEnd > addrStart && addrEnd == full.length() - 1) {
                            int index = addrStart - 1;
                            while (Strings.isWhitespace(full.charAt(index))) {
                                index--;
                            }
                            // Inject missing quotes & parse
                            fullB.setLength(0);
                            full = fullB.append('"').append(full.substring(0, index + 1)).append('"').append(full.substring(index + 1)).toString();
                            l.add(new QuotedInternetAddress(full, strict));
                            i = j;
                            added = true;
                        }
                    }
                }
            }

            if (!added) {
                l.add(new QuotedInternetAddress(addr, strict));
            }
        }
        return l.toArray(new InternetAddress[l.size()]);
    }

    private static String dropComments(String str, boolean ignoreErrors) throws AddressException {
        String s = str;
        boolean nextRun = true;
        int start = 0;
        while (nextRun) {
            boolean lookForBracket = true;
            boolean found = false;
            boolean quoted = false;
            int pos = 0;
            int nest = 0;
            int length = s.length();
            int i;
            for (i = start; lookForBracket && i < length; i++) {
                char c = s.charAt(i);
                switch (c) {
                    case '\\':
                        i++; // skip both; '\' and the escaped character
                        break;
                    case '"':
                        quoted = !quoted;
                        break;
                    case '(':
                        if (!quoted) {
                            nest++;
                            if (nest == 1) {
                                found = true;
                                pos = i;
                            }
                        }
                        break;
                    case ')':
                        if (!quoted) {
                            nest--;
                            if (nest == 0) {
                                i++; // Skip closing bracket
                                s = s.substring(0, pos) + s.substring(i);
                                lookForBracket = false;
                                start = pos + 1;
                            }
                        }
                        break;
                    default:
                        break;
                }
            }
            if (nest > 0) {
                if (!ignoreErrors) {
                    throw new AddressException("Missing ')'", s, i);
                }
                // pretend the first paren was a regular character and
                // continue parsing after it
                i = pos + 1;
                if (i < length) {
                    s = s.substring(0, pos) + s.substring(i);
                } else {
                    s = s.substring(0, pos);
                }
            }
            if (!found) {
                nextRun = false;
            }
        }
        return s;
    }

    private static CharSequenceAndInfo addToWatcherIfPossible(String s) {
        EmailAddressInfo info = null;
        try {
            CharSequence toParse = s;

            int waitMillis = MailProperties.getInstance().getEmailAddressParserWaitMillis();
            if (waitMillis > 0) {
                InterruptibleCharSequence interruptible = InterruptibleCharSequence.valueOf(s);
                info = new EmailAddressInfo(Thread.currentThread(), interruptible, waitMillis);
                if (EmailAddressParserWatcher.getInstance().add(info)) {
                    // Added to watcher. Reset character sequence to parse.
                    toParse = interruptible;
                } else {
                    // Could not be added to watcher. Drop info instance.
                    info = null;
                }
            }

            CharSequenceAndInfo retval = new CharSequenceAndInfo(toParse, info);
            info = null; // All fine
            return retval;
        } finally {
            if (info != null) {
                // Exception path...
                EmailAddressParserWatcher.getInstance().remove(info);
            }
        }
    }

    /*
     * RFC822 Address parser. XXX - This is complex enough that it ought to be a real parser, not this ad-hoc mess, and because of that,
     * this is not perfect. XXX - Deal with encoded Headers too.
     */
    private static InternetAddress[] parse(String str, boolean strict, boolean parseHdr, boolean decodeFirst) throws AddressException {
        String s = init(decodeFirst ? MimeMessageUtility.decodeMultiEncodedHeader(str) : str);

        /*-
         * Parse with EmailAddressParser if address listing appears to contain a (comment) since EmailAddressParser allows to extract CFWS
         * personal names:
         *
         * Regarding the parameter extractCfwsPersonalNames:
         * It allows the not-totally-kosher-but-happens-in-the-real-world practice of:
         *
         *   <bob@example.com> (Bob Smith)
         *
         * In this case, "Bob Smith" is not technically the personal name, just a comment. If this is included, the methods will convert
         * this into: Bob Smith <bob@example.com>
         *
         * This also happens somewhat more often and appropriately with
         *
         *   mailer-daemon@blah.com (Mail Delivery System)
         *
         * If a personal name appears to the left and CFWS appears to the right of an address, the methods will favor the personal name to
         * the left. If the methods need to use the CFWS following the address, they will take the first comment token they find.
         */
        InternetAddress[] emailAddressParserAddresses;
        if (s.indexOf('(') < 0) {
            // Apparently no comments
            emailAddressParserAddresses = null;
        } else {
            // Possible comments. Parse with (watched) EmailAddressParser to maintain CFWS personal names (if any)
            CharSequenceAndInfo sequenceAndInfo = addToWatcherIfPossible(s);
            try {
                emailAddressParserAddresses = EmailAddressParser.extractHeaderAddresses(sequenceAndInfo.toParse, EmailAddressCriteria.RFC_COMPLIANT, true);
            } catch (InterruptedRuntimeException e) {
                LOG.warn("Interrupted while parsing address listing with EmailAddress RFC2822: {}", s, e);
                emailAddressParserAddresses = null;
            } catch (Exception e) {
                LOG.warn("Failed to parse address listing with EmailAddress RFC2822: {}", s, e);
                emailAddressParserAddresses = null;
            } catch (StackOverflowError e) {
                if (LOG.isDebugEnabled()) {
                    LOG.warn("A stack overflow occurred (application recursed too deeply) while parsing address listing with EmailAddress RFC2822: {}", s, e);
                } else {
                    LOG.warn("A stack overflow occurred (application recursed too deeply) while parsing address listing with EmailAddress RFC2822: {}", s);
                }
                emailAddressParserAddresses = null;
            } finally {
                if (sequenceAndInfo.info != null) {
                    EmailAddressParserWatcher.getInstance().remove(sequenceAndInfo.info);
                }
            }
        }

        int start, end, index, nesting;
        int start_personal = -1, end_personal = -1;
        boolean ignoreErrors = parseHdr && !strict;

        s = dropComments(s, ignoreErrors);
        int length = s.length();
        List<InternetAddress> list = new LinkedList<InternetAddress>();
        boolean in_group = false; // we're processing a group term
        boolean route_addr = false; // address came from route-addr term
        boolean rfc822 = false; // looks like an RFC822 address
        QuotedInternetAddress qia;

        for (start = end = -1, index = 0; index < length; index++) {
            char c = s.charAt(index);
            switch (c) {
            case '(': // We are parsing a Comment. Ignore everything inside.
                // XXX - comment fields should be parsed as whitespace,
                // more than one allowed per address
                rfc822 = true;
                if (start >= 0 && end == -1) {
                    end = index;
                }
                final int pindex = index;
                for (index++, nesting = 1; index < length && nesting > 0; index++) {
                    c = s.charAt(index);
                    switch (c) {
                    case '\\':
                        index++; // skip both '\' and the escaped char
                        break;
                    case '(':
                        nesting++;
                        break;
                    case ')':
                        nesting--;
                        break;
                    default:
                        break;
                    }
                }
                if (nesting > 0) {
                    if (!ignoreErrors) {
                        throw new AddressException("Missing ')'", s, index);
                    }
                    // pretend the first paren was a regular character and
                    // continue parsing after it
                    index = pindex + 1;
                    break;
                }
                index--; // point to closing paren
                if (start_personal == -1) {
                    start_personal = pindex + 1;
                }
                if (end_personal == -1) {
                    end_personal = index;
                }
                break;

            case ')':
                if (!ignoreErrors) {
                    throw new AddressException("Missing '('", s, index);
                }
                // pretend the left paren was a regular character and
                // continue parsing
                if (start == -1) {
                    start = index;
                }
                break;

            case '<':
                rfc822 = true;
                if (route_addr) {
                    if (!ignoreErrors) {
                        throw new AddressException("Extra route-addr", s, index);
                    }

                    // assume missing comma between addresses
                    if (start == -1) {
                        route_addr = false;
                        rfc822 = false;
                        start = end = -1;
                        break; // nope, nothing there
                    }
                    if (!in_group) {
                        // got a token, add this to our InternetAddress vector
                        if (end == -1) {
                            end = index;
                        }
                        final String addr = s.substring(start, end).trim();

                        qia = new QuotedInternetAddress();
                        qia.setAddress(toACE(addr));
                        if (start_personal >= 0) {
                            qia.encodedPersonal = unquote(s.substring(start_personal, end_personal).trim());
                        }
                        list.add(qia);

                        route_addr = false;
                        rfc822 = false;
                        start = end = -1;
                        start_personal = end_personal = -1;
                        // continue processing this new address...
                    }
                }

                final int rindex = index;
                boolean inquote = false;
                int opens = 1;
                outf: for (index++; index < length; index++) {
                    c = s.charAt(index);
                    switch (c) {
                    case '\\': // XXX - is this needed?
                        index++; // skip both '\' and the escaped char
                        break;
                    case '"':
                        inquote ^= true;
                        break;
                    case '<':
                        // Another opening
                        opens++;
                        break;
                    case '>':
                        if (inquote) {
                            continue;
                        }
                        if (--opens == 0) {
                            break outf; // out of for loop
                        }
                        break;
                    default:
                        break;
                    }
                }

                // did we find a matching quote?
                if (inquote) {
                    if (!ignoreErrors) {
                        throw new AddressException("Missing '\"'", s, index);
                    }
                    // didn't find matching quote, try again ignoring quotes
                    // (e.g., ``<"@foo.com>'')
                    for (index = rindex + 1; index < length; index++) {
                        c = s.charAt(index);
                        if (c == '\\') {
                            index++; // skip both '\' and the escaped char
                        } else if (c == '>') {
                            break;
                        }
                    }
                }

                // did we find a terminating '>'?
                if (index >= length) {
                    if (!ignoreErrors) {
                        throw new AddressException("Missing '>'", s, index);
                    }
                    // pretend the "<" was a regular character and
                    // continue parsing after it (e.g., ``<@foo.com'')
                    index = rindex + 1;
                    if (start == -1) {
                        start = rindex; // back up to include "<"
                    }
                    break;
                }

                if (!in_group) {
                    start_personal = start;
                    if (start_personal >= 0) {
                        end_personal = rindex;
                    }
                    start = rindex + 1;
                }
                route_addr = true;
                end = index;
                break;

            case '>':
                if (!ignoreErrors) {
                    throw new AddressException("Missing '<'", s, index);
                }
                // pretend the ">" was a regular character and
                // continue parsing (e.g., ``>@foo.com'')
                if (start == -1) {
                    start = index;
                }
                break;

            case '"': // parse quoted string
                final int qindex = index;
                rfc822 = true;
                if (start == -1) {
                    start = index;
                }
                outq: for (index++; index < length; index++) {
                    c = s.charAt(index);
                    switch (c) {
                    case '\\':
                        index++; // skip both '\' and the escaped char
                        break;
                    case '"':
                        break outq; // out of for loop
                    default:
                        break;
                    }
                }
                if (index >= length) {
                    if (!ignoreErrors) {
                        throw new AddressException("Missing '\"'", s, index);
                    }
                    // pretend the quote was a regular character and
                    // continue parsing after it (e.g., ``"@foo.com'')
                    index = qindex + 1;
                }
                break;

            case '[': // a domain-literal, probably
                int lindex = index;
                rfc822 = true;
                if (start == -1) {
                    start = index;
                }
                outb: for (index++; index < length; index++) {
                    c = s.charAt(index);
                    switch (c) {
                    case '\\':
                        index++; // skip both '\' and the escaped char
                        break;
                    case ']':
                        break outb; // out of for loop
                    default:
                        break;
                    }
                }
                if (index >= length) {
                    if (!ignoreErrors) {
                        throw new AddressException("Missing ']'", s, index);
                    }
                    // pretend the "[" was a regular character and
                    // continue parsing after it (e.g., ``[@foo.com'')
                    index = lindex + 1;
                }
                break;

            case ';':
                if (start == -1) {
                    route_addr = false;
                    rfc822 = false;
                    start = end = -1;
                    break; // nope, nothing there
                }
                if (in_group) {
                    in_group = false;
                    /*
                     * If parsing headers, but not strictly, peek ahead. If next char is "@", treat the group name like the local part of
                     * the address, e.g., "Undisclosed-Recipient:;@java.sun.com".
                     */
                    if (parseHdr && !strict && index + 1 < length && s.charAt(index + 1) == '@') {
                        break;
                    }
                    qia = new QuotedInternetAddress();
                    end = index + 1;
                    qia.setAddress(toACE(s.substring(start, end).trim()));
                    list.add(qia);

                    route_addr = false;
                    rfc822 = false;
                    start = end = -1;
                    start_personal = end_personal = -1;
                    break;
                }
                if (!ignoreErrors) {
                    throw new AddressException("Illegal semicolon, not in group", s, index);
                }

                // otherwise, parsing a header; treat semicolon like comma
                // fall through to comma case...
                //$FALL-THROUGH$

            case ',': // end of an address, probably
                if (start == -1) {
                    route_addr = false;
                    rfc822 = false;
                    start = end = -1;
                    break; // nope, nothing there
                }
                if (in_group) {
                    route_addr = false;
                    break;
                }
                // got a token, add this to our InternetAddress vector
                if (end == -1) {
                    end = index;
                }

                String addr = s.substring(start, end).trim();
                String pers = null;
                if (rfc822 && start_personal >= 0) {
                    pers = unquote(s.substring(start_personal, end_personal).trim());
                    if (isEmpty(pers)) {
                        pers = null;
                    }
                }

                /*
                 * If the personal name field has an "@" and the address field does not, assume they were reversed, e.g., ``"joe doe"
                 * (john.doe@example.com)''.
                 */
                if (parseHdr && !strict && pers != null && pers.indexOf('@') >= 0 && addr.indexOf('@') < 0 && addr.indexOf('!') < 0) {
                    final String tmp = addr;
                    addr = pers;
                    pers = tmp;
                }
                if (rfc822 || strict || parseHdr) {
                    final String ace = toACE(addr);
                    if (!ignoreErrors) {
                        checkAddress(ace, route_addr, false);
                    }
                    qia = new QuotedInternetAddress();
                    qia.setAddress(ace);
                    if (pers != null) {
                        qia.encodedPersonal = pers;
                    }
                    list.add(qia);
                } else {
                    // maybe we passed over more than one space-separated addr
                    final StringTokenizer st = new StringTokenizer(addr);
                    while (st.hasMoreTokens()) {
                        final String a = st.nextToken();
                        final String ace = toACE(a);
                        checkAddress(ace, false, false);
                        qia = new QuotedInternetAddress();
                        qia.setAddress(ace);
                        list.add(qia);
                    }
                }

                route_addr = false;
                rfc822 = false;
                start = end = -1;
                start_personal = end_personal = -1;
                break;

            case ':':
                rfc822 = true;
                if (in_group) {
                    if (!ignoreErrors) {
                        throw new AddressException("Nested group", s, index);
                    }
                }
                if (start == -1) {
                    start = index;
                }
                if (parseHdr && !strict) {
                    /*
                     * If next char is a special character that can't occur at the start of a valid address, treat the group name as the
                     * entire address, e.g., "Date:, Tue", "Re:@foo".
                     */
                    if (index + 1 < length) {
                        final String addressSpecials = ")>[]:@\\,.";
                        char nc = s.charAt(index + 1);
                        if (addressSpecials.indexOf(nc) >= 0) {
                            if (nc != '@') {
                                break; // don't change in_group
                            }
                            /*
                             * Handle a common error: ``Undisclosed-Recipient:@example.com;'' Scan ahead. If we find a semicolon before one
                             * of these other special characters, consider it to be a group after all.
                             */
                            for (int i = index + 2; i < length; i++) {
                                nc = s.charAt(i);
                                if (nc == ';') {
                                    break;
                                }
                                if (addressSpecials.indexOf(nc) >= 0) {
                                    break;
                                }
                            }
                            if (nc == ';') {
                                break; // don't change in_group
                            }
                        }
                    }

                    // ignore bogus "mailto:" prefix in front of an address,
                    // or bogus mail header name included in the address field
                    final String gname = s.substring(start, index);
                    if (IGNORE_BOGUS_GROUP_NAME && (gname.equalsIgnoreCase("mailto") || gname.equalsIgnoreCase("From") || gname.equalsIgnoreCase("To") || gname.equalsIgnoreCase("Cc") || gname.equalsIgnoreCase("Subject") || gname.equalsIgnoreCase("Re"))) {
                        start = -1; // we're not really in a group
                    } else {
                        in_group = true;
                    }
                } else {
                    in_group = true;
                }
                break;

            // Ignore whitespace
            case ' ':
            case '\t':
            case '\r':
            case '\n':
                break;

            default:
                if (start == -1) {
                    start = index;
                }
                break;
            }
        }

        if (start >= 0) {
            /*
             * The last token, add this to our InternetAddress vector. Note that this block of code should be identical to the block above
             * for "case ','".
             */
            if (end == -1) {
                end = length;
            }

            String addr = s.substring(start, end).trim();
            while (addr.length() > 2 && addr.charAt(0) == '<' && addr.charAt(addr.length() - 1) == '>') {
                addr = addr.substring(1, addr.length() - 1);
            }
            if (addr.indexOf(':') > 0 && asciiLowerCase(addr).startsWith("mailto:")) {
                addr = addr.substring(7).trim();
            }
            String pers = null;
            if (rfc822 && start_personal >= 0) {
                pers = unquote(s.substring(start_personal, end_personal).trim());
                if (isEmpty(pers)) {
                    pers = null;
                }
            }

            /*
             * If the personal name field has an "@" and the address field does not, assume they were reversed, e.g., ``"joe doe"
             * (john.doe@example.com)''.
             */
            if (parseHdr && !strict && pers != null && pers.indexOf('@') >= 0 && addr.indexOf('@') < 0 && addr.indexOf('!') < 0) {
                final String tmp = addr;
                addr = pers;
                pers = tmp;
            }
            if (rfc822 || strict || parseHdr) {
                String ace = toACE(init(MimeMessageUtility.decodeMultiEncodedHeader(addr)));
                if (!ignoreErrors) {
                    checkAddress(ace, route_addr, false);
                }
                qia = new QuotedInternetAddress();
                qia.setAddress(ace);
                if (pers != null) {
                    qia.encodedPersonal = pers;
                }
                list.add(qia);
            } else {
                // maybe we passed over more than one space-separated addr
                final StringTokenizer st = new StringTokenizer(addr);
                while (st.hasMoreTokens()) {
                    final String a = st.nextToken();
                    final String ace = toACE(a);
                    checkAddress(ace, false, false);
                    qia = new QuotedInternetAddress();
                    qia.setAddress(ace);
                    list.add(qia);
                }
            }
        }

        // Inject CFWS personal names
        InternetAddress[] parsedAddresses = list.toArray(new InternetAddress[list.size()]);
        list = null;
        if ((null != emailAddressParserAddresses) && (parsedAddresses.length == emailAddressParserAddresses.length)) {
            String defaultMimeCharset = null;
            for (int i = parsedAddresses.length; i-- > 0;) {
                InternetAddress parsedAddress = parsedAddresses[i];
                if (null == parsedAddress.getPersonal()) {
                    String possiblePersonal = emailAddressParserAddresses[i].getPersonal();
                    if (null != possiblePersonal) {
                        try {
                            possiblePersonal = dropComments(possiblePersonal, true).trim();
                            if (possiblePersonal.indexOf("=?") >= 0) {
                                possiblePersonal = MimeUtility.decodeText(possiblePersonal);
                            }
                            if (null == defaultMimeCharset) {
                                defaultMimeCharset = MailProperties.getInstance().getDefaultMimeCharset();
                            }
                            parsedAddress.setPersonal(possiblePersonal, defaultMimeCharset);
                        } catch (UnsupportedEncodingException e) {
                            // Ignore
                            LOG.debug("Unsupported encoding", e);
                        }
                    }
                }
            }
        }

        return parsedAddresses;
    }

    /**
     * Check that the address is a valid "mailbox" per RFC822. (We also allow simple names.) XXX - much more to check XXX - doesn't handle
     * domain-literals properly (but no one uses them)
     */
    private static void checkAddress(String addr, boolean routeAddr, boolean validate) throws AddressException {
        int i, start = 0;

        final int len = addr.length();
        if (len == 0) {
            throw new AddressException("Empty address", addr.toString());
        }

        /*
         * routeAddr indicates that the address is allowed to have an RFC 822 "route".
         */
        if (routeAddr && addr.charAt(0) == '@') {
            /*
             * Check for a legal "route-addr": [@domain[,@domain ...]:]local@domain
             */
            for (start = 0; (i = indexOfAny(addr, ",:", start)) >= 0; start = i + 1) {
                if (addr.charAt(start) != '@') {
                    throw new AddressException("Illegal route-addr", addr.toString());
                }
                if (addr.charAt(i) == ':') {
                    // end of route-addr
                    start = i + 1;
                    break;
                }
            }
        }

        /*
         * The rest should be "local@domain", but we allow simply "local" unless called from validate. local-part must follow RFC 822 - no
         * specials except '.' unless quoted.
         */

        char c = (char) -1;
        char lastc = (char) -1;
        boolean inquote = false;
        for (i = start; i < len; i++) {
            lastc = c;
            c = addr.charAt(i);
            // a quoted-pair is only supposed to occur inside a quoted string,
            // but some people use it outside so we're more lenient
            if (c == '\\' || lastc == '\\') {
                continue;
            }
            if (c == '"') {
                if (inquote) {
                    // peek ahead, next char must be "@"
                    if (validate && i + 1 < len && addr.charAt(i + 1) != '@') {
                        throw new AddressException("Quote not at end of local address", addr.toString());
                    }
                    inquote = false;
                } else {
                    if (validate && i != 0) {
                        throw new AddressException("Quote not at start of local address", addr.toString());
                    }
                    inquote = true;
                }
                continue;
            } else if (c == '\r') {
                // peek ahead, next char must be LF
                if (i + 1 < len && addr.charAt(i + 1) != '\n') {
                    throw new AddressException("Quoted local address contains CR without LF", addr);
                }
            } else if (c == '\n') {
                /*
                 * CRLF followed by whitespace is allowed in a quoted string.
                 * We allowed naked LF, but ensure LF is always followed by
                 * whitespace to prevent spoofing the end of the header.
                 */
                if (i + 1 < len && addr.charAt(i + 1) != ' ' && addr.charAt(i + 1) != '\t') {
                    throw new AddressException("Quoted local address contains newline without whitespace", addr);
                }
            }
            if (inquote) {
                continue;
            }
            // dot rules should not be applied to quoted-string
            if (c == '.') {
                if (i == start) {
                    throw new AddressException("Local address starts with dot", addr);
                }
                if (lastc == '.') {
                    throw new AddressException("Local address contains dot-dot", addr);
                }
            }
            if (c == '@') {
                if (i == 0) {
                    throw new AddressException("Missing local name", addr.toString());
                }
                break; // done with local part
            }
            if (c <= 32 || c >= 127) {
                throw new AddressException("Local address contains control/whitespace or non-ascii character", addr.toString());
            }
            if (SPECIALS_NO_DOT.indexOf(c) >= 0) {
                throw new AddressException("Local address contains illegal character", addr.toString());
            }
        }
        if (inquote) {
            throw new AddressException("Unterminated quote", addr.toString());
        }

        /*
         * Done with local part, now check domain. Note that the MimeMessage class doesn't remember addresses as separate objects; it writes
         * them out as headers and then parses the headers when the addresses are requested. In order to support the case where a "simple"
         * address is used, but the address also has a personal name and thus looks like it should be a valid RFC822 address when parsed, we
         * only check this if we're explicitly called from the validate method.
         */

        if (c != '@') {
            if (validate && !MsisdnCheck.checkMsisdn(addr)) {
                throw new AddressException("Missing final '@domain'", addr.toString());
            }
            return;
        }

        // check for illegal chars in the domain, but ignore domain literals

        start = i + 1;
        if (start >= len) {
            throw new AddressException("Missing domain", addr.toString());
        }

        if (addr.charAt(start) == '.') {
            throw new AddressException("Domain starts with dot", addr.toString());
        }
        boolean inliteral = false;
        for (i = start; i < len; i++) {
            c = addr.charAt(i);
            if (c == '[') {
                if (i != start) {
                    throw new AddressException("Domain literal not at start of domain", addr);
                }
                inliteral = true;   // domain literal, don't validate
            } else if (c == ']') {
                if (i != len - 1) {
                    throw new AddressException("Domain literal end not at end of domain", addr);
                }
                inliteral = false;
            } else if (c <= 040 || c == 0177) {
                throw new AddressException("Domain contains control or whitespace", addr);
            } else {
                // RFC 2822 rule
                //if (specialsNoDot.indexOf(c) >= 0)
                /*
                 * RFC 1034 rule is more strict
                 * the full rule is:
                 *
                 * <domain> ::= <subdomain> | " "
                 * <subdomain> ::= <label> | <subdomain> "." <label>
                 * <label> ::= <letter> [ [ <ldh-str> ] <let-dig> ]
                 * <ldh-str> ::= <let-dig-hyp> | <let-dig-hyp> <ldh-str>
                 * <let-dig-hyp> ::= <let-dig> | "-"
                 * <let-dig> ::= <letter> | <digit>
                 */
                if (!inliteral) {
                    if (!(Character.isLetterOrDigit(c) || c == '-' || c == '.')) {
                        throw new AddressException("Domain contains illegal character", addr);
                    }
                    if (c == '.' && lastc == '.') {
                        throw new AddressException("Domain contains dot-dot", addr);
                    }
                }
            }
            lastc = c;
        }
        if (lastc == '.') {
            throw new AddressException("Domain ends with dot", addr.toString());
        }
    }

    /**
     * Converts a unicode representation of an internet address to ASCII using the procedure in RFC3490 section 4.1. Unassigned characters
     * are not allowed and STD3 ASCII rules are enforced.
     * <p>
     * This implementation already supports EsZett character. Thanks to <a
     * href="http://blog.http.net/code/gnu-libidn-eszett-hotfix/">http.net</a>!
     * <p>
     * <code>"someone@m&uuml;ller.de"</code> is converted to <code>"someone@xn--mller-kva.de"</code>
     *
     * @param idnAddress The unicode representation of an internet address
     * @return The ASCII-encoded (punycode) of given internet address
     * @throws AddressException If ASCII representation of given internet address cannot be created
     */
    public static String toACE(String idnAddress) throws AddressException {
        return IDNA.toACE(idnAddress);
    }

    /**
     * Converts an ASCII-encoded address to its unicode representation. Unassigned characters are not allowed and STD3 hostnames are
     * enforced.
     * <p>
     * This implementation already supports EsZett character. Thanks to <a
     * href="http://blog.http.net/code/gnu-libidn-eszett-hotfix/">http.net</a>!
     * <p>
     * <code>"someone@xn--mller-kva.de"</code> is converted to <code>"someone@m&uuml;ller.de"</code>
     *
     * @param aceAddress The ASCII-encoded (punycode) address
     * @return The unicode representation of given internet address
     * @see #getIDNAddress()
     */
    public static String toIDN(String aceAddress) {
        return IDNA.toIDN(aceAddress);
    }

    private String jcharset;

    /**
     * Initializes a new {@link QuotedInternetAddress}.
     */
    public QuotedInternetAddress() {
        super();
        jcharset = MailProperties.getInstance().getDefaultMimeCharset();
    }

    /**
     * Copy constructor.
     */
    private QuotedInternetAddress(InternetAddress src) throws AddressException {
        this();
        address = toACE(src.getAddress());
        try {
            setPersonal(getPersonal(), null);
        } catch (UnsupportedEncodingException e) {
            // Cannot occur
            throw new IllegalStateException("Unsupported default charset.");
        }
    }

    /**
     * Initializes a new {@link QuotedInternetAddress}.
     * <p>
     * Parse the given string and create an InternetAddress. See the parse method for details of the parsing. The address is parsed using
     * "strict" parsing. This constructor does not perform the additional syntax checks that the InternetAddress(String address, boolean
     * strict) constructor does when strict is true. This constructor is equivalent to InternetAddress(address, false).
     *
     * @param address The address in RFC822 format
     * @throws AddressException If parsing the address fails
     */
    public QuotedInternetAddress(String address) throws AddressException {
        super();
        parseAddress0(address);
        jcharset = MailProperties.getInstance().getDefaultMimeCharset();
    }

    /**
     * Initializes a new {@link QuotedInternetAddress}.
     * <p>
     * Parse the given string and create an InternetAddress. If strict is <code>false</code>, the detailed syntax of the address isn't
     * checked. toACE
     *
     * @param address The address in RFC822 format
     * @param strict <code>true</code> enforce RFC822 syntax; otherwise <code>false</code>
     * @throws AddressException If parsing the address fails
     */
    public QuotedInternetAddress(String address, boolean strict) throws AddressException {
        this(address);
        if (strict) {
            if (isGroup()) {
                getGroup(true); // throw away the result
            } else {
                checkAddress(this.address, true, true);
            }
        }
    }

    private static final Pattern WHITESPACE_OR_CONTROL = Pattern.compile("[\\p{Space}&&[^ ]]|\\p{Cntrl}|[^\\p{Print}\\p{L}]");
    private static final Pattern WHITESPACE_OR_CONTROL_WITHOUT_LETTERS = Pattern.compile("[\\p{Space}&&[^ ]]|\\p{Cntrl}");

    /**
     * Initializes specified address by dropping possibly contained white-space and control characters.
     *
     * @param address The address to initialize
     * @return The possibly initialized address string
     */
    public static String init(String address) {
        return init(address, true);
    }

    /**
     * Initializes specified address by dropping possibly contained white-space and control characters.
     *
     * @param address The address to initialize
     * @param withoutLetters <code>true</code> to only drop space (<code>[\t\n\x0B\f\r]</code>) and control (<code>[\x00-\x1F\x7F]</code>) characters; otherwise <code>false</code>
     * @return The possibly initialized address string
     */
    public static String init(String address, boolean withoutLetters) {
        return null == address ? null : (withoutLetters ? WHITESPACE_OR_CONTROL_WITHOUT_LETTERS : WHITESPACE_OR_CONTROL).matcher(address).replaceAll("");
    }

    /**
     * Initializes a new {@link QuotedInternetAddress}.
     * <p>
     * Construct an instance given the address and personal name. The address is assumed to be a syntactically valid RFC822 address.
     *
     * @param address The address in RFC822 format
     * @param personal The personal name
     * @throws AddressException If parsing the address fails
     * @throws UnsupportedEncodingException If encoding is not supported
     */
    public QuotedInternetAddress(String address, String personal) throws AddressException, UnsupportedEncodingException {
        this(address, personal, null);
    }

    /**
     * Initializes a new {@link QuotedInternetAddress}.
     *
     * @param address The address in RFC822 format
     * @param personal The personal name
     * @param charset The MIME charset for the name
     * @throws AddressException If parsing the address fails
     * @throws UnsupportedEncodingException If encoding is not supported
     */
    public QuotedInternetAddress(String address, String personal, String charset) throws AddressException, UnsupportedEncodingException {
        super();
        this.address = toACE(init(address));
        if (charset == null) {
            // use default charset
            jcharset = MailProperties.getInstance().getDefaultMimeCharset();
        } else {
            // MIME charset -> java charset
            String javaCharset = MimeUtility.javaCharset(charset);
            if ("utf8".equalsIgnoreCase(javaCharset)) {
                javaCharset = "UTF-8";
            }
            jcharset = javaCharset;
        }
        setPersonal(personal, charset);
    }

    /**
     * Parses the given string into this {@link QuotedInternetAddress}.
     *
     * @param address The address in RFC822 format
     * @throws AddressException If parsing the address fails
     */
    public void parseAddress(String address) throws AddressException {
        parseAddress0(address);
    }

    /**
     * Internal parse routine.
     *
     * @param address The address in RFC822 format
     * @throws AddressException If parsing the address fails
     */
    private void parseAddress0(String address) throws AddressException {
        if (Strings.isEmpty(address)) {
            throw new AddressException("Empty address");
        }
        InternetAddress[] a;
        try {
            // use our address parsing utility routine to parse the string
            a = parse0(address, true);

            // if we got back anything other than a single address, it's an error
            if (a.length != 1) {
                a = parse(address, true, false, false);
                if (a.length != 1) {
                    throw new AddressException("Illegal address", address);
                }
            }
        } catch (FatalAddressException e) {
            throw e;
        } catch (AddressException e) {
            // use our address parsing utility routine to parse the string
            a = parse(address, true, false, false);

            // if we got back anything other than a single address, it's an error
            if (a.length != 1) {
                throw new AddressException("Illegal address", address);
            }
        }

        /*
         * Now copy the contents of the single address we parsed into the current object, which will be returned from the constructor. XXX -
         * this sure is a round-about way of getting this done.
         */
        final QuotedInternetAddress internetAddress = (QuotedInternetAddress) a[0];
        this.address = internetAddress.address;
        personal = internetAddress.personal;
        encodedPersonal = internetAddress.encodedPersonal;
    }

    /**
     * Gets the email address in its internationalized, unicode form.
     *
     * @return The IDN email address
     * @see #toIDN(String)
     */
    public String getIDNAddress() {
        return toIDN(address);
    }

    @Override
    public void setPersonal(String name, String charset) throws UnsupportedEncodingException {
        String n = init(name, true);
        personal = n;
        if (isEmpty(personal)) {
            personal = null;
        }
        if (n != null) {
            if (charset == null) {
                // use default charset
                jcharset = MailProperties.getInstance().getDefaultMimeCharset();
            } else {
                // MIME charset -> java charset
                String javaCharset = MimeUtility.javaCharset(charset);
                if ("utf8".equalsIgnoreCase(javaCharset)) {
                    javaCharset = "UTF-8";
                }
                jcharset = javaCharset;
            }
            encodedPersonal = MimeUtility.encodeWord(n, charset, null);
        } else {
            encodedPersonal = null;
        }
    }

    /**
     * Sets the email address.
     *
     * @param address The email address
     */
    @Override
    public void setAddress(String address) {
        String a = init(address);
        try {
            this.address = toACE(a);
        } catch (AddressException e) {
            LOG.error("ACE string could not be parsed from IDN string: {}", a, e);
            this.address = a;
        }
    }

    /**
     * Gets the email address in Unicode characters.
     *
     * @return The email address in Unicode characters
     */
    public String getUnicodeAddress() {
        return toIDN(address);
    }

    /**
     * Get the personal name. If the name is encoded as per RFC 2047, it is decoded and converted into Unicode. If the decoding or
     * conversion fails, the raw data is returned as is.
     *
     * @return personal name
     */
    @Override
    public String getPersonal() {
        if (personal != null) {
            return personal;
        }

        if (encodedPersonal != null) {
            try {
                personal = init(MimeMessageUtility.decodeMultiEncodedHeader(encodedPersonal), true);
                if (isEmpty(personal)) {
                    encodedPersonal = null;
                    personal = null;
                }
                return personal;
            } catch (Exception ex) {
                // 1. ParseException: either its an unencoded string or
                // it can't be parsed
                // 2. UnsupportedEncodingException: can't decode it.
                return encodedPersonal;
            }
        }
        // No personal or encodedPersonal, return null
        return null;
    }

    /**
     * Convert this address into a RFC 822 / RFC 2047 encoded address. The resulting string contains only US-ASCII characters, and hence is
     * mail-safe.
     *
     * @return possibly encoded address string
     */
    @Override
    public String toString() {
        if (encodedPersonal == null && personal != null) {
            try {
                encodedPersonal = MimeUtility.encodeWord(personal, jcharset, null);
            } catch (UnsupportedEncodingException ex) {
                LOG.error("", ex);
            }
        }

        if (encodedPersonal != null) {
            if (encodedPersonal.length() > 0) {
                if (null == personal) {
                    try {
                        personal = init(MimeMessageUtility.decodeMultiEncodedHeader(encodedPersonal), true);
                    } catch (Exception ex) {
                        // 1. ParseException: either its an unencoded string or
                        // it can't be parsed
                        // 2. UnsupportedEncodingException: can't decode it.
                        personal = encodedPersonal;
                    }
                }

                if (quoted(personal)) {
                    if (checkQuotedPersonal(personal)) {
                        return new StringBuilder(32).append(encodedPersonal).append(" <").append(address).append('>').toString();
                    }
                    personal = personal.substring(1, personal.length() - 1);
                    try {
                        encodedPersonal = MimeUtility.encodeWord(personal, jcharset, null);
                    } catch (UnsupportedEncodingException ex) {
                        LOG.error("", ex);
                    }
                }

                if (keepQuotesInEncodedPersonal()) {
                    // A workaround to fix bug 14050 (fat clients interpreting a comma as address separator), which became obsolete in the meantime
                    if (needQuoting(personal, true)) {
                        // Personal phrase needs to be quoted
                        try {
                            encodedPersonal = MimeUtility.encodeWord(quotePhrase(personal, true), jcharset, null);
                        } catch (UnsupportedEncodingException e) {
                            LOG.error("", e);
                        }
                    } else if (!isAscii(personal)) {
                        try {
                            encodedPersonal = MimeUtility.encodeWord(quotePhrase(personal, true), jcharset, null);
                        } catch (UnsupportedEncodingException e) {
                            LOG.error("", e);
                        }
                    }
                    return new StringBuilder(32).append(encodedPersonal).append(" <").append(address).append('>').toString();
                }

                if (!isAscii(personal)) {
                    try {
                        encodedPersonal = MimeUtility.encodeWord(personal, jcharset, null);
                    } catch (UnsupportedEncodingException e) {
                        LOG.error("", e);
                    }
                }

                if (isEmpty(personal)) {
                    encodedPersonal = null;
                    personal = null;
                }

                return encodedPersonal == null ? address : new StringBuilder(32).append(quotePhrase(encodedPersonal, false)).append(" <").append(address).append('>').toString();
            } else if (toUpperCase(address).endsWith("/TYPE=PLMN")) {
                return new StringBuilder().append('<').append(address).append('>').toString();
            } else if (isGroup() || isSimple()) {
                return address;
            } else {
                return new StringBuilder().append('<').append(address).append('>').toString();
            }
        } else if (isGroup() || isSimple()) {
            return address;
        } else {
            return new StringBuilder().append('<').append(address).append('>').toString();
        }
    }

    /**
     * Returns a properly formatted address (RFC 822 syntax) of Unicode characters.
     *
     * @return The Unicode address string
     */
    @Override
    public String toUnicodeString() {
        final String p = getPersonal();
        if (p != null) {
            if (p.length() > 0) {
                if (quoted(p)) {
                    return new StringBuilder(32).append(p).append(" <").append(toIDN(address)).append('>').toString();
                }
                return new StringBuilder(32).append(quotePhrase(p, true)).append(" <").append(toIDN(address)).append('>').toString();
            } else if (toUpperCase(address).endsWith("/TYPE=PLMN")) {
                return new StringBuilder().append('<').append(address).append('>').toString();
            } else if (isGroup() || isSimple()) {
                return toIDN(address);
            } else {
                return new StringBuilder(32).append('<').append(toIDN(address)).append('>').toString();
            }
        } else if (isGroup() || isSimple()) {
            return toIDN(address);
        } else {
            return new StringBuilder(32).append('<').append(toIDN(address)).append('>').toString();
        }
    }

    // @Override
    // public boolean equals(Object a) {
    // if (this == a) {
    // return true;
    // }
    // if (!(a instanceof InternetAddress)) {
    // return false;
    // }
    // final String s = ((InternetAddress) a).getAddress();
    // if (address == null) {
    // if (s != null) {
    // return false;
    // }
    // } else if (!address.equalsIgnoreCase(s) && !toIDN(address).equalsIgnoreCase(s)) {
    // return false;
    // }
    // return true;
    // }

    /**
     * Is this a "simple" address? Simple addresses don't contain quotes or any RFC822 special characters other than '@' and '.'.
     */
    private boolean isSimple() {
        return null == address || indexOfAny(address, SPECIALS_NO_DOT_NO_AT_NO_QUOTE) < 0;
    }

    /**
     * Return the first index of any of the characters in "any" in "s", or -1 if none are found. This should be a method on String.
     */
    private static int indexOfAny(CharSequence s, String any) {
        return indexOfAny(s, any, 0);
    }

    private static int indexOfAny(CharSequence s, String any, int start) {
        try {
            final int len = s.length();
            for (int i = start; i < len; i++) {
                if (any.indexOf(s.charAt(i)) >= 0) {
                    return i;
                }
            }
            return -1;
        } catch (StringIndexOutOfBoundsException e) {
            return -1;
        }
    }

    private static final String SPECIALS_NO_DOT_NO_AT_NO_QUOTE = "()<>,;:\\[]";

    private static final String SPECIALS_NO_DOT = "()<>@,;:\\\"[]";

    private final static String RFC822 = "()<>@,;:\\\".[]";

    private static String quotePhrase(String phrase, boolean allowNonAscii) {
        int len = phrase.length();
        boolean needQuoting = false;

        for (int i = 0; i < len; i++) {
            char c = phrase.charAt(i);
            if (c == '"' || c == '\\') {
                // Need to escape that character and then quote the whole string
                StringBuilder sb = new StringBuilder(len + 8);
                sb.append('"');
                if (i > 0) {
                    sb.append(phrase, 0, i);
                }
                sb.append('\\').append(c);

                // Check remainder, too
                for (int j = i + 1; j < len; j++) {
                    c = phrase.charAt(j);
                    if (c == '"' || c == '\\') {
                        // Escape the character
                        sb.append('\\');
                    }
                    sb.append(c);
                }
                sb.append('"');
                return sb.toString();
            } else if ((c < 32 && c != '\r' && c != '\n' && c != '\t') || (!allowNonAscii && c >= 127) || RFC822.indexOf(c) >= 0) {
                // These characters cause the string to be quoted
                needQuoting = true;
            }
        }

        return needQuoting ? new StringBuilder(len + 2).append('"').append(phrase).append('"').toString() : phrase;
    }

    private static boolean needQuoting(String phrase, boolean allowNonAscii) {
        final int len = phrase.length();
        boolean needQuoting = false;

        for (int i = 0; !needQuoting && i < len; i++) {
            final char c = phrase.charAt(i);
            if (c == '"' || c == '\\') {
                // need to escape them and then quote the whole string
                needQuoting = true;
            } else if ((c < 32 && c != '\r' && c != '\n' && c != '\t') || (!allowNonAscii && c >= 127) || RFC822.indexOf(c) >= 0) {
                // These characters cause the string to be quoted
                needQuoting = true;
            }
        }
        return needQuoting;
    }

    private static boolean quoted(String s) {
        final int length = s.length();
        if (length <= 0) {
            return false;
        }
        return ('"' == s.charAt(0) && length > 1 && '"' == s.charAt(length - 1));
    }

    private static boolean checkQuotedPersonal(String p) {
        // Every '"' and '\' needs a heading '\' character
        final String phrase = p.substring(1, p.length() - 1);
        final int len = phrase.length();
        boolean valid = true;

        int i = 0;
        while (valid && i < len) {
            final char c = phrase.charAt(i);
            if (c == '"') {
                valid = i > 1 && '\\' == phrase.charAt(i - 1);
                i++;
            } else if (c == '\\') {
                final int ni = i + 1;
                final char c2 = ni < len ? phrase.charAt(ni) : '\0';
                valid = (c2 == '"' || c2 == '\\');
                i += 2;
            } else {
                i++;
            }
        }

        return valid;
    }

    private static String unquote(String str) {
        if (isEmpty(str)) {
            return str;
        }
        String s = str;
        int length = s.length();
        if (1 == length) {
            return str;
        }
        if ('"' == s.charAt(0) && '"' == s.charAt(length - 1)) {
            s = s.substring(1, length - 1);
            // check for any escaped characters
            if (s.indexOf('\\') >= 0) {
                length = length - 2;
                final StringBuilder sb = new StringBuilder(length); // approx
                for (int i = 0; i < length; i++) {
                    char c = s.charAt(i);
                    if (c == '\\' && i < length - 1) {
                        c = s.charAt(++i);
                    }
                    sb.append(c);
                }
                s = sb.toString();
            }
        }
        return s;
    }

    /**
     * Determines whether a String is purely ASCII, meaning its characters' code points are all less than 128.
     */
    private static boolean isAscii(String str) {
        if (null == str || 0 == str.length()) {
            return true;
        }
        final int len = str.length();
        boolean ret = true;
        for (int i = 0; ret && i < len; i++) {
            final char c = str.charAt(i);
            ret = (c >= 32 && c <= 127);
        }
        return ret;
    }

    /** Simple tuple for character sequence and info instance */
    private static class CharSequenceAndInfo {

        /** The character sequence to pass to <code>org.hazlewood.connor.bottema.emailaddress.EmailAddressParser</code> */
        final CharSequence toParse;

        /** The watched info instance or <code>null</code> */
        final EmailAddressInfo info;

        /**
         * Initializes a new {@link CharSequenceAndInfo}.
         *
         * @param toParse The character sequence to parse
         * @param info The watched info instance or <code>null</code>
         */
        CharSequenceAndInfo(CharSequence toParse, EmailAddressInfo info) {
            super();
            this.toParse = toParse;
            this.info = info;
        }
    }

}
