/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */


package com.openexchange.mail.api;

import static com.openexchange.java.Autoboxing.I;
import java.util.Map;

/**
 * {@link AuthenticationFailedStateUtility} - Utility class for authentication failed state.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v8.x
 */
public final class AuthenticationFailedStateUtility {

    /**
     * Initializes a new instance of {@link AuthenticationFailedStateUtility}.
     */
    private AuthenticationFailedStateUtility() {
        super();
    }

    /**
     * Gets the invocation count from given authentication failed state.
     *
     * @param state The authentication failed state to read from
     * @return The invocation count as a positive number or <code>0</code> if absent
     */
    public static int getInvocationCount(Map<String, Object> state) {
        Integer invocationCount = state == null ? null : AuthenticationFailedStateProperty.INVOCATION_COUNT.getValueFromState(state);
        return invocationCount == null ? 0 : invocationCount.intValue();
    }

    /**
     * Sets the invocation count in given authentication failed state.
     * <p>
     * Does nothing at all if <code>state</code> is <code>null</code>.
     *
     * @param invocationCount The invocation count to set; needs to be greater than <code>0</code> (zero)
     * @param state The authentication failed state to set in
     * @throws IllegalArgumentException If invocation count is equal to or less than <code>0</code> (zero)
     */
    public static void setInvocationCount(int invocationCount, Map<String, Object> state) {
        if (state == null) {
            // Nothing to do
            return;
        }

        if (invocationCount <= 0) {
            throw new IllegalArgumentException("Illegal invocation count: " + invocationCount);
        }

        try {
            state.put(AuthenticationFailedStateProperty.INVOCATION_COUNT.getName(), I(invocationCount));
        } catch (UnsupportedOperationException e) {
            throw new IllegalArgumentException("Illegal state. State must not be unmodifiable", e);
        }
    }

}
