/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */


package com.openexchange.mail.api;

import java.util.Map;

/**
 * {@link AuthenticationFailedStateProperty} - Known properties of the state (<code>Map&lt;String, Object&gt;</code>) passed to
 * authentication failed handler instances.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v8.x
 */
public enum AuthenticationFailedStateProperty {

    /**
     * The property for the invocation count.
     * <p>
     * The value is expected to be of type <code>java.lang.Integer</code>.
     */
    INVOCATION_COUNT("com.openexchange.mail.authentication.failed.invocationCount"),
    ;

    private final String name;

    /**
     * Initializes a new instance of {@link AuthenticationFailedStateProperty}.
     *
     * @param name The property name
     */
    private AuthenticationFailedStateProperty(String name) {
        this.name = name;
    }

    /**
     * Gets the name of the known property.
     *
     * @return The property name
     */
    public String getName() {
        return name;
    }

    /**
     * Gets this property's value from given state.
     *
     * @param <V> The type of the property's value
     * @param state The authentication failed state to read from
     * @return The property value or <code>null</code> if absent or of unexpected type
     */
    public <V> V getValueFromState(Map<String, Object> state) {
        Object value = state == null ? null : state.get(name);
        if (value == null) {
            return null;
        }

        try {
            return (V) value;
        } catch (ClassCastException e) {
            return null;
        }
    }

}
