/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.groupware.infostore.webdav;

import static com.openexchange.tools.sql.DBUtils.getStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import com.openexchange.database.provider.DBProvider;
import com.openexchange.database.tx.DBService;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.contexts.Context;
import com.openexchange.groupware.impl.IDGenerator;
import com.openexchange.groupware.infostore.InfostoreExceptionCodes;
import com.openexchange.user.User;

public abstract class AbstractLockManager<T extends Lock> extends DBService implements LockManager{

    private static final String PAT_TABLENAME = "%%tablename%%";

    private static final long MILLIS_WEEK = 604800000L;
    private static final long MILLIS_YEAR = 52 * MILLIS_WEEK;
    private static final long MILLIS_10_YEARS = 10 * MILLIS_YEAR;

    private String insertStatement = "INSERT INTO %%tablename%% (entity, timeout, scope, type, ownerDesc, cid, userid, id %%additional_fields%% ) VALUES (?, ?, ?, ?, ?, ?, ?, ? %%additional_question_marks%%)";
    private String deleteStatement = "DELETE FROM %%tablename%% WHERE cid = ? AND id = ? ";
    private String reassignStatement = "UPDATE %%tablename%% SET userid = ? WHERE userid = ? and cid = ?";
    private String findByEntityStatement = "SELECT entity, timeout, scope, type, ownerDesc, cid, userid, id %%additional_fields%% FROM %%tablename%% WHERE entity IN %%entity_ids%% and cid = ? ";
    private String findBySingleEntityStatement = "SELECT entity, timeout, scope, type, ownerDesc, cid, userid, id %%additional_fields%% FROM %%tablename%% WHERE entity = ? and cid = ? ";
    private String existsByEntityStatement = "SELECT 1 FROM %%tablename%% WHERE entity IN %%entity_ids%% and cid = ? ";
    private String existsBySingleEntityStatement = "SELECT 1 FROM %%tablename%% WHERE entity = ? and cid = ? ";
    private String deleteByEntityStatement = "DELETE FROM %%tablename%% WHERE cid = ? AND entity = ?";
    private String updateByIdStatement = "UPDATE %%tablename%% SET timeout = ? , scope = ?, type = ? , ownerDesc = ? %%additional_updates%% WHERE id = ? AND cid = ?";

    private final List<LockExpiryListener> expiryListeners = new ArrayList<LockExpiryListener>();

    protected AbstractLockManager(String tablename) {
        this(null, tablename);
    }

    protected AbstractLockManager(DBProvider provider, String tablename) {
        super();
        setProvider(provider);
        initTablename(tablename);
    }

    private void initTablename(final String tablename) {
        insertStatement = insertStatement.replace(PAT_TABLENAME, tablename);
        insertStatement = initAdditionalINSERT(insertStatement);
        deleteStatement = deleteStatement.replace(PAT_TABLENAME, tablename);

        findByEntityStatement = findByEntityStatement.replace(PAT_TABLENAME, tablename);
        findByEntityStatement = initAdditionalFIND_BY_ENTITY(findByEntityStatement);
        findBySingleEntityStatement = findBySingleEntityStatement.replace(PAT_TABLENAME, tablename);
        findBySingleEntityStatement = initAdditionalFIND_BY_ENTITY(findBySingleEntityStatement);

        existsByEntityStatement = existsByEntityStatement.replace(PAT_TABLENAME, tablename);
        existsBySingleEntityStatement = existsBySingleEntityStatement.replace(PAT_TABLENAME, tablename);

        deleteByEntityStatement = deleteByEntityStatement.replace(PAT_TABLENAME, tablename);

        updateByIdStatement = updateByIdStatement.replace(PAT_TABLENAME, tablename);
        updateByIdStatement = initAdditionalUPDATE_BY_ID(updateByIdStatement);

        reassignStatement = reassignStatement.replace(PAT_TABLENAME, tablename);
    }


    private static String initAdditionalUPDATE_BY_ID(String query) {
        return query.replace("%%additional_updates%%","");
    }

    protected String initAdditionalINSERT(String insert) {
        return insert.replace("%%additional_fields%%","").replace("%%additional_question_marks%%", "");
    }

    protected String initAdditionalFIND_BY_ENTITY(String findByEntity) {
        return findByEntity.replace("%%additional_fields%%", "");
    }

    protected int getType() {
        return com.openexchange.groupware.Types.WEBDAV; // FIXME
    }

    protected abstract T newLock();

    protected void fillLock(T lock, ResultSet rs) throws SQLException {
        lock.setId(rs.getInt("id"));
        lock.setOwner(rs.getInt("userid"));
        final int scopeNum = rs.getInt("scope");
        for(final Scope scope : Scope.values()) {
            if (scopeNum == scope.ordinal()) {
                lock.setScope(scope);
            }
        }
        lock.setType(Type.WRITE);
        final long timeout =  (rs.getLong("timeout") - System.currentTimeMillis());
        lock.setTimeout(timeout);
        lock.setOwnerDescription(rs.getString("ownerDesc"));
        lock.setEntity(rs.getInt("entity"));
    }

    protected int createLockForceId(int entity, int id, long timeout, Scope scope, Type type, String ownerDesc, Context ctx, User user, Object...additional) throws OXException {
        Connection con = null;
        PreparedStatement stmt = null;
        try {
            con = getWriteConnection(ctx);
            stmt = con.prepareStatement(insertStatement);
            long tm = 0;
            if (timeout == INFINITE) {
                tm = System.currentTimeMillis() + MILLIS_10_YEARS;
            } else {
                tm = System.currentTimeMillis() + timeout;
                // Previously Infinite Locks exceed long range if ms counter increased by 1 since loading
                if (tm < 0) {
                    tm = System.currentTimeMillis() + MILLIS_10_YEARS;
                }
            }
            set(1, stmt, additional, Integer.valueOf(entity), Long.valueOf(tm), Integer.valueOf(scope.ordinal()), Integer.valueOf(type.ordinal()), ownerDesc, Integer.valueOf(ctx.getContextId()), Integer.valueOf(user.getId()), Integer.valueOf(id));
            stmt.executeUpdate();
            return id;
        } catch (SQLException x) {
            throw InfostoreExceptionCodes.SQL_PROBLEM.create(x, getStatement(stmt));
        } catch (OXException e) {
            throw e;
        } finally {
            close(stmt, null);
            releaseWriteConnection(ctx, con);
        }
    }

    protected int createLock(int entity, long timeout, Scope scope, Type type, String ownerDesc, Context ctx, User user, Object...additional) throws OXException {
        try {
            return createLockForceId(entity, IDGenerator.getId(ctx, getType()), timeout, scope, type, ownerDesc, ctx, user, additional);
        } catch (SQLException e) {
            throw InfostoreExceptionCodes.NEW_ID_FAILED.create(e);
        }
    }

    protected void updateLock(int lockId, long timeout, Scope scope, Type type, String ownerDesc, Context ctx, Object...additional) throws OXException {
        Connection con = null;
        PreparedStatement stmt = null;
        try {
            con = getWriteConnection(ctx);
            stmt = con.prepareStatement(updateByIdStatement);
            long tm = 0;
            if (timeout == INFINITE) {
                tm = System.currentTimeMillis() + MILLIS_10_YEARS;
            } else {
                tm = System.currentTimeMillis() + timeout;
            }
            final int index = set(1, stmt, additional, Long.valueOf(tm), Integer.valueOf(scope.ordinal()), Integer.valueOf(type.ordinal()), ownerDesc);
            set(index, stmt, null, Integer.valueOf(lockId), Integer.valueOf(ctx.getContextId()));
            stmt.executeUpdate();
        } catch (SQLException x) {
            throw InfostoreExceptionCodes.SQL_PROBLEM.create(x, getStatement(stmt));
        } catch (OXException e) {
            throw e;
        } finally {
            close(stmt, null);
            releaseWriteConnection(ctx, con);
        }
    }

    protected void removeLock(int id, Context ctx) throws OXException {
        Connection con = null;
        PreparedStatement stmt = null;
        try {
            con = getWriteConnection(ctx);
            stmt = con.prepareStatement(deleteStatement);
            set(1, stmt, null, Integer.valueOf(ctx.getContextId()), Integer.valueOf(id));
            stmt.executeUpdate();
        } catch (SQLException x) {
            throw InfostoreExceptionCodes.SQL_PROBLEM.create(x, getStatement(stmt));
        } catch (OXException e) {
            throw e;
        } finally {
            close(stmt, null);
            releaseWriteConnection(ctx, con);
        }
    }

    public Map<Integer,List<T>> findLocksByEntity(List<Integer> entities, Context ctx) throws OXException {
        if (null == entities || entities.isEmpty()) {
            return Collections.emptyMap();
        }

        Connection con = null;
        PreparedStatement stmt = null;
        ResultSet rs = null;
        try {
            con = getReadConnection(ctx);
            int size = entities.size();
            if (1 == size) {
                stmt = con.prepareStatement(findBySingleEntityStatement);
                stmt.setInt(1, entities.get(0).intValue());
                stmt.setInt(2, ctx.getContextId());
            } else {
                StringBuilder entityIds = new StringBuilder().append('(');
                join(entities, entityIds);
                entityIds.append(')');

                stmt = con.prepareStatement(findByEntityStatement.replace("%%entity_ids%%", entityIds.toString()));
                set(1, stmt, null, Integer.valueOf(ctx.getContextId()));
            }

            rs = stmt.executeQuery();

            Map<Integer, List<T>> locks = new HashMap<Integer, List<T>>(size);
            Set<Integer> entitySet = new HashSet<Integer>(entities);
            while (rs.next()) {
                int entity = rs.getInt(1);
                entitySet.remove(Integer.valueOf(entity));
                List<T> lockList = locks.get(Integer.valueOf(entity));
                if (null == lockList) {
                    lockList = new ArrayList<T>();
                    locks.put(Integer.valueOf(entity), lockList);
                }

                T lock = newLock();
                fillLock(lock, rs);
                if (lock.getTimeout() < 1) {
                    removeLock(lock.getId(), ctx);
                    lockExpired(lock);
                } else {
                    lockList.add(lock);
                }
            }

            for (Integer entity : entitySet) {
                locks.put(entity, Collections.<T> emptyList());
            }
            return locks;
        } catch (SQLException x) {
            throw InfostoreExceptionCodes.SQL_PROBLEM.create(x, getStatement(stmt));
        } catch (OXException e) {
            throw e;
        } finally {
            close(stmt, rs);
            releaseReadConnection(ctx, con);
        }
    }

    public boolean existsLockForEntity(List<Integer> entities, Context ctx) throws OXException {
        Connection con = null;
        PreparedStatement stmt = null;
        ResultSet rs = null;
        try {
            con = getReadConnection(ctx);
            if (1 == entities.size()) {
                stmt = con.prepareStatement(existsBySingleEntityStatement);
                stmt.setInt(1, entities.get(0).intValue());
                stmt.setInt(2, ctx.getContextId());
            } else {
                StringBuilder entityIds = new StringBuilder().append('(');
                join(entities, entityIds);
                entityIds.append(')');

                stmt = con.prepareStatement(existsByEntityStatement.replace("%%entity_ids%%", entityIds.toString()));
                set(1, stmt, null, Integer.valueOf(ctx.getContextId()));
            }

            rs = stmt.executeQuery();
            return rs.next();
        } catch (SQLException x) {
            throw InfostoreExceptionCodes.SQL_PROBLEM.create(x, getStatement(stmt));
        } catch (OXException e) {
            throw e;
        } finally {
            close(stmt, rs);
            releaseReadConnection(ctx, con);
        }
    }

    public void reassign(Context ctx, int from, int to) throws OXException {
        Connection writeCon = null;
        PreparedStatement stmt = null;
        try {
            writeCon = getWriteConnection(ctx);
            stmt = writeCon.prepareStatement(reassignStatement);
            stmt.setInt(1, to);
            stmt.setInt(2, from);
            stmt.setInt(3, ctx.getContextId());
            stmt.executeUpdate();
        } catch (SQLException x) {
            throw InfostoreExceptionCodes.SQL_PROBLEM.create(x, getStatement(stmt));
        } catch (OXException e) {
            throw e;
        } finally {
            close(stmt, null);
            releaseWriteConnection(ctx, writeCon);
        }
    }

    protected void removeAllFromEntity(int entity, Context ctx) throws OXException {
        Connection writeCon = null;
        PreparedStatement stmt = null;
        try {
            writeCon = getWriteConnection(ctx);
            stmt = writeCon.prepareStatement(deleteByEntityStatement);
            stmt.setInt(1, ctx.getContextId());
            stmt.setInt(2, entity);
            stmt.executeUpdate();
        } catch (SQLException x) {
            throw InfostoreExceptionCodes.SQL_PROBLEM.create(x, getStatement(stmt));
        } catch (OXException e) {
            throw e;
        } finally {
            close(stmt, null);
            releaseWriteConnection(ctx, writeCon);
        }
    }

    protected static CharSequence join(List<Integer> entities, StringBuilder b) {
        Iterator<Integer> it = entities.iterator();
        if (it.hasNext()) {
            Integer entity = it.next();
            b.append(entity.intValue());

            while (it.hasNext()) {
                entity = it.next();
                b.append(", ");
                b.append(entity.intValue());
            }
        }

        return b;
    }

    protected static int set(int index, PreparedStatement stmt, Object[] additional, Object... values) throws SQLException {
        int idx = index;
        for (Object o : values) {
            stmt.setObject(idx++, o);
        }
        if (null != additional) {
            for (Object o : additional) {
                stmt.setObject(idx++, o);
            }
        }
        return idx;
    }

    public void addExpiryListener(LockExpiryListener listener) {
        expiryListeners.add( listener );
    }

    protected void lockExpired(Lock lock) throws OXException {
        for (LockExpiryListener listener : expiryListeners) {
            listener.lockExpired(lock);
        }
    }

}
