/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.groupware.attach.json.actions;

import org.json.JSONValue;
import com.openexchange.ajax.AJAXServlet;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.annotation.restricted.RestrictedAction;
import com.openexchange.ajax.writer.AttachmentWriter;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.attach.AttachmentMetadata;
import com.openexchange.json.OXJSONWriter;
import com.openexchange.oauth.provider.resourceserver.OAuthAccess;
import com.openexchange.oauth.provider.resourceserver.annotations.OAuthScopeCheck;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.OAuthContentTypes;
import com.openexchange.tools.TimeZoneUtils;
import com.openexchange.tools.servlet.AjaxExceptionCodes;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.user.User;

/**
 * {@link GetAction}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
@RestrictedAction(module = AbstractAttachmentAction.MODULE, type = RestrictedAction.Type.READ, hasCustomOAuthScopeCheck = true)
public final class GetAction extends AbstractAttachmentAction {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(GetAction.class);

    /**
     * Initializes a new {@link GetAction}.
     */
    public GetAction(final ServiceLookup serviceLookup) {
        super(serviceLookup);
    }

    @Override
    public AJAXRequestResult doPerform(final AJAXRequestData requestData, final ServerSession session) throws OXException {
        try {
            require(
                requestData,
                AJAXServlet.PARAMETER_FOLDERID,
                AJAXServlet.PARAMETER_ATTACHEDID,
                AJAXServlet.PARAMETER_MODULE,
                AJAXServlet.PARAMETER_ID);
            final int folderId = requireNumber(requestData, AJAXServlet.PARAMETER_FOLDERID);
            final int attachedId = requireNumber(requestData, AJAXServlet.PARAMETER_ATTACHEDID);
            final int moduleId = requireNumber(requestData, AJAXServlet.PARAMETER_MODULE);
            final int id = requireNumber(requestData, AJAXServlet.PARAMETER_ID);

            return new AJAXRequestResult(get(folderId, attachedId, moduleId, id, session), "apiResponse");
        } catch (RuntimeException e) {
            throw AjaxExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    private JSONValue get(final int folderId, final int attachedId, final int moduleId, final int id, final ServerSession session) throws OXException {
        try {
            ATTACHMENT_BASE.startTransaction();

            final User user = session.getUser();
            final AttachmentMetadata attachment =
                ATTACHMENT_BASE.getAttachment(
                    session, folderId,
                    attachedId,
                    moduleId,
                    id,
                    session.getContext(),
                    user,
                    session.getUserConfiguration());

            final OXJSONWriter w = new OXJSONWriter();
            final AttachmentWriter aWriter = new AttachmentWriter(w);
            aWriter.timedResult(attachment.getCreationDate().getTime());
            aWriter.write(attachment, TimeZoneUtils.getTimeZone(user.getTimeZone()));
            aWriter.endTimedResult();

            ATTACHMENT_BASE.commit();

            return w.getObject();
        } catch (Throwable t) {
            rollback();
            if (t instanceof OXException) {
                throw (OXException) t;
            }
            throw new OXException(t);
        } finally {
            try {
                ATTACHMENT_BASE.finish();
            } catch (OXException e) {
                LOG.error("", e);
            }
        }
    }
    
    @OAuthScopeCheck
    public boolean accessAllowed(AJAXRequestData request, ServerSession session, OAuthAccess access) throws OXException {
        return OAuthContentTypes.mayReadViaOAuthRequest(getContentType(requireNumber(request, AJAXServlet.PARAMETER_MODULE)), access);
    }
}
