/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.groupware.attach.json.actions;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.Attachment;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.annotation.restricted.RestrictedAction;
import com.openexchange.conversion.ConversionService;
import com.openexchange.conversion.Data;
import com.openexchange.conversion.DataArguments;
import com.openexchange.conversion.DataProperties;
import com.openexchange.conversion.DataSource;
import com.openexchange.exception.OXException;
import com.openexchange.folderstorage.ContentType;
import com.openexchange.groupware.attach.AttachmentBase;
import com.openexchange.groupware.attach.AttachmentBatch;
import com.openexchange.groupware.attach.AttachmentConfig;
import com.openexchange.groupware.attach.AttachmentField;
import com.openexchange.groupware.attach.AttachmentMetadata;
import com.openexchange.groupware.attach.AttachmentUtility;
import com.openexchange.groupware.contexts.Context;
import com.openexchange.groupware.upload.UploadFile;
import com.openexchange.groupware.upload.impl.UploadEvent;
import com.openexchange.groupware.userconfiguration.UserConfiguration;
import com.openexchange.oauth.provider.resourceserver.OAuthAccess;
import com.openexchange.oauth.provider.resourceserver.annotations.OAuthScopeCheck;
import com.openexchange.server.ServiceExceptionCode;
import com.openexchange.server.ServiceLookup;
import com.openexchange.server.services.ServerServiceRegistry;
import com.openexchange.tools.OAuthContentTypes;
import com.openexchange.tools.servlet.AjaxExceptionCodes;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.user.User;

/**
 * {@link AttachAction}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
@RestrictedAction(module = AbstractAttachmentAction.MODULE, type = RestrictedAction.Type.WRITE, hasCustomOAuthScopeCheck = true)
public final class AttachAction extends AbstractAttachmentAction {

    private static final String DATASOURCE = "datasource";
    private static final String IDENTIFIER = "identifier";

    /**
     * Initializes a new {@link AttachAction}.
     */
    public AttachAction(final ServiceLookup serviceLookup) {
        super(serviceLookup);
    }

    public static transient final AttachmentField[] REQUIRED = Attachment.REQUIRED;

    @Override
    public AJAXRequestResult doPerform(final AJAXRequestData requestData, final ServerSession session) throws OXException {
        try {
            User user = session.getUser();
            UserConfiguration userConfiguration = session.getUserConfiguration();
            long maxUploadSize = AttachmentConfig.getMaxUploadSize();
            if (requestData.hasUploads(-1, maxUploadSize > 0 ? maxUploadSize : -1L)) {
                AttachmentInfo attachmentInfo = parseAttachments(requestData);
                return attach(attachmentInfo.attachments, attachmentInfo.uploadFiles, session, session.getContext(), user, userConfiguration);
            }
            JSONObject object = getJSONObject(requestData);
            if (object == null) {
                return new AJAXRequestResult(Integer.valueOf(0), new Date(System.currentTimeMillis()), "int");
            }

            final AttachmentMetadata attachment = parseAttachmentMetadata(object);
            final ConversionService conversionService = ServerServiceRegistry.getInstance().getService(ConversionService.class);

            if (conversionService == null) {
                throw ServiceExceptionCode.SERVICE_UNAVAILABLE.create(ConversionService.class.getName());
            }

            final JSONObject datasourceDef = object.getJSONObject(DATASOURCE);
            final String datasourceIdentifier = datasourceDef.getString(IDENTIFIER);

            final DataSource source = conversionService.getDataSource(datasourceIdentifier);
            if (source == null) {
                throw AjaxExceptionCodes.INVALID_PARAMETER_VALUE.create("datasource", datasourceIdentifier);
            }

            final List<Class<?>> types = Arrays.asList(source.getTypes());

            final Map<String, String> arguments = new HashMap<String, String>();

            for (final String key : datasourceDef.keySet()) {
                arguments.put(key, datasourceDef.getString(key));
            }

            InputStream is;
            if (types.contains(InputStream.class)) {
                final Data<InputStream> data = source.getData(InputStream.class, new DataArguments(arguments), session);
                final String sizeS = data.getDataProperties().get(DataProperties.PROPERTY_SIZE);
                final String contentTypeS = data.getDataProperties().get(DataProperties.PROPERTY_CONTENT_TYPE);

                if (sizeS != null) {
                    attachment.setFilesize(Long.parseLong(sizeS));
                }

                if (contentTypeS != null) {
                    attachment.setFileMIMEType(contentTypeS);
                }

                final String name = data.getDataProperties().get(DataProperties.PROPERTY_NAME);
                if (name != null && null == attachment.getFilename()) {
                    attachment.setFilename(name);
                }

                is = data.getData();

            } else if (types.contains(byte[].class)) {
                final Data<byte[]> data = source.getData(byte[].class, new DataArguments(arguments), session);
                final byte[] bytes = data.getData();
                is = new ByteArrayInputStream(bytes);
                attachment.setFilesize(bytes.length);

                final String contentTypeS = data.getDataProperties().get(DataProperties.PROPERTY_CONTENT_TYPE);
                if (contentTypeS != null) {
                    attachment.setFileMIMEType(contentTypeS);
                }

                final String name = data.getDataProperties().get(DataProperties.PROPERTY_NAME);
                if (name != null && null == attachment.getFilename()) {
                    attachment.setFilename(name);
                }

            } else {
                throw AjaxExceptionCodes.INVALID_PARAMETER_VALUE.create("datasource", datasourceIdentifier);
            }

            if (attachment.getFilename() == null) {
                attachment.setFilename("unknown" + System.currentTimeMillis());
            }

            attachment.setId(AttachmentBase.NEW);

            ATTACHMENT_BASE.startTransaction();
            long ts;
            try {
                ts =
                    ATTACHMENT_BASE.attachToObject(
                        attachment,
                        is,
                        session,
                        session.getContext(),
                        user,
                        userConfiguration);
                ATTACHMENT_BASE.commit();
            } catch (OXException x) {
                ATTACHMENT_BASE.rollback();
                throw x;
            } finally {
                ATTACHMENT_BASE.finish();
            }

            return new AJAXRequestResult(Integer.valueOf(attachment.getId()), new Date(ts), "int");
        } catch (JSONException e) {
            throw AjaxExceptionCodes.JSON_ERROR.create(e, e.getMessage());
        } catch (RuntimeException e) {
            throw AjaxExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }
    
    /**
     * Returns the JSON object payload from the request data
     * 
     * @param requestData The request data
     * @return The JSON object payload or <code>null</code> if no payload is present
     * @throws OXException if a required parameter is missing from the payload
     */
    private JSONObject getJSONObject(AJAXRequestData requestData) throws OXException {
        final JSONObject object = (JSONObject) requestData.getData();
        if (object == null) {
            return null;
        }
        for (final AttachmentField required : Attachment.REQUIRED) {
            if (!object.has(required.getName())) {
                throw AjaxExceptionCodes.MISSING_PARAMETER.create(required.getName());
            }
        }
        if (!object.has(DATASOURCE)) {
            throw AjaxExceptionCodes.MISSING_PARAMETER.create(DATASOURCE);
        }
        return object;
    }

    /**
     * Parses and returns the attachment information from the specified request data
     * 
     * @param requestData the request data
     * @return The attachment information
     * @throws JSONException if a JSON error is occurred
     * @throws OXException If an error is occurred
     */
    private AttachmentInfo parseAttachments(AJAXRequestData requestData) throws JSONException, OXException {
        final UploadEvent upload = requestData.getUploadEvent();
        final List<AttachmentMetadata> attachments = new ArrayList<AttachmentMetadata>();
        final List<UploadFile> uploadFiles = new ArrayList<UploadFile>();

        long sum = 0;
        final JSONObject json = new JSONObject();
        final List<UploadFile> l = upload.getUploadFiles();
        final int size = l.size();
        final Iterator<UploadFile> iter = l.iterator();
        for (int a = 0; a < size; a++) {
            final UploadFile uploadFile = iter.next();
            final String fileField = uploadFile.getFieldName();
            final int index = Integer.parseInt(fileField.substring(5));
            final String obj = upload.getFormField("json_" + index);
            if (obj == null || obj.length() == 0) {
                continue;
            }
            json.reset();
            json.parseJSONString(obj);
            for (final AttachmentField required : REQUIRED) {
                if (!json.has(required.getName())) {
                    throw AjaxExceptionCodes.MISSING_PARAMETER.create(required.getName());
                }
            }

            final AttachmentMetadata attachment = parseAttachmentMetadata(json);
            assureSize(index, attachments, uploadFiles);

            attachments.set(index, attachment);
            uploadFiles.set(index, uploadFile);
            sum += uploadFile.getSize();

            AttachmentUtility.checkSize(sum, requestData);
        }
        return new AttachmentInfo(attachments, uploadFiles);
    }

    private AJAXRequestResult attach(final List<AttachmentMetadata> attachments, final List<UploadFile> uploadFiles, final ServerSession session, final Context ctx, final User user, final UserConfiguration userConfig) throws OXException {
        initAttachments(attachments, uploadFiles);
        boolean rollback = false;
        try {
            ATTACHMENT_BASE.startTransaction();
            rollback = true;

            final Iterator<UploadFile> ufIter = uploadFiles.iterator();
            final JSONArray arr = new JSONArray();
            long timestamp = 0;
            final UUID batchId = UUID.randomUUID();
            
            Iterator<AttachmentMetadata> iterator = attachments.iterator();
            while (iterator.hasNext()) {
                AttachmentMetadata attachment = iterator.next();
                final UploadFile uploadFile = ufIter.next();

                attachment.setId(AttachmentBase.NEW);

                AttachmentBatch batch = new AttachmentBatch(batchId, !iterator.hasNext());
                attachment.setAttachmentBatch(batch);

                BufferedInputStream data = new BufferedInputStream(new FileInputStream(uploadFile.getTmpFile()), 65536);
                final long modified = ATTACHMENT_BASE.attachToObject(attachment, data, session, ctx, user, userConfig);
                if (modified > timestamp) {
                    timestamp = modified;
                }
                arr.put(attachment.getId());

            }

            ATTACHMENT_BASE.commit();
            rollback = false;
            return new AJAXRequestResult(arr, new Date(timestamp), "json");
        } catch (IOException e) {
            throw AjaxExceptionCodes.IO_ERROR.create(e, e.getMessage());
        } finally {
            if (rollback) {
                AttachmentUtility.rollback();
            }
            AttachmentUtility.finish();
        }
    }

    private void initAttachments(final List<AttachmentMetadata> attachments, final List<UploadFile> uploads) {
        final List<AttachmentMetadata> attList = new ArrayList<AttachmentMetadata>(attachments);
        // final Iterator<AttachmentMetadata> attIter = new ArrayList<AttachmentMetadata>(attachments).iterator();
        final Iterator<UploadFile> ufIter = new ArrayList<UploadFile>(uploads).iterator();

        int index = 0;
        for (final AttachmentMetadata attachment : attList) {
            // while (attIter.hasNext()) {
            // final AttachmentMetadata attachment = attIter.next();
            if (attachment == null) {
                attachments.remove(index);
                ufIter.next();
                uploads.remove(index);
                continue;
            }
            final UploadFile upload = ufIter.next();
            if (upload == null) {
                attachments.remove(index);
                uploads.remove(index);
                continue;
            }
            if (attachment.getFilename() == null || "".equals(attachment.getFilename())) {
                attachment.setFilename(upload.getPreparedFileName());
            }
            if (attachment.getFilesize() <= 0) {
                attachment.setFilesize(upload.getSize());
            }
            if (attachment.getFileMIMEType() == null || "".equals(attachment.getFileMIMEType())) {
                attachment.setFileMIMEType(upload.getContentType());
            }
            index++;
        }
    }

    private void assureSize(final int index, final List<AttachmentMetadata> attachments, final List<UploadFile> uploadFiles) {
        int enlarge = index - (attachments.size() - 1);
        for (int i = 0; i < enlarge; i++) {
            attachments.add(null);
        }

        enlarge = index - (uploadFiles.size() - 1);
        for (int i = 0; i < enlarge; i++) {
            uploadFiles.add(null);
        }

    }
    
    @OAuthScopeCheck
    public boolean accessAllowed(AJAXRequestData request, ServerSession session, OAuthAccess access) throws OXException, JSONException {
        long maxUploadSize = AttachmentConfig.getMaxUploadSize();
        if (request.hasUploads(-1, maxUploadSize > 0 ? maxUploadSize : -1L)) {
            return OAuthContentTypes.mayWriteViaOAuthRequest(collectContentTypes(parseAttachments(request)), access);
        }

        JSONObject object = (JSONObject) request.getData();
        if (object == null) {
            return false;
        }
        AttachmentMetadata metadata = parseAttachmentMetadata(object);
        return OAuthContentTypes.mayWriteViaOAuthRequest(getContentType(metadata.getModuleId()), access);
    }
    
    /**
     * Collects all folder content types from the request
     * 
     * @return A Set with all content types from the request
     */
    private Set<ContentType> collectContentTypes(AttachmentInfo attachmentInfos) {
        short items = 0;
        Set<ContentType> contentTypes = new HashSet<>(4);
        for (AttachmentMetadata attachment : attachmentInfos.attachments) {
            if (false == contentTypes.add(getContentType(attachment.getModuleId()))) {
                continue;
            }
            items++;
            if (items == ACCEPTED_OAUTH_MODULES.size()) {
                break;
            }
        }
        return contentTypes;
    }

    private static class AttachmentInfo {

        final List<AttachmentMetadata> attachments;
        final List<UploadFile> uploadFiles;

        /**
         * Initialises a new {@link AttachAction.AttachmentInfo}.
         */
        public AttachmentInfo(List<AttachmentMetadata> attachments, List<UploadFile> uploadFiles) {
            super();
            this.attachments = attachments;
            this.uploadFiles = uploadFiles;
        }
    }
}
